// isounit.h: Interactive Screen Object (Iso) Class

#ifndef H_ISOUNIT
#define H_ISOUNIT

#include <stdlib.h>
#include "fsounit.h"
#include "keybrd.h"
                   
// Some event codes (The msmouse.h and keybrd.h files provide others)
const int Close    = 0x011b;  // Same as the ESC key
const int ShutDown = 0x2d00;  // Same as the ALT-X key
const int Cycle    = 0x0f09;  // Same as the Tab key
const int StrMsg   = 0xffff;  // Indicates there is a string message

class Iso;         // Forward declarations
class IsoMgr;

struct MsgPkt {     // Message packet      
  Iso *Focus;       // Holds current focus 
  unsigned Code;    // The event code      
  unsigned RtnCode; // Return event code   
  int Mx, My;       // Mouse coordinates   
  char Str[80];     // The message         
};

extern MsgPkt NullMsg; // Defined in isounit.cpp

class Iso {
public:
  Fso *Panel;        // Can be a Tfso or Gfso pointer 
  char Active;       // True if iso is selected      
  char Visible;      // True if iso is showing       
  char IsClosed;     // True if Iso is closed        
  char TouchFlag;    // True if touching another Iso 
  char ClipToFrame;  // Used for scroll bars         
  Iso *Base;         // The base for this iso        
  Iso *Under, *Over; // Iso stack pointers           
  IsoMgr *SubMgr;    // The child iso stack          
  Iso(Fso *P);
  virtual ~Iso(void);
  virtual Rso *ClippingRect(void);
  virtual void SetLocn(int Xl, int Yl, CoordType Ctype); 
  virtual void SetSize(int W, int H); 
  virtual void Move(int X, int Y); 
  virtual void DeltaMove(int Dx, int Dy); 
  virtual void MoveLoop(MsgPkt &M); 
  virtual void Stretch(int W, int H); 
  virtual void DeltaStretch(int Dw, int Dh); 
  virtual void StretchLoop(MsgPkt &M); 
  virtual int Obscured(void);
  virtual void Draw(void) { ; }
  virtual void DrawPanel(void); 
  virtual void Redraw(void) { DrawPanel(); }
  virtual void Open(Iso *B, int X, int Y);
  virtual void Reopen(int X, int Y); 
  virtual void Swap(void); 
  virtual void Hide(void); 
  virtual void Show(void); 
  virtual void SetVisibleFlag(int F);
  virtual void Select(void); 
  virtual void Remove(void); 
  virtual void Prompt(void); 
  virtual void UnPrompt(void); 
  virtual void Enter(MsgPkt &M); 
  virtual void Leave(MsgPkt &M); 
  virtual void SwitchFocus(MsgPkt &M); 
  virtual void Activate(MsgPkt &) { ; } 
  virtual void OnMouseEnter(MsgPkt &M); 
  virtual void OnMouseLeave(MsgPkt &M); 
  virtual void OnMouseWithin(MsgPkt &) { ; }
  virtual void OnClose(MsgPkt &M); 
  virtual void OnMouseUp(MsgPkt &M) { Activate(M); }
  virtual void OnMouseDown(MsgPkt &M); 
  virtual void BorderHandler(MsgPkt &M); 
  virtual void OnMouseStillDown(MsgPkt &) { ; }
  virtual void OnKeyStroke(MsgPkt &M); 
  virtual void OnShiftArrow(MsgPkt &M); 
  virtual void Dispatch(MsgPkt &M);
};

class IsoMgr {
public:
  Iso *Top, *Bottom; // Top and bottom of iso stack   
  Iso *Base;         // Who this stack belongs to     
  Iso *Hot;          // Who the mouse is currently on 
  Iso *Marker;       // A marker used when cycling    
  IsoMgr(Iso *B);
  virtual ~IsoMgr(void); 
  virtual void Push(Iso *Ip); 
  virtual void MoveToFront(Iso *Me, int Keep); 
  virtual void ResetTouchFlags(Iso *Me);
  virtual void SetTouchFlags(Iso *Me);
  virtual void OnIso(int Mx, int My, Iso **I);
  virtual Iso *CycleForw(Iso *Curr);
  virtual void ProcessCycle(MsgPkt &M);
  virtual void EventLoop(MsgPkt &M);
  virtual void EventStep(MsgPkt &M);
};

#endif
