#include "rsounit.h"

Rect::Rect(int X, int Y, int W, int H)
// Initializes the dimensions of an Rso
{
  Xul = X;  Yul = Y;  // Upper left-hand corner
  Xlr = X + W - 1;    // Lower right-hand corner
  Ylr = Y + H - 1;
  Wd  = W;  Ht  = H;  // Overall size
}

void Rect::SetSize(int W, int H)
// Sets the size of a rectangle and adjusts the lower-right 
// coordinates 
{
  Wd  = W;  Ht = H; 
  Xlr = Xul + W - 1;    // Adjust lower-right corner
  Ylr = Yul + H - 1;
}

void Rect::SetLocn(int Xl, int Yl)
// Sets the top-left coordinates and adjust the lower-right 
// coordinates also 
{
  Xul = Xl;  Yul = Yl; 
  Xlr = Xl + Wd - 1;    
  Ylr = Yl + Ht - 1;
}

void Rect::ClipSize(int &W, int &H)
// Checks width (W) and height (H) to ensure that they are not 
// larger than the rectangle's dimensions. If they are, then 
// they are adjusted to fit. Also, the sizes must be greater 
// than zero.
{
  if (W > Wd) W = Wd;
  if (W < 1)  W = 1;  
  if (H > Ht) H = Ht;
  if (H < 1)  H = 1;
}

void Rect::ClipDim(int &X, int &Y, int &W, int &H)
// Checks the incoming dimensions to ensure they are in range 
// of the rectangle. If not, they are adjusted to fit.
{
  int Xl, Yl;
  ClipSize(W, H);              // Make sure wd and ht are in range
  if (X < Xul) X = Xul;        // Make sure upper left-hand
  if (Y < Yul) Y = Yul;        // corner is in range
  Xl = X + W - 1;              // Check right-hand coordinate
  if (Xl > Xlr) X -= Xl - Xlr;
  Yl = Y + H - 1;              // Check lower coordinate
  if (Yl > Ylr) Y -= Yl - Ylr;
}

int Rect::HzClip(int &X, int &Y, int &N)
// Given a starting coordinate (X,Y) (relative to the rectangle),
// and a length N, this function clips N to fit in the rectangle, 
// assuming that the length runs horizontal. If X is negative, it 
// is set to zero and the length adjusted accordingly. Y must be in
// range. The function returns true if a non-null clip is found;
// otherwise, false is returned. 
{  
  int Rval = False;
  if ((Y >=0) && (Y < Ht)) { // Y must be in range
     if (X < 0) {            // Clip to left
        N += X;
        X = 0;
     }
     if ((N > 0) && (X < Wd)) {
        if ((X + N) > Wd) N = Wd - X; // Clip to right
        Rval = True;
     }
  }
  return Rval;
}

int Rect::VtClip(int &X, int &Y, int &N)
// Given a starting coordinate (X,Y) (relative to the rectangle),
// and a length N, this function clips N to fit in the rectangle, 
// assuming that the length runs vertical. If Y is negative, it 
// is set to zero and the length adjusted accordingly. X must be in
// range. This function returns true if a non-null clip is found; 
// otherwise, false is returned.
{
  int Rval = False;
  if ((X >=0) && (X < Wd)) { // X must be in range
     if (Y < 0)  {           // Clip above
        N += Y;
        Y = 0;
     }
     if ((N > 0) && (Y < Ht))  {
        if ((Y + N) > Ht)  N = Ht - Y; // Clip below
        Rval = True;
     }
  }
  return Rval;
}

int Rect::Contains(int X, int Y)
// Returns true if (X,Y) are contained in the rectangle
{
  return (X >= Xul) && (X <= Xlr) &&
         (Y >= Yul) && (Y <= Ylr);
}

int Rect::Touches(Rect *R)
// Returns true if R overlaps Self. They overlap if their 
// horizontal && vertical extents both cross. Note that as 
// defined, a rectangle can't overlap itself. 
{
  if (R == this) return False; 
  else return 
      (Xul <= R->Xlr) && (Xlr >= R->Xul) && // Hz crossing
      (Yul <= R->Ylr) && (Ylr >= R->Yul);   // Vt crossing
}


