// tfsounit.cpp: Tfso Class Implementation

#include <dos.h>
#include "tfsounit.h"

// Set up a default TxBuff for Tfso and Tskel objects.
// We use the current screen mode, and assume an 80x25 screen.

TxBuff ScrnBuff(80, 25, VideoPtr());

Tfso::Tfso(int Ba, int Fa, ColorPak &Cp)
// Makes a text-based Fso. (We default the Trso's so that they 
// use the full screen texel buffer). We set up the shadows and 
// null out the swap buffer.
: Fso(Ba, Fa, Cp)
{
  Overall  = new Trso(&ScrnBuff);  // Type as Trso's
  Frame    = new Trso(&ScrnBuff); 
  Interior = new Trso(&ScrnBuff); 
  SaveBuff = NULL;   // To be allocated by SetSize
  if (HasShadow()) { // Shadows are Tfso's themselves
     HzShadow = new Tfso(0x00, Swappable, Cp);
     VtShadow = new Tfso(0x00, Swappable, Cp);
  }
  else { // We don't have any shadows
    HzShadow = NULL;  VtShadow = NULL;
  }
}

Tfso::~Tfso(void)
// Destroy the save buffer and the shadow objects. 
// Then destroy the frame rectangles.
{
  if (IsSwappable()) delete SaveBuff;
  if (HasShadow()) { delete HzShadow; delete VtShadow; }
  delete Frame; 
  delete Interior;
  delete Overall;
}

void Tfso::SetSize(int W, int H)
// The size (W,H) is interpreted to be the interior size. 
// The other sizes are based off of it. Allocates space
// for the save buffer if Swappable.
{
  Interior->SetSize(W, H);
  Frame->SetSize(W+Bwd*2, H+Bwd*2);
  if (HasShadow()) {
     Overall->SetSize(Frame->Wd+2, Frame->Ht+1);
     HzShadow->SetSize(Frame->Wd+1, 1);
     VtShadow->SetSize(2, Frame->Ht-1);
  }
  else {
     Overall->SetSize(Frame->Wd, Frame->Ht);
  }
  if (IsSwappable()) {
     if (SaveBuff == NULL) {
        SaveBuff = new Trso(NULL);
     }
     SaveBuff->SetSize(Frame->Wd, Frame->Ht);
  }
}

void Tfso::SetLocn(int Xl, int Yl)
// Set the upper left hand corner location of the frames.
// The location is interpreted to be that of the frame
// rectangle. Also, set the shadow locations depending on
// the type of shadow. 
{
  Frame->SetLocn(Xl, Yl);
  Interior->SetLocn(Xl+Bwd, Yl+Bwd);
  switch (Fattr & AnyShadow) {
    case SEShadow: 
      Overall->SetLocn(Frame->Xul, Frame->Yul);
      HzShadow->SetLocn(Frame->Xul+1, Frame->Ylr+1);
      VtShadow->SetLocn(Frame->Xlr+1, Frame->Yul+1);
    break;
    case NEShadow: 
      Overall->SetLocn(Frame->Xul, Frame->Yul-1);
      HzShadow->SetLocn(Frame->Xul+1, Frame->Yul-1);
      VtShadow->SetLocn(Frame->Xlr+1, Frame->Yul);
    break;
    case NWShadow: 
      Overall->SetLocn(Frame->Xul-2, Frame->Yul-1);
      HzShadow->SetLocn(Frame->Xul-2, Frame->Yul-1);
      VtShadow->SetLocn(Frame->Xul-2, Frame->Yul);
    break;
    case SWShadow: 
      Overall->SetLocn(Frame->Xul-2, Frame->Yul);
      HzShadow->SetLocn(Frame->Xul-2, Frame->Ylr+1);
      VtShadow->SetLocn(Frame->Xul-2, Frame->Yul+1);
    break;
    default: // No shadow 
      Overall->SetLocn(Frame->Xul, Frame->Yul);
  }
}

void Tfso::DrawFrame(char Ba, char Attr)
// Draws the frame by simply drawing a box. A close button
// is added if needed. 
{
  if (Bwd > 0) {
     if (Ba == 0) Ba = (Bstyle << 4) + Bwd;
     if (Attr == 0) Attr = Colors.Bc;
     Frame->Box(0, 0, Frame->Wd, Frame->Ht, Ba, Attr);
     if (IsCloseable())
        Frame->HzWrt(1, 0, "\xb4\x08\xc3", Colors.Bc);
  }
}

int Tfso::OnCloseButton(int X, int Y)
// Returns true if location (X,Y) is on the close button.
// The close button is near the top left hand corner.
{
  return 
    IsCloseable() && (Bwd > 0) && 
    (Y == Frame->Yul) && (X == Frame->Xul + 2);
}

void Tfso::Clear(char Ch, char Attr)
// Clears the interior rectangle using the specified character
// and attribute. The attribute can be defaulted to use the
// value stored with the object.
{
  if (Attr > 0) Colors.Wc = Attr;
  Interior->Fill(0, 0, Interior->Wd, Interior->Ht, Ch, Colors.Wc);
}

void Tfso::GetImage(Rect *C)
// Get the screen image region bounded by the frame rectangle
// and store it in the save buffer (if there is one).
// Get the shadow image too. C is the clipping rectangle
// for the shadow.
{
  if (IsSwappable()) {
     SaveBuff->
       Xfr(0, 0, SaveBuff->Wd, SaveBuff->Ht, (Trso *)Frame, 0, 0);
     DrawShadows(C, GetIm, False);
  }
}

void Tfso::PutImage(Rect *C)
// Put the image stored in the save buffer (if there is
// one) onto the screen at the region bounded by the frame 
// rectangle. Put back the saved shadow buffer too. C is 
// the clipping rectangle for the shadow.
{
  if (IsSwappable()) {
     Trso *Tf = (Trso *)Frame;
     Tf->Xfr(0, 0, Tf->Wd, Tf->Ht, SaveBuff, 0, 0);
     DrawShadows(C, PutIm, False);
  }
}

void Tfso::Swap(Rect *C, XfrDirn Xd)
// Swap the screen image bounded by the frame rectangle and the
// save buffer. The Xfr direction Xd and clipping rectangle C
// are used only for the shadows, which are drawn/erased accordingly.
{
  if (IsSwappable()) {
     Trso *Tf = (Trso *)Frame;
     Tf->Swap(0, 0, Tf->Wd, Tf->Ht, SaveBuff, 0, 0);
     DrawShadows(C, Xd, True);
  }
}

void Tfso::DrawShadows(Rect *C, XfrDirn Xd, int DrawIt)
{
  if (HasShadow()) {
     HzShadow->ShadowXfr(C, Xd, DrawIt);
     VtShadow->ShadowXfr(C, Xd, DrawIt);
  }
}

void Tfso::ShadowXfr(Rect *C, XfrDirn Xd, int DrawIt)
// Depending on Xd, either put the image in the shadow buffer
// onto the screen, or get the image from the screen into
// the buffer. If getting the image, and if DrawIt is True, 
// after saving the screen image, draw the shadow 
// by using a grey fill color. It's assumed this is a  
// shadow, and it's assumed the buffer is not NULL. C is 
// the clipping rectangle for the shadow. 
{
  int X, Y, W, H;
  Trso *Tp = (Trso *)Overall;
  X = Tp->Xul - C->Xul;  // Must compute coordinates relative
  Y = Tp->Yul - C->Yul;  // to the clipping rectangle        
  W = Tp->Wd; H = Tp->Ht;
  if (C->HzClip(X, Y, W) && C->VtClip(X, Y, H)) {
     X -= Tp->Xul - C->Xul; // Compute coordinates relative
     Y -= Tp->Yul - C->Yul; // to the shadow
     if (Xd == PutIm) {  // From buffer to screen
        Tp->Xfr(X, Y, W, H, SaveBuff, 0, 0); 
     }
     else { // From screen to buffer, then possibly draw
       SaveBuff->Xfr(0, 0, W, H, Tp, X, Y);
       if (DrawIt) Tp->FillB(X, Y, W, H, 8, 1); // Grey fill
     }
  }
}

// ----------- Now for the Text Skeleton Methods -----------

Tskel::Tskel(ColorPak &Cp)
// Much like the Tfso class, except we have swap buffers for
// the sides, and none for the interior. Sets the size to
// 80x25. It's assumed the size will be changed during
// moving/stretching. Setting it to 80x25 here will help
// alleviate heap fragmentation. Border style has dashed lines, 
// and frame style is swappable and stretchable.
: Fso(0x31, Swappable+Stretchable, Cp) 
{
  int I;

  Overall  = new Trso(&ScrnBuff); // Type as Trso's
  Frame    = new Trso(&ScrnBuff);
  Interior = new Trso(&ScrnBuff);
  for (I=0; I<4; I++) Sides[I] = NULL;
  SetSize(80, 25); // This allocates the sides
}


Tskel::~Tskel(void)
// Does away with the side swap buffers
{
  int I;
  for (I=0; I<4; I++) delete Sides[I];
}

void Tskel::SetSize(int W, int H)
// Sets the size of the rectangles and the side swap buffers
{
  int I;

  Interior->SetSize(W, H);
  Frame->SetSize(Interior->Wd+2, Interior->Ht+2);
  Overall->SetSize(Interior->Wd+2, Interior->Ht+2);

  for (I=0; I<4; I++) {
      if (Sides[I] == NULL) Sides[I] = new Trso(NULL);
      if (I < 2) { // The top and bottom
         Sides[I]->SetSize(Frame->Wd, 1);
      }
      else { // the left and right sides
         Sides[I]->SetSize(1, Frame->Ht);
      }
      Sides[I]->Pic->Fill(0, 0,  // Blank out the sides
        Sides[I]->Pic->Wd, Sides[I]->Pic->Ht, ' ', Colors.Bc);
  }
}


void Tskel::DrawFrame(char Ba, char Attr)
// Just like the Tfso version
{
  if (Bwd > 0) {
     if (Ba == 0) Ba = (Bstyle << 4) + Bwd;
     if (Attr == 0) Attr = Colors.Bc;
     Frame->Box(0, 0, Frame->Wd, Frame->Ht, Ba, Attr);
  }
}

void Tskel::GetImage(Rect *)
// Get the sides from screen to buffer.
// Note: we don't use the rectangle parameter.
{
  Trso *Tf = (Trso *)Frame;

  if ((Tf->Ht < 1) || (Tf->Wd < 1)) return;
  Sides[0]->Xfr(0, 0, Tf->Wd, 1, Tf, 0, 0);             // Top row
  Sides[1]->Xfr(0, 0, Tf->Wd, 1, Tf, 0, Tf->Ht-1);      // Bottom row
  if (Tf->Ht > 2) {
     Sides[2]->Xfr(0, 0, 1, Tf->Ht-2, Tf, 0, 1);        // Left side
     Sides[3]->Xfr(0, 0, 1, Tf->Ht-2, Tf, Tf->Wd-1, 1); // Right side
  }
}

void Tskel::PutImage(Rect *)
// Get the sides from buffer to screen.
// Note: we don't use the rectangle parameter
{
  Trso *Tf = (Trso *)Frame;

  if ((Tf->Ht < 1) || (Tf->Wd < 1)) return;
  Tf->Xfr(0, 0, Tf->Wd, 1, Sides[0], 0, 0);             // Top row
  Tf->Xfr(0, Tf->Ht-1, Tf->Wd, 1, Sides[1], 0, 0);      // Bottom row
  if (Tf->Ht > 2) {
     Tf->Xfr(0, 1, 1, Tf->Ht-2, Sides[2], 0, 0);        // Left side
     Tf->Xfr(Tf->Wd-1, 1, 1, Tf->Ht-2, Sides[3], 0, 0); // Right side
  }
}

void Tskel::Swap(Rect *, XfrDirn)
// Swap save buffer image and frame buffer image.
// Note: we don't use the parameters. 
{
  Trso *Tf = (Trso *)Frame;

  if ((Tf->Ht < 1) || (Tf->Wd < 1)) return;
  Tf->Swap(0, 0, Tf->Wd, 1, Sides[0], 0, 0);             // Top row
  Tf->Swap(0, Tf->Ht-1, Tf->Wd, 1, Sides[1], 0, 0);      // Bottom row
  if (Tf->Ht > 2) {
     Tf->Swap(0, 1, 1, Tf->Ht-2, Sides[2], 0, 0);        // Left side
     Tf->Swap(Tf->Wd-1, 1, 1, Tf->Ht-2, Sides[3], 0, 0); // Right side
  }
}

