// trsounit.cpp: Trso class implementation
// These routines use the built-in rectangle to clip any screen
// transfers. They also take care of the mouse cursor as well.

#include <stdlib.h>
#include <string.h>
#include "trsounit.h"

Trso::Trso(TxBuff *T) : Rso(0, 0, 0, 0)
// Initializes a rectangle and sets up the texel buffer
{
  if (T == 0) {
     Pic = new TxBuff(0);
  }
  else {
    Pic = new TxBuff(T->Txram);
    Rso::SetSize(T->Wd, T->Ht);
    Pic->SetSize(T->Wd, T->Ht);
  }
}

Trso::~Trso(void)
// De-allocates the texel buffer 
{
  delete Pic;
}

void Trso::SetSize(int W, int H)
// Sets the size of a rectangle. If the rectangle's buffer
// isn't aliased (which means it was allocated on the heap),
// the buffer is re-sized as well. 
{
  Rso::SetSize(W, H);
  if (!Pic->Aliased) Pic->SetSize(W, H);
}

void Trso::SetLocn(int Xl, int Yl)
// Sets the rectangle's location && the origin reference
// (OrgPtr) in the texel buffer 
{
  Rso::SetLocn(Xl, Yl);
  Pic->SetLocn(Xl, Yl);
}

void Trso::HzWrt(int X, int Y, char *Str, char Attr)
// Writes a string horizontally in a specified attribute. 
// Supports both left- and right-hand clipping. 
{
  int Cnt, Offset;
  Cnt = strlen(Str);
  if (X < 0) Offset = -X; else Offset = 0; 
  if (HzClip(X,Y,Cnt)) {
     Mouse.Hide();
     Pic->HzWrt(X,Y,Str+Offset,Attr,Cnt);
     Mouse.Show();
  }
}

void Trso::HzWrtB(int X, int Y, char *Str)
// Writes a string horizontally without changing the screen 
// attribute. Supports both left- and right-hand clipping. 
{
  int Cnt, Offset;
  Cnt = strlen(Str);  
  if (X < 0)  Offset = -X; else Offset = 0; 
  if (HzClip(X,Y,Cnt))  {
     Mouse.Hide();
     Pic->HzWrtB(X,Y,Str+Offset,Cnt);
     Mouse.Show();
  }
}

void Trso::Fill(int X, int Y, int W, int H, char Ch, char Attr)
// Fills a rectangular region with both a character && an
// attribute. Supports clipping on all sides. Note the checks 
// for a horizontal (single row) and vertical (single column) 
// fills so that the clipping works correctly. 
{
  if (((H == 1) && HzClip(X, Y, W)) ||
      ((W == 1) && VtClip(X, Y, H)) ||
      (VtClip(X,Y,H) && HzClip(X,Y,W)))  {
     Mouse.Hide();
     Pic->Fill(X, Y, W, H, Ch, Attr);
     Mouse.Show();
  }
}

void Trso::FillB(int X, int Y, int W, int H, char Ch, char Opt)
// Fills a rectangular region with either a character or an
// attribute, but not both. Here's how the control parameter
// works:
//
//  Opt == 0 -- character fill 
//  Opt == 1 -- attribute fill
//
// Clipping is performed on all four sides. !e the checks for a
// horizontal (single row) && vertical (single column) fills so that 
// the clipping is performed properly. 
{
  if (((H == 1) && HzClip(X, Y, W)) || // Single row 
      ((W == 1) && VtClip(X, Y, H)) || // Single column
      (VtClip(X,Y,H) && HzClip(X,Y,W))) {
     Mouse.Hide();
     Pic->FillB(X, Y, W, H, Ch, Opt);
     Mouse.Show();
  }
}

void Trso::Box(int X, int Y, int W, int H, char Ba, char Attr)
// Draws a text-based box of a specified style and width 
{
  unsigned char Bw, Bs, I;
  Bw = (unsigned char)Ba & 0x000f;         // Unpack the border width
  Bs = ((unsigned char)Ba >> 4) & 0x000f;  // Unpack the border style
  Mouse.Hide();
  if ((Bs < 4))  {
     for (I = 0; I<Bw; I++, X++, Y++, W-=2, H-=2) {
       Fill(X, Y, W, 1, LineChars[Bs][HzBar], Attr);
       Fill(X, Y, 1, H, LineChars[Bs][VtBar], Attr);
       Fill(X, Y, 1, 1, LineChars[Bs][Ulc], Attr);
       Fill(X+W-1, Y, 1, H, LineChars[Bs][VtBar], Attr);
       Fill(X+W-1, Y, 1, 1, LineChars[Bs][Urc], Attr);
       Fill(X, Y+H-1, W, 1, LineChars[Bs][HzBar], Attr);
       Fill(X, Y+H-1, 1, 1, LineChars[Bs][Llc], Attr);
       Fill(X+W-1, Y+H-1, 1, 1, LineChars[Bs][Lrc], Attr);
     }
  }
  Mouse.Show();
}

void Trso::Scroll(ScrollDir Sd, int Amt)
// Scrolls the region defined by the rectangle. The region
// can be scrolled up or down a specified amount. 
{
  Mouse.Hide();
  Pic->Scroll(0, 0, Wd, Ht, Sd, Amt);
  Mouse.Show();
}

void Trso::Swap(int X, int Y, int W, int H, Trso *Other, int Xs, int Ys)
// Swaps data between this Self's Pic buffer and another Trso's Pic. The 
// Other Pic region starts at (Xs, Ys) and Self's Pic region starts 
// at X,Y. The swapping region is W by H in size. The coordinates 
// are checked to ensure that they are in range and the width && 
// height are clipped if necessary. 
{
  if (HzClip(X,Y,W) && Other->HzClip(Xs,Ys,W))  {
     if (VtClip(X,Y,H) && Other->VtClip(Xs,Ys,H))  {
        Mouse.Hide();
        Pic->Swap(X, Y, W, H, Other->Pic, Xs, Ys);
        Mouse.Show();
     }
  }
}

void Trso::Xfr(int X, int Y, int W, int H, Trso *Src, int Xs, int Ys)
// Transfers data from the Src Trso Pic region, starting at (Xs, Ys)
// to the Self's Pic region, starting at X,Y. The region transferred 
// has size W by H. The coordinates are checked to ensure they are in 
// range && the width && height are clipped if necessary. 
{
  if (HzClip(X,Y,W) && Src->HzClip(Xs,Ys,W)) {
     if (VtClip(X,Y,H) && Src->VtClip(Xs,Ys,H)) {
        Mouse.Hide();
        Pic->Xfr(X, Y, W, H, Src->Pic, Xs, Ys);
        Mouse.Show();
     }
  }
}



