/* REDTOGL.C: Toggle redirection

   A redirection example that toggles redirection between standard
   out (STDOUT) and another output channel (NUL, CON, PRN, AUX or
   a file). */

#include <process.h>        // for system()
#include <bios.h>           // for biosprint()
#include <errno.h>          // for errno
#include <stdio.h>          // for printf()
#include <fcntl.h>          // for O_RDWR, O_TRUNC and O_CREAT
#include <string.h>         // for strlen()
#include <sys\stat.h>       // for S_IWRITE
#include <io.h>             // for open(), close(), dup() and dup2()

#define PSTATUS 2           // biosprint(): return printer status cmd
#define PNUM    0           // biosprint(): 0 = LPT1:, 1 = LPT2:, etc.

int redtogl(char *redstr);  // Toggle redirect rtn prototype

//*******************************************************************
void main(void) {
  int status, abyte = 0;

  status = system("DIR *.C");

  printf("Output going to twilight zone (i.e., 'NUL')...\n");
  if (redtogl(NULL) == -1)
    printf("REDIRECT ERROR: %s\n", strerror(errno));
  status = system("DIR *.C");
  if (redtogl(NULL) == -1)
    printf("REDIRECT ERROR: %s\n", strerror(errno));

  printf("Output going to file 'JUNK'....\n");
  if (redtogl("JUNK") == -1) // Redirect output to file called "JUNK"
    printf("REDIRECT ERROR: %s\n", strerror(errno));
  status = system("DIR *.C");
  if (redtogl(NULL) == -1)   // Parameter is ignored; restore STDOUT
    printf("REDIRECT ERROR: %s\n", strerror(errno));

  /* 'PRN' (printer) and 'CON' (console) are standard DOS devices.
     Since output to 'CON' has same effect as output to 'STDOUT'
     we won't bother demonstrating it.

     Because trying to output to a non-existent or non-selectable
     printer would cause DOS to display the infamous "Abort, Retry,
     Fail" error message, we'll use the 'biosprint()' function to
     skip redirected output if the printer can't be used (not
     selected, out of paper, etc.) */

  printf("Will attempt to send output to PRN.\n");
  status = biosprint(PSTATUS, abyte, PNUM);
  if ((status & 0xBF) == 0x90) {
    if (redtogl("PRN") == -1) // Try redirecting to DOS List device
      printf("REDIRECT ERROR: %s\n", strerror(errno));
        status = system("DIR *.C");
      if (redtogl("PRN") == -1) // ignore parameter & restore STDOUT
        printf("REDIRECT ERROR: %s\n", strerror(errno));
  }
  else
    printf("PRN not ready. Skipping redirected output to PRN.\n");

  printf("End of program.\n"); }  // end of main()

/*-------------------------------------------------------------------
  redtogl - Alternates (toggles) standard output between the 'NUL'
            or user specified output device and the currently set
            standard output device (in handle 1). If a null
            (zero-length) string is passed to this routine on even
            numbered calls, output is sent to 'NUL' which effectively
            suppresses output.  If a valid DOS device name (CON,
            PRN, AUX, NUL) or a file name is passed to the routine
            output is redirected to the named device.

            Odd numbered calls restores (toggles) output back to
            the device that was set on the last call to the routine
            (probably the DOS console device, i.e., 'CON').

  SYNOPSIS: redtogl(tostring)
    char *tostring - destination of redirected output or NULL

  RETURNS: Current standard output handle if successful or -1
           on failure and one of the following in global 'errno'
           variable:

             ENOENT  - no such file or directory
             EMFILE  - too many open files
             EACCES  - permission denied
             EINVACC - invalid access code
             EBADF   - bad file number */

int redtogl(char *redstr) {
  /* NOTE: 'oldstdout' is static because we have to retain it
           until the next call to 'redtogl' when it is used to
           restore the original output device, that is, so the
           routine can toggle redirection. */

  static int oldstdout;  // save handle of std output device here
  int nul;               // put handle for redirect device here

#define STDOUT  1

/* 'oldstdout' is static so we can depend on the compiler to
   initialize it to 0. Subsequent calls to redtogl() assure it
   is maintained so we can safely disable the 'Possible use of
   oldstdout before definition' warning. Note, however, that you
   should always clearly understand why a warning is being
   generated before disabling it! */
#pragma warn -def

  if (!oldstdout) {
    if (!strlen(redstr))
      nul = open("NUL", O_RDWR);
    else
      nul = open(redstr, O_CREAT|O_TRUNC|O_RDWR, S_IWRITE);

    if (nul < 0) {
      close(nul);
      return(-1);
    }
    else {
      oldstdout = dup(STDOUT);
      dup2(nul, STDOUT);
      close(nul);
    }
  }
  else {
    dup2(oldstdout, STDOUT);
    oldstdout = close(oldstdout);
  }
  return(nul);

/* This restores the 'possible use of xxxx before definition'
   warning to its state before it was disabled it; i.e., if it
   was on it is enabled else it is left disabled. This assures
   that later occurances of such warnings aren't missed. */
#pragma warn .def
}  // end of redtogl()
