/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: array.hpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for the set of Array clases, implemented using
*				templates. Defines the following classes:
*
*					Array			- Array of T's
*					SArray			- Sortable array of T's
*					IArray			- Indirect array (array of pointers to T)
*					ISArray			- Indirect sortable array
*
*					ArrayIterator	- Iterator for array's
*					IArrayIterator	- Iterator for indirect array's
*
*				Sortable array's are _not_ maintained in explicit sorted
*				order unless the addSorted() member function is used,
*				or the sort() member function is called. This allows
*				the arrays to be used to get data into sorted order
*				efficiently by filling the array normally and then calling
*				sort().
*
* $Id: array.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__ARRAY_HPP
#define	__ARRAY_HPP

#ifndef	__TECHLIB_HPP
#include "techlib.hpp"
#endif

#ifndef	__STRING_H
#include <string.h>
#endif

#ifndef	__STDLIB_H
#include <stdlib.h>
#endif

#ifndef	__LIMITS_H
#include <limits.h>
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The following class maintains an array of T's in contiguous memory.
// The array will dynamically adjust it's size depending on how many items
// are stored in it. The class T must have a valid default constructor and
// copy semantics, as well as valid operator == semantics. For efficiency
// reasons, when direct array's are copied, we copy the data with memcpy(),
// rather than invoking the copy constructor for each element. This
// essentially always performs a shallow copy on each of the elements in
// the array, which would have to be manually deepened if a deep copy is
// required.
//
// The resizeable array uses a simple delta technique to expand the size
// of the array to a new possible size. A better technique that guarantees
// amortised O(1) access to array elements no matter how large the array
// needs to be is to use expand the size of the array by a ratio, rather
// than a set delta value.
//---------------------------------------------------------------------------

template <class T> class Array {
protected:
	T		*data;				// Pointer to the array of data
	uint	_size,_delta;		// Size and increment values
	uint	count;				// Number of items in array

			// Resize the array to a new size
			void expand(uint newSize);
			void shrink();

			// Member to compute the actual size
			uint computeSize(uint sz)
			{	return (_delta != 0) && (sz % _delta) ?
					(((sz + _delta) / _delta) * _delta) : sz;
			};

public:
			// Default constructor
			Array();

			// Constructor
			Array(uint size,uint delta = 0);

			// Destructor
			~Array()	{ delete [] data; };

			// Copy constructors
			Array(const Array<T>& a,uint delta = UINT_MAX);
			const Array<T>& operator = (const Array<T>& a);

			// Comparision methods for arrays
			bool operator == (const Array<T>& a);
			bool operator != (const Array<T>& a)
				{ return !(*this == a); };

			// Indexing operators
			T& operator [] (uint index)
			{ 	PRECONDITION(data != NULL && index < count);
				return data[index];
			};

			const T& operator [] (uint index) const
			{ 	PRECONDITION(data != NULL && index < count);
				return data[index];
			};

			// Methods to add elements to the array
			void add(T item);
			void insert(T item,uint index,uint count = 1);

			// Method to replace an element in the array
			void replace(T item,uint index);

			// Method to remove an element from the array
			void remove(uint index);

			// Method to search for an element in the array
			uint search(T item,uint first,uint last,int direction = +1) const;
			uint search(T item,int direction = +1) const
				{ return search(item,0,count-1,direction); };

			// Overloaded cast to a const T*
			operator const T*() const	{ return data; };

			// Method to set the number of elements in the array (empties it)
			void setCount(uint newCount);

			// Return the number of items in the array
			uint numberOfItems() const	{ return count; };

			// Returns true if the array is empty
			bool isEmpty() const		{ return count == 0; };

			// Returns true if the array is full
			bool isFull() const
				{ return (_delta == 0) && (count == _size); };

			// Return the size/delta for the array
			uint size() const			{ return _size; };
			uint delta() const			{ return _delta; };

			// Modify the delta value for the array
			void setDelta(uint delta);

			// Resize the array to a new size (array is emptied of data)
			void resize(uint newSize);

			// Empties the array of all data
			void empty();

			// Returns true if the array is valid
			bool valid() const			{ return data != NULL; };

private:
	friend	class ArrayIterator<T>;
	};

//---------------------------------------------------------------------------
// The following class maintains an array of T's in contiguous memory, and
// is derived from the Array class. The items in the array are assumed to
// be sortable, and have valid <, == and > operators. The items are _not_
// maintained in explicit sorted order unless the addSorted() member
// function is used to add items, or the sort() member function is called.
//---------------------------------------------------------------------------

template <class T> class SArray : public Array<T> {
protected:
			// Comparison function for calling qsort().
	static	int cmp(const void *t1,const void *t2);

public:
			// Default constructor
			SArray() : Array<T>() {};

			// Constructor
			SArray(uint size,uint delta = 0)
				: Array<T>(size,delta) {};

			// Method to add elements to the array in sorted order
			void addSorted(T item);

			// Method to do a binary search for an item
			uint binarySearch(T item,uint first,uint last) const;
			uint binarySearch(T item) const
				{ return binarySearch(item,0,count-1); };

			// Method to sort the elements in the array
			void sort()
				{ qsort(data,count,sizeof(T),SArray<T>::cmp); };
	};

//---------------------------------------------------------------------------
// The following class maintains an array of pointers to T in contiguous
// memory. The array will dynamically adjust it's size depending on how
// many items are stored in it.
//
// If a copy is made of an array, shouldDelete is set to false so that
// we do not attempt to delete the data twice. You should still however be
// careful about copying indirect array's and freeing the memory used.
//---------------------------------------------------------------------------

template <class T> class IArray : public Array<T*> {
protected:
	bool	shouldDelete;			// Flags if elements should be deleted

public:
			// Default constructor
			IArray() : Array<T*>()
				{ shouldDelete = true; };

			// Constructor
			IArray(uint size,uint delta = 0) : Array<T*>(size,delta)
				{ shouldDelete = true; };

			// Destructor
			~IArray()	{ empty(); };

			// Copy constructors
			IArray(const IArray<T>& a,uint delta = UINT_MAX)
				: Array<T*>(a,delta) { shouldDelete = false; };
			const IArray<T>& operator = (const IArray<T>& a);

			// Comparision methods for indirect arrays
			bool operator == (const IArray<T>& a);
			bool operator != (const IArray<T>& a)
				{ return !(*this == a); };

			// Method to replace an element in the array
			void replace(T* item,uint index);

			// Method to destroy an element in the array
			void destroy(uint index);

			// Methods to search for an element in the array
			uint search(const T* item,uint first,uint last,int direction = +1) const;
			uint search(const T* item,int direction = +1) const
				{ return search(item,0,count-1,direction); };

			// Method to set the number of elements in the array (empties it)
			void setCount(uint newCount);

			// Resize the array to a new size (array is emptied of data)
			void resize(uint newSize);

			// Empties the array of all data
			void empty();

private:
	friend	class IArrayIterator<T>;
	};

//---------------------------------------------------------------------------
// The following class maintains an array of pointers to T in contiguous
// memory, and is derived from the IArray class. The items in the array are
// assumed to be sortable, and have valid <, == and > operators. The items
// are _not_ maintained in explicit sorted order unless the addSorted()
// member function is used to add items, or the sort() member function is
// called.
//---------------------------------------------------------------------------

template <class T> class ISArray : public IArray<T> {
protected:
			// Comparison function for calling qsort().
	static	int cmp(const void *t1,const void *t2);

public:
			// Default constructor
			ISArray() : IArray<T>() {};

			// Constructor
			ISArray(uint size,uint delta = 0)
				: IArray<T>(size,delta) {};

			// Method to add elements to the array in sorted order
			void addSorted(T* item);

			// Method to do a binary search for an item
			uint binarySearch(const T* item,uint first,uint last) const;
			uint binarySearch(const T* item) const
				{ return binarySearch(item,0,count-1); };

			// Method to sort the elements in the array
			void sort()
				{ qsort(data,count,sizeof(T*),ISArray<T>::cmp); };
	};

//---------------------------------------------------------------------------
// The following classes are used to iterate through the elements in the
// array. Generally you can simply do normal array indexing rather than
// using an iterator, but iterators are provided for all the fundamental
// data structures such as linked lists and thus arrays and lists can be
// treated the same through the use of iterators.
//---------------------------------------------------------------------------

template <class T> class ArrayIterator {
protected:
	const Array<T>	*a;
	uint			cursor,lower,upper;

public:
			// Default constructor
			ArrayIterator()		{ a = NULL; cursor = lower = upper = 0; };

			// Constructor given an array reference
			ArrayIterator(const Array<T>& arr)
				{ a = &arr; restart(0,arr.numberOfItems()); };

			// Constructor given an array reference and range
			ArrayIterator(const Array<T>& arr,uint start,uint stop)
				{ a = &arr; restart(start,stop); };

			// Initialise an array iterator from an array
			const ArrayIterator<T>& operator = (const Array<T>& arr)
			{ 	a = &arr; restart(0,arr.numberOfItems());
				return *this;
			};

			// Initialise an array iterator from another iterator
			const ArrayIterator<T>& operator = (const ArrayIterator<T>& i)
			{	a = i.a; restart(i.lower,i.upper);
				return *this;
			};

			// Overloaded cast to an integer
			operator int ()		{ return cursor < upper; };

			// Convert the iterator to the corresponding node
			T node()
			{	PRECONDITION(cursor < upper);
				return (*a)[cursor];
			};

			// Pre-increment operator
			T operator ++ ()
			{	if (++cursor < upper)
					return (*a)[cursor];
				else
					return (*a)[upper-1];
			};

			// Post-increment operator
			T operator ++ (int)
			{	if (cursor < upper)
					return (*a)[cursor++];
				else
					return (*a)[upper-1];
			};

			// Method to restart the array iterator
			void restart()	{ restart(lower,upper); };

			// Method to restart with a new range
			void restart(uint start,uint stop)
				{  cursor = lower = start; upper = stop; };
	};

template <class T> class IArrayIterator : public ArrayIterator<T*> {
public:
			// Default constructor
			IArrayIterator() : ArrayIterator<T*>() {};

			// Constructor given an array reference
			IArrayIterator(const IArray<T>& arr) : ArrayIterator<T*>(arr) {};

			// Constructor given an array reference and range
			IArrayIterator(const IArray<T>& arr,uint start,uint stop)
				: ArrayIterator<T*>(arr,start,stop) {};

			// Initialise an array iterator from an array
			const IArrayIterator<T>& operator = (const IArray<T>& arr)
				{ return (IArrayIterator<T>&)ArrayIterator<T*>::operator=(arr); };

			// Initialise an array iterator from another iterator
			const IArrayIterator<T>& operator = (const IArrayIterator<T>& i)
				{ return (IArrayIterator<T>&)ArrayIterator<T*>::operator=(i); };
	};

/*------------------------ Inline member functions ------------------------*/

template <class T> inline void Array<T>::add(T item)
/****************************************************************************
*
* Function:		Array<T>::add
* Parameters:	item	- Item to add to the array
*
* Description:	Resizes the array if the count exceeds the size of the
*				array, and copies the item into the last position.
*
****************************************************************************/
{
	CHECK(valid());
	if (count == _size)
		expand(count+1);
	CHECK(count < _size);
	if (valid())
		data[count++] = item;
}

template <class T> inline void Array<T>::setCount(uint newCount)
/****************************************************************************
*
* Function:		Array<T>::setCount
* Parameters:	newCount	- New count for the array
*
* Description:	Sets the count value for the array, expanding the size
*				if necessary. The array will be filled with the default
*				value.
*
****************************************************************************/
{
	resize(newCount);
	count = newCount;
}

template <class T> inline void Array<T>::replace(T item,uint index)
/****************************************************************************
*
* Function:		Array<T>::replace
* Parameters:	item	- Item to replace in the array
*				index	- Index of the item in the array to replace
*
* Description:	Replaces the item in the array with the new item. The index
*				MUST fall within the current bounds of the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(index < count);
	data[index] = item;
}

template <class T> inline void IArray<T>::setCount(uint newCount)
/****************************************************************************
*
* Function:		IArray<T>::setCount
* Parameters:	newCount	- New count for the array
*
* Description:	Sets the count value for the array, expanding the size
*				if necessary. The array will be filled with the default
*				value.
*
****************************************************************************/
{
	resize(newCount);
	count = newCount;
}

/*---------------------------- Implementation -----------------------------*/

template <class T> Array<T>::Array()
	: _size(1), _delta(1), count(0)
/****************************************************************************
*
* Function:		Array<T>::Array
*
* Description:	Default constructor for the array template. This creates
*				an empty array of size 1, with an increment of 1.
*
****************************************************************************/
{
	data = new T[_size];
}

template <class T> void Array<T>::Array(uint size,uint delta)
	: _delta(delta), count(0)
/****************************************************************************
*
* Function:		Array<T>::Array
* Parameters:	size	- Initial size of the array
*				delta	- Initial increment value for the array
*
* Description:	Constructor for the array template, given the size of the
*				array to create, and the expansion ratio. If the increment
*				is zero, the array is of a fixed size and will not be
*				expanded.
*
*				The array is initially empty containing no valid data.
*
****************************************************************************/
{
	_size = computeSize(size);
	data = new T[_size];
}

template <class T> Array<T>::Array(const Array<T>& a,uint delta)
	: count(a.count)
/****************************************************************************
*
* Function:		Array<T>::Array
* Parameters:	a	- Array to copy
*
* Description:	Copy constructor for array's. We simply allocate space for
*				the new array, and copy each item individually. We copy
*				the data with a fast memcpy.
*
****************************************************************************/
{
	CHECK(a.valid());
	_delta = (delta == UINT_MAX) ? a._delta : delta;
	_size = computeSize(a._size);
	data = new T[_size];
	if (valid())
		memcpy(data,a.data,count * sizeof(T));
}

template <class T> const Array<T>& Array<T>::operator = (const Array<T>& a)
/****************************************************************************
*
* Function:		Array<T>::operator =
* Parameters:	a	- Array to copy
*
* Description:	Assignment operator for array's. Allocates space for the
*				new array and copies the data with a fast memcpy call.
*
****************************************************************************/
{
	CHECK(valid() && a.valid());
	if (data != a.data) {
		_size = a._size;
		_delta = a._delta;
		count = a.count;
		delete [] data;
		data = new T[_size];
		if (valid())
			memcpy(data,a.data,count * sizeof(T));
		}
	return *this;
}

template <class T> bool Array<T>::operator == (const Array<T>& a)
/****************************************************************************
*
* Function:		Array<T>::operator ==
* Parameters:	a	- Array to compare to this array
* Returns:		True if the array elements are all equal, false if not.
*
****************************************************************************/
{
	CHECK(valid() && a.valid());
	if (count != a.count)
		return false;
	T *p = data, *pa = a.data;
	for (uint i = 0; i < count; i++)
		if (!(*p++ == *pa++))
			return false;
	return true;
}

template <class T> void Array<T>::insert(T item,uint index,uint aCount)
/****************************************************************************
*
* Function:		Array<T>::insert
* Parameters:	item	- Item to insert into the array
*				index	- Index to insert the item in front of
*				aCount	- Count of elements to insert
*
* Description:	Inserts the specified item into the array at the index'th
*				position. All the elements from [index,count] are moved
*				up one position to make room. The index must be in the
*				range [0,count], and if the value is count it is simply
*				tacked onto the end of the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(index <= count);
	if (count+aCount > _size)	expand(count+aCount);

	// Move the items up one position in the array to make room, and insert

	if (valid()) {
		memmove(&data[index+aCount],&data[index],(count-index) * sizeof(T));
		for (uint i = 0; i < aCount; i++)
			data[index+i] = item;
		count += aCount;
		}
}

template <class T> void Array<T>::remove(uint index)
/****************************************************************************
*
* Function:		Array<T>::remove
* Parameters:	index	- Index of element to remove
*
* Description:	Removes an indexed element from the array, by copying all
*				the data down one position. The index must be in the
*				range [0,count).
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(index < count);

	// Move the items down one position, and shrink the allocated memory

	count--;
	memmove(&data[index],&data[index+1],(count-index) * sizeof(T));
	shrink();
}

template <class T> uint Array<T>::search(T item,uint first,uint last,
	int direction) const
/****************************************************************************
*
* Function:		Array<T>::search
* Parameters:	item		- Item to search for
*				first		- Index of first element to search
*				last		- Index of last element to search
*				direction	- End to search array from (+1 = start, -1 = end)
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a simple linear search for the item from the
*				specified end of the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(first < count && last < count);
	PRECONDITION(direction == +1 || direction == -1);
	if (direction == +1) {
		T *p = &data[first];
		for (uint i = first; i <= last; i++)
			if (*p++ == item)
				return i;
		}
	else {
		T *p = &data[last];
		for (uint i = last; i >= first; i--)
			if (*p-- == item)
				return i;
		}
	return UINT_MAX;
}

template <class T> void Array<T>::expand(uint newSize)
/****************************************************************************
*
* Function:		Array<T>::expand
* Parameters:	newSize	- New size of the array
*
* Description:	Expands the array to be a multiple of 'delta' that includes
*				the specified newSize for the array. The array data is
*				re-allocated and copied to the resized array.
*
*				Note that 'count' must contain the actual number of elements
*				in the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(_delta != 0);	// Array is not resizeable when _delta == 0
	PRECONDITION(newSize >= count);

	T *temp = new T[_size = computeSize(newSize)];

	// Now copy the data from the old array into the newly resized array.
	// Note that we use a fast memcpy to do this.

	if (temp)
		memcpy(temp,data,count * sizeof(T));
	delete [] data;
	data = temp;
}

template <class T> void Array<T>::shrink()
/****************************************************************************
*
* Function:		Array<T>::shrink
*
* Description:	Shrinks the allocated space for the array, if the threshold
*				point has been reached.
*
****************************************************************************/
{
	CHECK(valid());
	if (_delta == 0)		// Array is not resizeable when _delta == 0
		return;

	// Only shrink the array when the amount of free space gets smaller
	// than half of the delta value, and it is at least delta in size

	if ((_size - count) > (_delta + _delta/2) && (_size > _delta)) {
		T *temp = new T[_size = computeSize(_size - _delta)];
		if (temp)
			memcpy(temp,data,count * sizeof(T));
		delete [] data;
		data = temp;
		}
}

template <class T> void Array<T>::setDelta(uint delta)
/****************************************************************************
*
* Function:		Array<T>::setDelta
* Parameters:	delta	- New delta value for the array
*
* Description:	Sets the delta value for the array, expanding or shrinking
*				the array as need be.
*
****************************************************************************/
{
	if (delta >= _delta) {
		_delta = delta;
		expand(_size);
		}
	else {
		_delta = delta;
		shrink();
		}
}

template <class T> void Array<T>::resize(uint newSize)
/****************************************************************************
*
* Function:		Array<T>::resize
* Parameters:	newSize	- New size for the array
*
* Description:	Resizes the array to the new size. If the array is non-
*				resizeable, we bomb out. Note that the array will be empty
*				after this operation.
*
****************************************************************************/
{
	PRECONDITION(_delta != 0);
	empty();
	expand(newSize);
}

template <class T> void Array<T>::empty()
/****************************************************************************
*
* Function:		Array<T>::empty
*
* Description:	Empties the array of all elements.
*
****************************************************************************/
{
	count = 0;
	shrink();
}

template <class T> int SArray<T>::cmp(const void *t1,const void *t2)
/****************************************************************************
*
* Function:		SArray<T>::cmp
* Parameters:	t1,t2	- Elements to compare
* Returns:		Result of comparision:
*
*					t1 < t2,  -1
*					t1 == t2, 0
*					t1 > t2,  1
*
****************************************************************************/
{
	if (*((T*)t1) < *((T*)t2))
		return -1;
	else if (*((T*)t1) > *((T*)t2))
		return 1;
	return 0;
}

template <class T> void SArray<T>::addSorted(T item)
/****************************************************************************
*
* Function:		SArray<T>::addSorted
* Parameters:	item	- Item to add to the array
*
* Description:	Adds the element to the array in sorted order. This
*				function will only work if the elements are already in
*				sorted order, which can be achieved by calling sort().
*
****************************************************************************/
{
	// Search for the spot to put the new item, and insert it into the
	// array.

	CHECK(valid());
	T *p = data;
	for (uint i = 0; i < count; i++)
		if (*p++ > item)
			break;
	insert(item,i);
}

template <class T> uint SArray<T>::binarySearch(T item,uint L,uint R) const
/****************************************************************************
*
* Function:		SArray<T>::binarySearch
* Parameters:	item	- Item to search for in the array
*				L		- Index of first element to search
*				R		- Index of last element to search
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a standard binary search on the array looking
*				for the specified item. The elements in the array _must_
*				be in sorted order for this function to work (by calling
*				the sort() member function).
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(L < count && R < count);

	while (L < R) {
		uint M = (L+R)/2;
		if (data[M] == item)
			return M;
		if (data[M] < item)
			L = M+1;
		else R = M-1;
		}
	if (data[L] == item)
		return L;
	return UINT_MAX;
}

template <class T> const IArray<T>& IArray<T>::operator = (const IArray<T>& a)
/****************************************************************************
*
* Function:		IArray<T>::operator =
* Parameters:	a	- Array to copy
*
* Description:	Assignment operator for IArray's. First empties the array
*				the copies it.
*
****************************************************************************/
{
	// Check to make sure we are not being assigned to ourselves :-)

	CHECK(valid() && a.valid());
	if (data != a.data) {
		empty();
		shouldDelete = false;
		Array<T*>::operator=(a);
		}
	return *this;
}

template <class T> bool IArray<T>::operator == (const IArray<T>& a)
/****************************************************************************
*
* Function:		IArray<T>::operator ==
* Parameters:	a	- Array to compare to this array
* Returns:		True if the array elements are all equal, false if not.
*
****************************************************************************/
{
	CHECK(valid() && a.valid());
	if (count != a.count)
		return false;
	T **p = data, **pa = a.data;
	for (uint i = 0; i < count; i++)
		if (!(*(*p++) == *(*pa++)))
			return false;
	return true;
}

template <class T> void IArray<T>::replace(T* item,uint index)
/****************************************************************************
*
* Function:		IArray<T>::replace
* Parameters:	item	- Item to replace in the array
*				index	- Index of the item in the array to replace
*
* Description:	Replaces the item in the array with the new item. The index
*				MUST fall within the current bounds of the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(index < count);
	if (shouldDelete)
		delete data[index];
	data[index] = item;
}

template <class T> void IArray<T>::destroy(uint index)
/****************************************************************************
*
* Function:		IArray<T>::destroy
* Parameters:	index	- Index of element to remove
*
* Description:	Removes an indexed element from the array, by copying all
*				the data down one position. The index must be in the
*				range [0,count). If shouldDelete is true, the element is
*				deleted.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(index < count);

	// Move the items down one position, and shrink the allocated memory

	if (shouldDelete)
		delete data[index];
	count--;
	memmove(&data[index],&data[index+1],(count-index) * sizeof(T*));
	shrink();
}

template <class T> uint IArray<T>::search(const T* item,uint first,uint last,
	int direction) const
/****************************************************************************
*
* Function:		IArray<T>::search
* Parameters:	item		- Item to search for
*				first		- Index of first element to search
*				last		- Index of last element to search
*				direction	- End to search array from (+1 = start, -1 = end)
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a simple linear search for the item from the
*				specified end of the array.
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(first < count && last < count);
	PRECONDITION(direction == +1 || direction == -1);
	if (direction == +1) {
		T **p = &data[first];
		for (uint i = first; i <= last; i++)
			if (*(*p++) == *item)
				return i;
		}
	else {
		T **p = &data[last];
		for (uint i = last; i >= first; i--)
			if (*(*p--) == *item)
				return i;
		}
	return UINT_MAX;
}

template <class T> void IArray<T>::resize(uint newSize)
/****************************************************************************
*
* Function:		IArray<T>::resize
* Parameters:	newSize	- New size for the array
*
* Description:	Resizes the array to the new size. If the array is non-
*				resizeable, we bomb out. Note that the array will be empty
*				after this operation.
*
****************************************************************************/
{
	PRECONDITION(_delta != 0);
	empty();
	expand(newSize);
}

template <class T> void IArray<T>::empty()
/****************************************************************************
*
* Function:		IArray<T>::empty
*
* Description:	Deletes all of the elements if shouldDelete is set to true
*				(the default).
*
****************************************************************************/
{
	if (shouldDelete) {
		for (uint i = 0; i < count; i++)
			delete (T*)data[i];
		}
	Array<T*>::empty();
}

template <class T> int ISArray<T>::cmp(const void *t1,const void *t2)
/****************************************************************************
*
* Function:		ISArray<T>::cmp
* Parameters:	t1,t2	- Elements to compare
* Returns:		Result of comparision:
*
*					t1 < t2,  -1
*					t1 == t2, 0
*					t1 > t2,  1
*
****************************************************************************/
{
	if (**((T**)t1) < **((T**)t2))
		return -1;
	else if (**((T**)t1) > **((T**)t2))
		return 1;
	return 0;
}

template <class T> void ISArray<T>::addSorted(T* item)
/****************************************************************************
*
* Function:		ISArray<T>::addSorted
* Parameters:	item	- Item to add to the array
*
* Description:	Adds the element to the array in sorted order. This
*				function will only work if the elements are already in
*				sorted order, which can be achieved by calling sort().
*
****************************************************************************/
{
	// Search for the spot to put the new item, and insert it into the
	// array.

	CHECK(valid());
	T **p = data;
	for (uint i = 0; i < count; i++)
		if (*(*p++) > *item)
			break;
	insert(item,i);
}

template <class T> uint ISArray<T>::binarySearch(const T* item,uint L,
	uint R) const
/****************************************************************************
*
* Function:		ISArray<T>::binarySearch
* Parameters:	item	- Item to search for in the array
*				L		- Index of first element to search
*				R		- Index of last element to search
* Returns:		Index of the item in the array, UINT_MAX if not found
*
* Description:	Performs a standard binary search on the array looking
*				for the specified item. The elements in the array _must_
*				be in sorted order for this function to work (by calling
*				the sort() member function).
*
****************************************************************************/
{
	CHECK(valid());
	PRECONDITION(L < count && R < count);

	while (L < R) {
		uint M = (L+R)/2;
		if (*data[M] == *item)
			return M;
		if (*data[M] < *item)
			L = M+1;
		else R = M-1;
		}
	if (*data[L] == *item)
		return L;
	return UINT_MAX;
}

#endif	// __ARRAY_HPP
