/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:     $RCSfile: dlist.hpp $
* Version:      $Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Header file for a class to link objects together into a
*				doubly linked list.
*
* $Id: dlist.hpp 1.1 1994/03/09 12:17:52 kjb release $
*
****************************************************************************/

#ifndef	__DLIST_HPP
#define	__DLIST_HPP

#ifndef	__DEBUG_H
#include "debug.h"
#endif

/*--------------------------- Class Definition ----------------------------*/

//---------------------------------------------------------------------------
// The DListNode class is a simple class used to link the objects in the list
// together. To put anything useful into the list, you must derive the
// object placed into the list from DListNode.
//---------------------------------------------------------------------------

class DListNode {
protected:
	DListNode	*next;
	DListNode	*prev;

	friend class GenDList;
	friend class GenDListIterator;
public:
			// Constructor to satisfy some compilers :-(
			DListNode() {};

			// Virtual destructor to delete a list node
	virtual	~DListNode();
	};

//---------------------------------------------------------------------------
// The list class is designed to manipulate a list of DListNode objects.
// In the simple form, DListNode objects contain nothing special. To add
// an arbitrary class to the list, you must derive the class from DListNode
// (either through single or multiple inheritance).
//---------------------------------------------------------------------------

class GenDList {
protected:
	ulong		count;		// Number of objects in list
	DListNode	*head;		// Pointer to first node in list
	DListNode	*z;			// Pointer to last node in list
	DListNode	hz[2];		// Space for head and z nodes

	static	int (*cmp)(DListNode*,DListNode*);

			// Protected member to merge two lists together
			DListNode* merge(DListNode* a,DListNode* b, DListNode*& end);

public:
			// Constructor
			GenDList();

			// Destructor
			~GenDList();

			// Method to examine the first node in the List
			DListNode* peekHead() const;

			// Method to examine the last node in the List
			DListNode* peekTail() const;

			// Method to return the next node in the list
			DListNode* next(DListNode* node) const;

			// Method to return the prev node in the list
			DListNode* prev(DListNode* node) const;

			// Method to add a node to the head of the list
			void addToHead(DListNode* node);

			// Method to add a node to the tail of the list
			void addToTail(DListNode* node);

			// Method to add a node after another node in the list
			void addAfter(DListNode* node,DListNode* after);

			// Method to detach a specified DListNode from the list.
			DListNode* remove(DListNode* node);

			// Method to detach a specified DListNode from the list.
			DListNode* removeNext(DListNode *prev);

			// Method to detach the first node from the list.
			DListNode* removeFromHead();

			// Method to detach the last node from the list.
			DListNode* removeFromTail();

			// Sort the linked list of objects
			void sort(int (*cmp)(DListNode*,DListNode*));

			// Empties the entire list by destroying all nodes
			void empty();

			// Returns the number of items in the list
			ulong numberOfItems() const	{ return count; };

			// Returns true if the list is empty
			bool isEmpty() const	{ return count == 0; };

private:
	friend class GenDListIterator;
	};

//---------------------------------------------------------------------------
// The list iterator is the class of iterator that is used to step through
// the elements in the list.
//---------------------------------------------------------------------------

class GenDListIterator {
protected:
	DListNode	*cursor;
	GenDList	*beingIterated;
public:
			// Constructor
			GenDListIterator();

			// Constructor given a list reference
			GenDListIterator(const GenDList& l);

			// Intialise a list iterator from a list
			void operator = (const GenDList& l);

			// assignment operator between two listIterators
			void operator = (const GenDListIterator& i);

			// Overloaded cast to an integer
			operator int ();

			// Convert the iterator to the corresponding node
			DListNode* node();

			// Pre-increment operator for the iterator
			DListNode* operator ++ ();

			// Post-increment operator for the iterator
			DListNode* operator ++ (int);

			// Pre-decrement operator for the iterator
			DListNode* operator -- ();

			// Post-decrement operator for the iterator
			DListNode* operator -- (int);

			// Method to restart the iterator at head of list
			void restart();

			// Method to restart the iterator at tail of list
			void restartTail();
	};

//---------------------------------------------------------------------------
// Template wrapper class for declaring Type Safe doubly linked lists.
//---------------------------------------------------------------------------

typedef	int (*_GenDListCmp)(DListNode*,DListNode*);

template <class T> class DList : public GenDList {
public:
			T* peekHead() const
				{ return (T*)GenDList::peekHead(); };
			T* peekTail() const
				{ return (T*)GenDList::peekTail(); };
			T* next(T* node) const
				{ return (T*)GenDList::next(node); };
			T* prev(T* node) const
				{ return (T*)GenDList::prev(node); };
			T* remove(T* node)
				{ return (T*)GenDList::remove(node); };
			T* removeNext(T* node)
				{ return (T*)GenDList::removeNext(node); };
			T* removeFromHead()
				{ return (T*)GenDList::removeFromHead(); };
			T* removeFromTail()
				{ return (T*)GenDList::removeFromTail(); };
			void sort(int (*cmp)(T*,T*))
				{ GenDList::sort((_GenDListCmp)cmp); };
	};

template <class T> class DListIterator : public GenDListIterator {
public:
			DListIterator()
				: GenDListIterator() {};
			DListIterator(const DList<T>& l)
				: GenDListIterator(l) {};
			void operator = (const DList<T>& l)
				{ GenDListIterator::operator=(l); };
			void operator = (const DListIterator<T>& i)
				{ GenDListIterator::operator=(i); };
			T* node()
				{ return (T*)GenDListIterator::node(); };
			T* operator ++ ()
				{ return (T*)GenDListIterator::operator++(); };
			T* operator ++ (int)
				{ return (T*)GenDListIterator::operator++(1); };
			T* operator -- ()
				{ return (T*)GenDListIterator::operator--(); };
			T* operator -- (int i)
				{ return (T*)GenDListIterator::operator--(i); };
	};

/*------------------------ Inline member functions ------------------------*/

inline DListNode* GenDList::peekHead() const
/****************************************************************************
*
* Function:		GenDList::peekHead
* Returns:		Returns a pointer to the head node on the list, or NULL if
*				the list is empty.
*
****************************************************************************/
{
	return (head->next == z ? NULL : head->next);
}

inline DListNode* GenDList::peekTail() const
/****************************************************************************
*
* Function:		GenDList::peekTail
* Returns:		Returns a pointer to the tail node on the list, or NULL if
*				the list is empty.
*
****************************************************************************/
{
	return (z->prev == head ? NULL : z->prev);
}

inline DListNode* GenDList::next(DListNode *node) const
/****************************************************************************
*
* Function:		GenDList::next
* Parameters:	node	- Node to obtain next from
* Returns:		Pointer to the next node in the list, NULL if none.
*
****************************************************************************/
{
	return (node->next == z ? NULL : node->next);
}

inline DListNode* GenDList::prev(DListNode *node) const
/****************************************************************************
*
* Function:		GenDList::prev
* Parameters:	node	- Node to obtain prev from
* Returns:		Pointer to the previous node in the list, NULL if none.
*
****************************************************************************/
{
	return (node->prev == head ? NULL : node->prev);
}

inline void GenDList::addAfter(DListNode* node,DListNode* after)
/****************************************************************************
*
* Function:		GenDList::addAfter
* Parameters:	node	- Node to attach new node after in list
*				after	- New node to attach to list
*
* Description:	Attaches a new node after a specified node in the list.
*				The list must contain at least one node, and after may
*				be the tail node of the list.
*
****************************************************************************/
{
	node->next = after->next;
	after->next = node;
	node->prev = after;
	node->next->prev = node;
	count++;
}

inline void GenDList::addToHead(DListNode* node)
/****************************************************************************
*
* Function:		GenDList::addToHead
* Parameters:	node	- Node to add to list
*
* Description:	Attaches the node to the head of the list.
*
****************************************************************************/
{
	addAfter(node,head);
}

inline void GenDList::addToTail(DListNode* node)
/****************************************************************************
*
* Function:		GenDList::addToTail
* Parameters:	node	- Node to add to list
*
* Description:	Attaches the node to the tail of the list.
*
****************************************************************************/
{
	addAfter(node,z->prev);
}

inline DListNode* GenDList::remove(DListNode* node)
/****************************************************************************
*
* Function:		GenDList::remove
* Parameters:	node	- Pointer to node remove from the list
* Returns:		Node removed from list.
*
* Description:	Removes the specified node from the list.
*
****************************************************************************/
{
	node->next->prev = node->prev;
	node->prev->next = node->next;
	count--;
	return node;
}

inline DListNode* GenDList::removeNext(DListNode* prev)
/****************************************************************************
*
* Function:		GenDList::removeNext
* Parameters:	prev	- Pointer to the previous node in the list
* Returns:		Node removed from list, or NULL if prev is the last node.
*
* Description:	Removes the specified node from the list.
*
****************************************************************************/
{
	DListNode*	node;

	if ((node = prev->next) != z)
		return remove(node);
	else
		return NULL;
}

inline DListNode* GenDList::removeFromHead()
/****************************************************************************
*
* Function:		GenDList::removeFromHead
* Returns:		Pointer to the node removed from the head of the list,
*				or NULL if the list is empty.
*
****************************************************************************/
{
	return removeNext(head);
}

inline DListNode* GenDList::removeFromTail()
/****************************************************************************
*
* Function:		GenDList::removeFromTail
* Returns:		Pointer to the node removed from the tail of the list,
*				or NULL if the list is empty.
*
****************************************************************************/
{
	return removeNext(z->prev->prev);
}

inline GenDListIterator::GenDListIterator()
/****************************************************************************
*
* Function:		GenDListIterator::GenDListIterator
*
* Description:	Default constructor for a dlist iterator.
*
****************************************************************************/
{
	cursor = NULL;
	beingIterated = NULL;
}

inline GenDListIterator::GenDListIterator(const GenDList& l)
/****************************************************************************
*
* Function:		GenDListIterator::GenDListIterator
* Parameters:	l	- DList to construct iterator from
*
* Description:	Constructor for a GenDListIterator given a reference to a list
*				to iterate.
*
****************************************************************************/
{
	beingIterated = (GenDList*)&l;
	cursor = l.head->next;
}

inline void GenDListIterator::operator = (const GenDList& l)
/****************************************************************************
*
* Function:		GenDListIterator::operator =
* Parameters:	l	- GenDList to assign to iterator
*
* Description:	Assignment operator for a DListIterator given a reference to
*				a list to iterate.
*
****************************************************************************/
{
	beingIterated = (GenDList*)&l;
	cursor = l.head->next;
}

inline void GenDListIterator::operator = (const GenDListIterator& i)
/****************************************************************************
*
* Function:		GenDListIterator::operator =
* Parameters:	i	- Iterator to assign from
*
* Description:	Assignment operator for a DListIterator given a reference to
*				another DListIterator.
*
****************************************************************************/
{
	beingIterated = i.beingIterated;
	cursor = i.cursor;
}

inline GenDListIterator::operator int()
/****************************************************************************
*
* Function:		GenDListIterator::operator int
*
* Description:	Overloaded cast to integer for the list iterator. Evaluates
*				to 0 when the end of the list is reached.
*
****************************************************************************/
{
	return (cursor != beingIterated->z && cursor != beingIterated->head);
}

inline DListNode* GenDListIterator::node()
/****************************************************************************
*
* Function:		GenDListIterator::node
* Returns:		Returns a reference to the node in the list.
*
****************************************************************************/
{
	return ((int)*this ? cursor : NULL);
}

inline DListNode* GenDListIterator::operator ++ ()
/****************************************************************************
*
* Function:		GenDListIterator::operator ++
* Returns:		Pointer to node after incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				after the increment takes place.
*
****************************************************************************/
{
	if ((int)*this)
		return cursor = cursor->next;
	else
		return NULL;
}

inline DListNode* GenDListIterator::operator ++ (int)
/****************************************************************************
*
* Function:		GenDListIterator::operator ++ (int)
* Returns:		Pointer to node before incrementing
*
* Description:	Increments the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				before the increment takes place.
*
****************************************************************************/
{
	DListNode	*prev = cursor;

	if ((int)*this) {
		cursor = cursor->next;
		return prev;
		}
	else
		return NULL;
}

inline DListNode* GenDListIterator::operator -- ()
/****************************************************************************
*
* Function:		GenDListIterator::operator --
* Returns:		Pointer to node after decrementing
*
* Description:	Decrements the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				after the decrement takes place.
*
****************************************************************************/
{
	if ((int)*this)
		return cursor = cursor->prev;
	else
		return NULL;
}

inline DListNode* GenDListIterator::operator -- (int)
/****************************************************************************
*
* Function:		GenDListIterator::operator -- (int)
* Returns:		Pointer to node before decrementing
*
* Description:	Decrements the iterator by moving it to the next object
*				in the list. We return a pointer to the node pointed to
*				before the decrement takes place.
*
****************************************************************************/
{
	DListNode	*prev = cursor;

	if ((int)*this) {
		cursor = cursor->prev;
		return prev;
		}
	else
		return NULL;
}

inline void GenDListIterator::restart()
/****************************************************************************
*
* Function:		GenDListIterator::restart
*
* Description:	Restart the iterator at the beginning of the list.
*
****************************************************************************/
{
	cursor = beingIterated->head->next;
}

inline void GenDListIterator::restartTail()
/****************************************************************************
*
* Function:		GenDListIterator::restartTail
*
* Description:	Restart the iterator at the end of the list.
*
****************************************************************************/
{
	cursor = beingIterated->z->prev;
}

#endif	// __DLIST_HPP
