/****************************************************************************
*
*						  Techniques Class Library
*
*                   Copyright (C) 1994 SciTech Software.
*							All rights reserved.
*
* Filename:		$RCSfile: substr.cpp $
* Version:		$Revision: 1.1 $
*
* Language:		C++ 3.0
* Environment:	any
*
* Description:	Member functions for the SubStr class, a class for
*				representing substrings of dynamically sized strings or
*				c style strings.
*
* $Id: substr.cpp 1.1 1994/03/09 12:18:03 kjb release $
*
****************************************************************************/

#include "str.hpp"
#include "techlib.hpp"
#include <limits.h>
#include <iomanip.h>

//------------------------- Global initialisations ------------------------//

char	empty[] = "Empty SubStr";		// Empty substring string

//----------------------------- Member functions --------------------------//

SubStr::SubStr()
/****************************************************************************
*
* Function:		SubStr::SubStr
*
* Description:	Constructor given no parameters. All we do here point
*				the SubStr at the empty string above.
*
****************************************************************************/
{
	len = sizeof(empty) + 1;		// Length includes the null terminator!
	text = empty;
}

SubStr::SubStr(const String& str,uint pos,uint count)
/****************************************************************************
*
* Function:		SubStr::SubStr
* Parameters:	str		- String to copy from
*				pos		- Starting position in the string
*				count	- Number of characters to copy
*
* Description:	Constructs a SubStr from another string, starting at the
*				position 'pos' and including 'count' characters.
*
****************************************************************************/
{
	CHECK(str.valid());
	if (pos > str.length())
		pos = str.length();
	if (count > str.length() - pos)
		count = str.length() - pos;
	len = count+1;
	text = (char *)((const char *)str + pos);
}

SubStr::SubStr(const char *cstr,uint pos,uint count)
/****************************************************************************
*
* Function:		SubStr::SubStr
* Parameters:	cstr	- C style string to copy from
*				pos		- Starting position in the string
*				count	- Number of characters to copy
*
* Description:	Constructs a SubStr from a C string, starting at the
*				position 'pos' and including 'count' characters.
*
*				The count and position are assumed to be valid if count
*				is not set to UINT_MAX.
*
****************************************************************************/
{
	CHECK(cstr != NULL);
	if (count == UINT_MAX) {
		len = strlen(cstr) + 1;
		if (pos >= len)
			pos = len-1;
		if (count >= len - pos)
			count = len-1 - pos;
		}
	len = count+1;
	text = (char*)cstr + pos;
}

SubStr& SubStr::operator = (const String& str)
/****************************************************************************
*
* Function:		SubStr::operator =
* Parameters:	str	- String to assign from
* Returns:		Reference to the newly created string.
*
* Description:	Assignment operator given a string.
*
****************************************************************************/
{
	CHECK(str.valid());
	len = str.length()+1;
	text = (char*)((const char *)str);
	return *this;
}

SubStr& SubStr::operator = (const char *cstr)
/****************************************************************************
*
* Function:		SubStr::operator =
* Parameters:	cstr	- C style string to assign
* Returns:		Reference to the newly allocated string.
*
* Description:	Assignment operator given a C style string.
*
****************************************************************************/
{
	CHECK(cstr != NULL);
	len = strlen(cstr)+1;
	text = (char*)cstr;
	return *this;
}

SubStr::operator DynStr () const
/****************************************************************************
*
* Function:		SubStr::operator DynStr
* Returns:		A newly constructed dynamic string.
*
* Description:	This routine casts a SubStr to a dynamically allocated. We
*				do this by constructing a new dynamic string and returning
*				this to the user.
*
****************************************************************************/
{
	return DynStr(*this,0,len-1);
}

SubStr& SubStr::left(uint count)
/****************************************************************************
*
* Function:		SubStr::left
* Parameters:	count	- Number of characters to keep
* Returns:		Reference to the new substring.
*
* Description:	Converts the substring to represent only the left count
*				characters.
*
****************************************************************************/
{
	len = MIN(count,length()) + 1;
	return *this;
}

SubStr& SubStr::right(uint count)
/****************************************************************************
*
* Function:		SubStr::right
* Parameters:	count	- Number of characters to keep
* Returns:		Reference to the new substring.
*
* Description:	Converts the substring to represent only the right count
*				characters.
*
****************************************************************************/
{
	if (count < length()) {
		text += length() - count;
		len = count+1;
		}
	return *this;
}

SubStr& SubStr::mid(uint pos,uint count)
/****************************************************************************
*
* Function:		SubStr::mid
* Parameters:	pos		- Position to start at
*				count	- Number of characters to keep
* Returns:		Reference to the new substring.
*
* Description:	Converts the substring to represent the middle count
*				characters starting at pos.
*
****************************************************************************/
{
	if (pos > length())
		pos = length();
	if (count > length() - pos)
		count = length() - pos;
	len = count+1;
	text += pos;
	return *this;
}

ostream& operator << (ostream& o,const SubStr& s)
/****************************************************************************
*
* Function:		operator <<
* Parameters:	o	- Stream to dump string to
*				s	- SubStr to dump
* Returns:		Stream to dump data to
*
****************************************************************************/
{
	int		i;
	char	*p;

	for (i = 0,p = s.text; i < s.length(); i++,p++)
		o << *p;
	return o;
}
