//	implementation of ScrollWindow class
//	David A. Seidel 8/90

#include "scrlwind.h"

static const char START_ICON = '[',
				  END_ICON   = ']',
				  UP_ARROW   = '\x018',
				  DN_ARROW   = '\x019';
static const char *BEGIN = "Top",
				  *END   = "Bot";

//	constructor
ScrollWindow::ScrollWindow (char *title,
							int left, int top, int right, int bottom,
							int attr, boxType boxT, int useIcons)
			 : FrameWindow (title, left, top, right, bottom, attr, boxT)
{
	cursor(_NOCURSOR);
	scroll(0);
	scrollFlag = 1;
	pageSize = curHeight();
	curPage = 0;
	hasUp = 0;
	hasDn = 0;
	hasTop = 0;
	hasBot = 0;
	iconFlag = useIcons ? 1 : 0;
	normAttr = attr;
	if ((attr % 16) > 8)
	{
		// high-intensity
		iconAttr = normAttr - 8;			// use low for icons
		emphAttr = (normAttr * 16) % 255;	// use reverse for emphasize
		highAttr = (iconAttr * 16) % 255;	// use reverse-low for highlight
	}
	else
	{
		// low intensity
		iconAttr = normAttr + 8;			// use high for icons
		emphAttr = (iconAttr * 16) % 255;	// use reverse-high for emphasize
		highAttr = (normAttr * 16) % 255;	// use reverse for highlight
	}
}

//	move up one line
void ScrollWindow::lineUp()
{
	if (!scrollFlag)
		return;
	if (curY() == 1 && curPage == 1)
		return;
	display();
	if (curY() == 1)
	{
		begLine();
		newLine();
		--curPage;
		if (prevPage > 0)
			--prevPage;
	}
	else
		relY(-1);
	reDisplay(highAttr);
	updateIcons();
}

//	move down one line
void ScrollWindow::lineDn()
{
	if (!scrollFlag)
		return;
	if (curItem == numItems)
		return;
	display();
	if (curY() == curHeight())
	{
		endLine();
		conout << endl;
		++curPage;
		++prevPage;
	}
	else
		relY(+1);
	reDisplay(highAttr);
	updateIcons();
}

//	move up one page
void ScrollWindow::pageUp()
{
	if (!scrollFlag)
		return;
	if (curPage == 1)
		return;
	clear();
	if (curPage == prevPage)
		curPage = prevPage = 1;
	else
	{
		curPage = prevPage;
		if (prevPage >= pageSize)
			prevPage -= pageSize;
		else
			prevPage = 1;
	}
	gotoItem(curPage);
	display(numItems >= pageSize ? pageSize : numItems);
	home();
	reDisplay(highAttr);
	updateIcons();
}

//	move down one page
void ScrollWindow::pageDn()
{
	if (!scrollFlag)
		return;
	if (curPage + pageSize >= numItems)
		return;
	clear();
	int balance = numItems - (curPage + pageSize - 1);
	if (balance <= pageSize)
	{
		if (prevPage != curPage)
			prevPage += balance;
		curPage += balance;
	}
	else
	{
		prevPage = curPage;
		curPage += pageSize;
	}
	gotoItem(curPage);
	display(numItems >= pageSize ? pageSize : numItems);
	home();
	reDisplay(highAttr);
	updateIcons();
}

//	move to the top
void ScrollWindow::pageTop()
{
	if (!scrollFlag)
		return;
	if (curPage == 1 && curY() == 1)
		return;
	if (curPage == 1)
		display();
	else
	{
		clear();
		curPage = prevPage = 1;
		gotoItem(curPage);
		display(numItems >= pageSize ? pageSize : numItems);
	}
	home();
	reDisplay(highAttr);
	updateIcons();
}

//	move to the end
void ScrollWindow::pageEnd()
{
	if (!scrollFlag)
		return;
	if (curItem == numItems)
		return;
	if (numItems > pageSize)
	{
		int lastPage = numItems - pageSize + 1;
		if (curPage < lastPage)
		{
			clear();
			curPage = lastPage;
			if (curPage > pageSize)
				prevPage = curPage - pageSize;
			else
				prevPage = 1;
			gotoItem(curPage);
			display(pageSize);
		}
		else
			display();
		go(1, curHeight());
	}
	else
	{
		display();
		relY(+(numItems-curY()));
		gotoItem(curY());
	}
	reDisplay(highAttr);
	updateIcons();
}

//	redisplay the current item with some special attribute
void ScrollWindow::reDisplay (UCHAR specAttr)
{
	gotoItem(curPage+curY()-1);
	attr(specAttr);
	display();
	attr(normAttr);
	begLine();
}

//	update the icons
ScrollWindow& ScrollWindow::updateIcons()
{
	if (!iconFlag)
		return *this;
	if (curPage == 1)
		putBegIcon();
	else
		putUpIcon();
	if (curPage + curHeight() >= numItems)
		putEndIcon();
	else
		putDnIcon();
	return *this;
}

//	make an up arrow icon on top line
ScrollWindow& ScrollWindow::putUpIcon()
{
	if (!iconFlag)
		return *this;
	if (!hasUp)
	{
		UCHAR tmp = boxSet[frameType].Horiz;
		curNumber = 0;
		restoreFrame();
		gotoxy(frameWidth()-5, 1);
		conout << tmp << START_ICON;
		textattr(iconAttr);
		conout << UP_ARROW;
	 	textattr(fState.attribute);
		conout << END_ICON << tmp;
		hasUp = 1;
		hasTop = 0;
		restoreState();
	}
	return *this;
}

//	make a 'Top' icon on top line
ScrollWindow& ScrollWindow::putBegIcon()
{
	if (!iconFlag)
		return *this;
	if (!hasTop)
	{
		curNumber = 0;
		restoreFrame();
		gotoxy(frameWidth()-5, 1);
		conout << START_ICON;
		textattr(iconAttr);
		conout << BEGIN;
	 	textattr(fState.attribute);
		conout << END_ICON;
		hasTop = 1;
		hasUp = 0;
		restoreState();
	}
	return *this;
}

//	make a down arrow icon on bottom line
ScrollWindow& ScrollWindow::putDnIcon()
{
	if (!iconFlag)
		return *this;
	if (!hasDn)
	{
		UCHAR tmp = boxSet[frameType].Horiz;
		curNumber = 0;
		restoreFrame();
		gotoxy(frameWidth()-5, frameHeight());
		conout << tmp << START_ICON;
		textattr(iconAttr);
		conout << DN_ARROW;
	 	textattr(fState.attribute);
		conout << END_ICON << tmp;
		hasDn = 1;
		hasBot = 0;
		restoreState();
	}
	return *this;
}

//	make an 'End' icon on bottom line
ScrollWindow& ScrollWindow::putEndIcon()
{
	if (!iconFlag)
		return *this;
	if (!hasBot)
	{
		curNumber = 0;
		restoreFrame();
		gotoxy(frameWidth()-5, frameHeight());
		conout << START_ICON;
		textattr(iconAttr);
		conout << END;
	 	textattr(fState.attribute);
		conout << END_ICON;
		hasBot = 1;
		hasDn = 0;
		restoreState();
	}
	return *this;
}

//	clean off any icons
ScrollWindow& ScrollWindow::cleanIcons()
{
	if (!iconFlag)
		return *this;
	UCHAR tmp[6];
	memset(tmp, boxSet[frameType].Horiz, 5);
	tmp[5] = '\0';
	curNumber = 0;
	restoreFrame();
	if (hasUp || hasTop)
	{
		gotoxy(frameWidth()-5, 1);
		conout << tmp;
	}
	if (hasDn || hasBot)
	{
		gotoxy(frameWidth()-5, frameHeight());
		conout << tmp;
	}
	hasUp = 0;
	hasDn = 0;
	hasTop = 0;
	hasBot = 0;
	restoreState();
	return *this;
}

//	end of scrlwind.cpp
