//	implementation of TextWindow class
//	David A. Seidel, 8/90

#include "textwind.h"
#include <alloc.h>
#include <string.h>

int TextWindow::wIndex = 0;

//	constructor
TextWindow::TextWindow (int left, int top, int right, int bottom, int attr)
{
	gettextinfo(&oldState);
	textattr(attr);
	scrollFlag = 1;
	cursorType = _NORMALCURSOR;
	_setcursortype(cursorType);
	window(left, top, right, bottom);
	refreshState();
	hideBuffer = 0;
	if (checkBounds(left, top, right, bottom))
	{
		saveBuffer = new char[2*(bottom-top+3)*(right-left+3)];
		if (saveBuffer != 0)
			gettext(left-1, top-1, right+1, bottom+1, saveBuffer);
	}
	clrscr();
	curNumber = thisNumber = ++wIndex;
}

//	destructor
TextWindow::~TextWindow()
{
	if (saveBuffer != 0)
	{
		restoreState();
		puttext(wState.winleft-1, wState.wintop-1,
				wState.winright+1, wState.winbottom+1,
				saveBuffer);
		delete saveBuffer;
		if (hideBuffer != 0)
			delete hideBuffer;
	}
	restoreOldState();
}

//	hide the window
void TextWindow::hide()
{
	hideBuffer = new char[2*(wState.winbottom-wState.wintop+3) * 
						  (wState.winright-wState.winleft+3)];
	gettext(wState.winleft-1, wState.wintop-1,
			wState.winright+1, wState.winbottom+1,
			hideBuffer);
	puttext(wState.winleft-1, wState.wintop-1,
			wState.winright+1, wState.winbottom+1,
			saveBuffer);
}

// unhide the window
void TextWindow::unhide()
{
	gettext(wState.winleft-1, wState.wintop-1,
			wState.winright+1, wState.winbottom+1,
			saveBuffer);
	puttext(wState.winleft-1, wState.wintop-1,
			wState.winright+1, wState.winbottom+1,
			hideBuffer);
}

void TextWindow::restoreState()
{
	if (curNumber != thisNumber)
	{
		curNumber = thisNumber;
		window(wState.winleft, wState.wintop,
			   wState.winright, wState.winbottom);
		textattr(wState.attribute);
		_wscroll = scrollFlag;
		_setcursortype(cursorType);
		gotoxy(wState.curx, wState.cury);
	}
}

void TextWindow::restoreOldState()
{
	window(oldState.winleft, oldState.wintop,
	   oldState.winright, oldState.winbottom);
	textattr(oldState.attribute);
	gotoxy(oldState.curx, oldState.cury);
	_setcursortype(_NORMALCURSOR);
}

int TextWindow::checkBounds(int left,  int top,
							   int right, int bottom)
{
	if (left >= minX &&
		top  >= minY &&
		right <= maxX &&
		bottom <= maxY)
		return 1;
	return 0;
}

TextWindow& TextWindow::attr (int attr)
{
	restoreState();
	textattr(attr);
	refreshState();
	return *this;
}

TextWindow& TextWindow::attr (int fore, int back)
{
	restoreState();
	textattr(fore + (back << 4));
	refreshState();
	return *this;
}

TextWindow& TextWindow::fore (int fg)
{
	restoreState();
	textcolor(fg);
	refreshState();
	return *this;
}

TextWindow& TextWindow::back (int bg)
{
	restoreState();
	textbackground(bg);
	refreshState();
	return *this;
}

TextWindow& TextWindow::scroll (int flag)
{
	restoreState();
	flag != 0 ? scrollFlag = _wscroll = 1 :
				scrollFlag = _wscroll = 0;
	refreshState();
	return *this;
}

TextWindow& TextWindow::cursor (int cursorT)
{
	restoreState();
	_setcursortype(cursorT);
	cursorType = cursorT;
	refreshState();
	return *this;
}

TextWindow& TextWindow::go (int x, int y)
{
	restoreState();
	gotoxy(x, y);
	refreshState();
	return *this;
}

TextWindow& TextWindow::clear()
{
	restoreState();
	clrscr();
	refreshState();
	return *this;
}

TextWindow& TextWindow::home()
{
	restoreState();
	gotoxy(1, 1);
	refreshState();
	return *this;
}

TextWindow& TextWindow::bottom()
{
	restoreState();
	gotoxy(1, curHeight());
	refreshState();
	return *this;
}

TextWindow& TextWindow::begLine()
{
	restoreState();
	gotoxy(1, wState.cury);
	refreshState();
	return *this;
}

TextWindow& TextWindow::endLine()
{
	restoreState();
	gotoxy(curWidth(), wState.cury);
	refreshState();
	return *this;
}

TextWindow& TextWindow::relX (int delta_x)
{
	restoreState();
	gotoxy(wState.curx+delta_x, wState.cury);
	refreshState();
	return *this;
}

TextWindow& TextWindow::relY (int delta_y)
{
	restoreState();
	gotoxy(wState.curx, wState.cury+delta_y);
	refreshState();
	return *this;
}

TextWindow& TextWindow::clrEOL()
{
	restoreState();
	clreol();
	refreshState();
	return *this;
}

TextWindow& TextWindow::delLine()
{
	restoreState();
	delline();
	refreshState();
	return *this;
}

TextWindow& TextWindow::newLine()
{
	restoreState();
	insline();
	refreshState();
	return *this;
}

TextWindow& TextWindow::operator>> (char *str)
{
	char *p;

	restoreState();
	inBuf[0] = maxIn;
	p = cgets(inBuf);
	strcpy(str, p);
	refreshState();
	return *this;
}

//	end of textwind.cpp
