//--------------------------------------------------------------------------
//
//      CLOCK.CPP: Pop-up clock demo for DOS TSR class.
//      (c) J.English 1993 (je@unix.brighton.ac.uk)
//
//      This example pops up a clock in the top right-hand corner of the
//      screen when the hotkey is pressed.
//
//--------------------------------------------------------------------------

#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <dos.h>
#include <ctype.h>
#include "tsr.h"

//----- Error messages for TSR status values -----
const char* status_msg [] = {
    "success",
    "DOS version 3 or higher required",
    "multiple TSR instances declared in program", 
    "can't allocate stack",
    "can't allocate multiplex function code",
    "already loaded",
    "can't make program resident",
    "startup error"
};

//----- Error messages for unload result -----
const char* unload_msg [] = {
    "unloaded",
    "not loaded",
    "can't be unhooked",
    "can't be unloaded",
    "environment can't be unloaded"
};

//----- The derived TSR class -----
class PopUpClock : public TSR
{
  private:
    virtual void main (int hotkey);     // body of clock TSR
    int visible;                        // true if clock is visible
  public:
    PopUpClock () :                     // constructor
        TSR ("TSR clock demo"),         // ... call base class constructor
        visible (0)                     // ... clock initially invisible
        { }                             // ... nothing else needed
};

//----- The main program -----
int main (int argc, char** argv)
{
    PopUpClock clock;                   // create clock instance

    switch (argc)
    {
      case 1:                           // invoked by CLOCK -- load it
        cout << "Installing " << clock.name() << " in memory\n";
        cout << "Press CTRL-SPACE to activate and deactivate\n";

        clock.run (TSR::CTRL + TSR::SPACE, 6);  // timeslice = 6 x 55 = 330ms
                                                // (error if this call returns)
        cout << "Error installing " << clock.name() << ": "
             << status_msg [clock.status()] << "\n";
        break;

      case 2:                           // invoked by CLOCK/U -- unload it
        if (argv[1][0] == '/' && toupper (argv[1][1]) == 'U')
        {   cout << clock.name() << " "
                 << unload_msg [clock.unload()] << "\n";
            break;
        }

      default:                          // give usage message otherwise
        cout << "Usage: CLOCK [/U]\n";
        return -1;
    }
    return clock.status ();
}

//----- Resident part of TSR, executed every 6 clock ticks -----
void PopUpClock::main (int hotkey)
{
    if (hotkey)
        visible = !visible;                 // hotkey toggles visibility
    if (!visible)
        return;                             // nothing to do if invisible

    union REGS r;                           // get video mode
    r.h.ah = 0x0F;
    int86 (0x10, &r, &r);

    if (r.h.al > 3 && r.h.al != 7)          // exit if not text mode (0-3, 7)
        return;

    struct time now;                        // get current time
    gettime (&now);

    char hms [11];                          // format time into string
    sprintf (hms, " %02d:%02d:%02d ",
             now.ti_hour, now.ti_min, now.ti_sec);

    char far* screen =                      // get address of screen
        (char far*) MK_FP(0xB800 + r.h.bh * 256, (r.h.ah - strlen(hms)) * 2);

    char* c = hms;                          // output formatted time
    while (*c != '\0')
    {   *screen++ = *c++;
        *screen++ = 0x1E;
    }
}
