/***
*spawnve.c - spawn a child process, final procedure called by spawnXX
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines spawnve() - the low level routine finally called by spawnXX
*
*******************************************************************************/

#include <register.h>
#include <process.h>
#include <errno.h>
#include <io.h>
#include <stdlib.h>
#include <string.h>
#include <internal.h>
#include <malloc.h>
#include <mbstring.h>

#pragma check_stack+        /* turn stack checking on */

#define SLASHCHAR   '\\'
#define XSLASHCHAR  '/'


/***
*static int comexebat(mode, name, argv, envp, notbatchmode)- do the spawn after
*                                name fixup
*
*Purpose:
*   Spawns a child process with given parameters and environment.  Either
*   overlays current process or loads in free memory while parent process
*   waits.  If the named file is a .bat file, modifies the calling sequence
*   and prepends the /c and filename arguments into the command string
*
*Entry:
*   int mode - mode to spawn (WAIT, NOWAIT, or OVERLAY)
*       only WAIT and OVERLAY currently supported
*   char *name - pathname of file to spawn.  Includes the extension
*   char **argv - vector of parameter strings
*   char **envp - vector of environment variables
*   int notbatchmode - equal to zero when processing a .BAT file
*
*Exit:
*   returns exit code of child process
*   if fails, returns -1
*
*Exceptions:
*
*******************************************************************************/

#define COM_STR_SIZE  0x80

static int comexebat (int, const char *, const char * const *,
            const char * const *, int);

static int
comexebat (mode, name, argv, envp, notbatchmode)
REG3 int mode;
REG2 const char *name;
const char * const *argv;
const char * const *envp;
int notbatchmode;  /* do special processing if zero*/

{
    char *envblock;
    char command[COM_STR_SIZE];
    char *save;
    REG4 int rc;
    const char *batname;


    if (!notbatchmode) {
        batname = name; /* cause the "/c filename" to be pre-pended */

        if(!(name = getenv("COMSPEC"))) {
        errno = ENOEXEC;
        return -1;
        }

    } else {
        batname = NULL;
    }

    if (_cenvarg(argv, envp, &save, &envblock, command, NULL, batname) == -1)
        return -1;

    rc = _dospawn(mode, name, command, envblock);
    free(save);
    return rc;
}


/***
*int _spawnve(mode, name, argv, envp) - spawn a child process
*
*Purpose:
*
*   If the filename given has no extension, tries .com, .exe, .bat in
*   that order.  Calls comexebat() to do the actual work
*
*Entry:
*   int mode - mode to spawn (WAIT, NOWAIT, or OVERLAY)
*       only WAIT and OVERLAY currently supported
*   char *name - pathname of file to spawn
*   char **argv - vector of parameters
*   char **envp - vector of environment variables
*
*Exit:
*   returns exit code of child process
*   if fails, retruns -1
*
*Exceptions:
*
*******************************************************************************/

static char near *exstrings[] = { ".bat", ".exe", ".com" };
            /* we actually try these in reverse order */

int
_spawnve (mode, name, argv, envp)
REG3 int mode;
REG2 const char *name;
const char * const *argv;
const char * const *envp;
{
    char *ext;      /* where the extension goes if we have to add one */
    REG1 char *p;
    char *q;
    REG4 int rc;
    REG5 int i;
    int tempamblksiz;   /* old amblksiz */

    if (mode == _OLD_P_OVERLAY)
    return(_execload(name, argv, envp));

    /* In a SBCS build, _mbsrchr maps to strrchr.   */

    p = _mbsrchr(name, SLASHCHAR);
    q = strrchr(name, XSLASHCHAR);

    if (!q) {
    if (!p)
        p = (char *)name;   /* p == q == NULL */
    }
    else if (!p || q > p)   /* p == NULL or q > p */
    p = q;

    rc = -1;            /* init to error value */

    if (ext = strchr(p, '.')) {
    /* extension given; only do filename */
    if (_access(name, 0) != -1)
        /* we compare the extension against .BAT and send comexebat()
           a zero if we are working on a .BAT file */
        rc = comexebat(mode, name, argv, envp, _stricmp(ext, exstrings[0]));
    }

    else    {
    /* no extension; try .bat, then .com and .exe */
    tempamblksiz = _amblksiz;
    _amblksiz = 0x10;       /* reduce _amblksiz so allocation */
                    /* is more efficient */
    p = malloc(strlen(name) + 5);
    _amblksiz = tempamblksiz;   /* restore _amblksiz */
    if (!p)
        return(-1);

    strcpy(p, name);
    ext = p + strlen(name);

    for (i = 2; i >= 0; --i) {
        strcpy(ext, exstrings[i]);
        if (_access(p, 0) != -1) {
        /* when i == 0, we know we have a .BAT file.  comexebat()
           needs this information */
        rc = comexebat(mode, p, argv, envp, i);
        break;
        }
    }
    free(p);
    }

    return rc;
}
