/***
*execvpe.c - execute a file with given environ; search along PATH
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines execvpe() - execute a file with given environ
*
*******************************************************************************/

#include <register.h>
#include <errno.h>
#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <mbstring.h>

#define SLASH "\\"
#define SLASHCHAR '\\'
#define XSLASHCHAR '/'
#define DELIMITER ";"

#ifdef _MBCS
/* note, the macro below assumes p is to pointer to a single-byte character
 * or the 1st byte of a double-byte character, in a string.
 */
#define ISPSLASH(p) ( ((p) == _mbschr((p), SLASHCHAR)) || ((p) == \
_mbschr((p), XSLASHCHAR)) )
#else
#define ISSLASH(c)  ( ((c) == SLASHCHAR) || ((c) == XSLASHCHAR) )
#endif


/***
*int _execvpe(filename, argvector, envvector) - execute a file
*
*Purpose:
*   Executes a file with given arguments and environment.
*   try to execute the file. start with the name itself (directory '.'), and
*   if that doesn't work start prepending pathnames from the environment until
*   one works or we run out. if the file is a pathname, don't go to the
*   environment to get alternate paths. If a needed text file is busy, wait
*   a little while and try again before despairing completely
*
*Entry:
*   char *filename - file to execute
*   char **argvector - vector of arguments
*   char **envvector - vector of environment variables
*
*Exit:
*   destroys the calling process (hopefully)
*   if fails, returns -1
*
*Exceptions:
*
*******************************************************************************/

int  _execvpe (filename, argvector, envptr)
REG3 const char *filename;
const char * const *argvector;
const char * const *envptr;
{
    REG1 char *env;
    char *bbuf = NULL;
    REG2 char *buf;
    char *pfin;

    assert(filename != NULL);
    assert(*filename != '\0');
    assert(argvector != NULL);
    assert(*argvector != NULL);
    assert(**argvector != '\0');

    _execve(filename,argvector,envptr);

    /* In a SBCS build, _mbschr will map to strchr. */

    if ( (errno != ENOENT)
    || (_mbschr(filename, SLASHCHAR) != NULL)
    || (_mbschr(filename, XSLASHCHAR) != NULL)
    || *filename && *(filename+1) == ':'
    || !(env=getenv("PATH"))
    )
    goto reterror;

    /* allocate a buffer to hold alternate pathnames for the executable
     */
    if ( (buf = bbuf = (char *)malloc(_MAX_PATH)) == NULL ) goto reterror;

    do {
    /* copy a component into bbuf[], taking care not to overflow it
     */
    while ( (*env) && (*env != ';') && (buf < bbuf+_MAX_PATH-2) )
        *buf++ = *env++;

    *buf = '\0';
    pfin = --buf;
    buf = bbuf;

#ifdef _MBCS
    if (*pfin == SLASHCHAR) {
        if (pfin != _mbsrchr(buf,SLASHCHAR))
        /* *pfin is the second byte of a double-byte char */
        strcat( buf, SLASH );
    }
    else if (*pfin != XSLASHCHAR)
        strcat(buf, SLASH);
#else
    if (*pfin != SLASHCHAR && *pfin != XSLASHCHAR)
        strcat(buf, SLASH);
#endif

    /* check that the final path will be of legal size. if so, build
     * it. otherwise, return to the caller (return value and errno
     * rename set from initial call to execve()).
     */
    if ( (strlen(buf) + strlen(filename)) < _MAX_PATH )
        strcat(buf, filename);
    else
        break;

    _execve(buf, argvector, envptr);

    if ( (errno != ENOENT)
#ifdef _MBCS
        && (!ISPSLASH(buf) || !ISPSLASH(buf+1)) )
#else
        && (!ISSLASH(*buf) || !ISSLASH(*(buf+1))) )
#endif
        break;
    } while ( *env && env++ );

reterror:
    if (bbuf != NULL) free(bbuf);
    return(-1);
}
