/***
*spawnvpe.c - spawn a child process with given environ (search PATH)
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines spawnvpe() - spawn a child process with given environ (search PATH)
*
*******************************************************************************/

#include <errno.h>
#include <assert.h>
#include <register.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <internal.h>
#include <process.h>
#include <mbstring.h>

#define SLASH "\\"
#define SLASHCHAR '\\'
#define XSLASHCHAR '/'
#define DELIMITER ";"

#ifdef _MBCS
/* note, the macro below assumes p is to pointer to a single-byte character
 * or the 1st byte of a double-byte character, in a string.
 */
#define ISPSLASH(p) ( ((p) == _mbschr((p), SLASHCHAR)) || ((p) == \
_mbschr((p), XSLASHCHAR)) )
#else
#define ISSLASH(c)  ( ((c) == SLASHCHAR) || ((c) == XSLASHCHAR) )
#endif

/***
*_spawnvpe(modeflag, filename, argv, envptr) - spawn a child process
*
*Purpose:
*   Spawns a child process with the given arguments and environ,
*   searches along PATH for given file until found.
*   Formats the parameters and calls spawnve to do the actual work.  The NULL
*   environment pointer indicates that the new process will inherit the
*   parents process's environment.  NOTE - at least one argument must be
*   present.  This argument is always, by convention, the name of the file
*   being spawned.
*
*Entry:
*   int modeflag - defines mode of spawn (WAIT, NOWAIT, or OVERLAY)
*           only WAIT and OVERLAY supported
*   char *filename - name of file to execute
*   char **argv - vector of parameters
*   char **envptr - vector of environment variables
*
*Exit:
*   returns exit code of spawned process
*   if fails, returns -1
*
*Exceptions:
*
*******************************************************************************/

int  _spawnvpe(modeflag, filename, argv, envptr)
int modeflag;
const REG3 char *filename;
const char * const *argv;
const char * const *envptr;
{
    int i;
    REG1 char *env;
    REG2 char *buf = NULL;
    char *pfin;
    int tempamblksiz;      /* old _amblksiz */
    assert(filename != NULL);
    assert(*filename != '\0');
    assert(argv != NULL);
    assert(*argv != NULL);
    assert(**argv != '\0');

    tempamblksiz = _amblksiz;
    _amblksiz = 0x10;       /* reduce _amblksiz for efficient mallocs */

    if (
    (i = _spawnve(modeflag, filename, argv, envptr)) != -1
        /* everything worked just fine; return i */

    || (errno != ENOENT)
        /* couldn't spawn the process, return failure */

    /* In a SBCS build, _mbschr will map to strchr. */

    || (_mbschr(filename, XSLASHCHAR) != NULL)
        /* filename contains a '/', return failure */

    || (_mbschr(filename,SLASHCHAR) != NULL)
        /* filename contains a '\', return failure */

    || *filename && *(filename+1) == ':'
        /* drive specification, return failure */

    || !(env = getenv("PATH"))
        /* no PATH environment string name, return failure */

    || ( (buf = (char *)malloc(_MAX_PATH)) == NULL )
        /* cannot allocate buffer to build alternate pathnames, return
         * failure */
    ) {
        _amblksiz = tempamblksiz;   /* restore old _amblksiz */
        goto done;
    }

    _amblksiz = tempamblksiz;       /* restore old _amblksiz */


    /* could not find the file as specified, search PATH. try each
     * component of the PATH until we get either no error return, or the
     * error is not ENOENT and the component is not a UNC name, or we run
     * out of components to try.
     */

    while ( (env = _getpath(env, buf, _MAX_PATH - 1)) && (*buf) ) {

        pfin = buf + strlen(buf) - 1;

        /* if necessary, append a '/'
         */
#ifdef _MBCS
        if (*pfin == SLASHCHAR) {
            if (pfin != _mbsrchr(buf,SLASHCHAR))
            /* fin is the second byte of a double-byte char */
                strcat(buf, SLASH );
        }
        else if (*pfin !=XSLASHCHAR)
            strcat(buf, SLASH);
#else
        if (*pfin != SLASHCHAR && *pfin != XSLASHCHAR)
            strcat(buf, SLASH);
#endif
        /* check that the final path will be of legal size. if so,
         * build it. otherwise, return to the caller (return value
         * and errno rename set from initial call to spawnve()).
         */
        if ( (strlen(buf) + strlen(filename)) < _MAX_PATH )
            strcat(buf, filename);
        else
            break;

        /* try spawning it. if successful, or if errno comes back with a
         * value other than ENOENT and the pathname is not a UNC name,
         * return to the caller.
         */
        if ( (i = _spawnve(modeflag, buf, argv, envptr)) != -1
            || ((errno != ENOENT)
#ifdef _MBCS
                && (!ISPSLASH(buf) || !ISPSLASH(buf+1))) )
#else
                && (!ISSLASH(*buf) || !ISSLASH(*(buf+1)))) )
#endif
            break;

    }

done:
    if (buf != NULL) free(buf);
    return(i);
}
