/***
*lcfile.h - Locale.sys file format
*
*   Copyright (c) 1990-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   Defines the format of the locale data file (i.e., data
*   as it resides on disk (locale.sys)).
**
*******************************************************************************/

#include <locale.h>

/*
 * Constants
 */

/* max. length of std. language/country abbreviations (incl. null) */
#define _SHORT_LANG_LEN 4
#define _SHORT_CTRY_LEN 4

#define _CDAYS    7
#define _CMONTHS 12

#ifndef _WCHAR_T_DEFINED
typedef unsigned short wchar_t;
#define _WCHAR_T_DEFINED
#endif

typedef union
{
    unsigned char ch;
    wchar_t   wch;
} utext;

/***
 * Locale File Definition
 ***
 *
 * The locale data file has 7 main portions:
 *
 *  (1) A header that contains general information
 *  (2) Lists of available Languages, Countries, and Codepages
 *  (3) Lists matching Language & Country codes to English text names
 *  (4) A list of Languages with associated info headers
 *  (5) A list of Countries with associated info headers
 *  (6) A list of Codepages with associated info headers
 *  (7) A block of data (strings, tables, etc.) referenced from headers
 *
 * Note: all "pointers" are 32-bit offsets from the beginning of (7).
 * This simplifies modification of an existing locale file.
 *
 * We are considering other variations, notably 16-bit indices like those
 * used in system resources.  This would require a table matching the indices
 * to actual offsets, although this level may be hidden.
 *
 *
 *      +-----------------------+
 *      |           |
 *      |     File Header   | (56+ bytes)
 *      |           |
 *      +-----------------------+
 *
 *
 *      +-----------------------+
 *      |           |
 *      |    Language Table | (1 entry per Language)
 *      |           |
 *      +-----------------------+
 *
 *      +-----------------------+
 *      |           |
 *      |     Country Table | (1 entry per Country)
 *      |           |
 *      +-----------------------+
 *
 *      +-----------------------+
 *      |           |
 *      |    Codepage Table | (1 entry per Codepage)
 *      |           |
 *      +-----------------------+
 *
 *
 *      +-----------------------+
 *      |           |
 *      |  Language Dictionary  | (1 entry per name, variant, or abbrev)
 *      |           |
 *      +-----------------------+
 *
 *      +-----------------------+
 *      |           |
 *      |   Country Dictionary  | (1 entry per name, variant, or abbrev)
 *      |           |
 *      +-----------------------+
 *
 *
 *      +-----------------------+
 *      |   Language #1 Info    | (34+ bytes) each
 *      +-----------------------+
 *      |   Language #2 Info    |
 *      +-----------------------+
 *              .
 *              .
 *              .
 *      +-----------------------+
 *      |   Language #N Info    |
 *      +-----------------------+
 *
 *
 *      +-----------------------+
 *      |   Country #1 Info | (34+ bytes each)
 *      +-----------------------+
 *      |   Country #2 Info |
 *      +-----------------------+
 *              .
 *              .
 *              .
 *      +-----------------------+
 *      |   Country #N Info |
 *      +-----------------------+
 *
 *
 *      +-----------------------+
 *      |   Codepage #1 Info    | (32+ bytes each)
 *      +-----------------------+
 *      |   Codepage #2 Info    |
 *      +-----------------------+
 *              .
 *              .
 *              .
 *      +-----------------------+
 *      |   Codepage #N Info    |
 *      +-----------------------+
 *
 *      +-----------------------+
 *      |   Misc. data block    | (variable size)
 *      |           |
 *      .           .
 *      .           .
 *      .           .
 *      |           |
 *      +-----------------------+
 */

/*
 * File Header
 * This structure defines the info in the file header.
 */

struct _lc_fileheader {
    short hdr_size;         /* size of fix-length header (bytes) */
    unsigned short cLangs;      /* number of languages in file */
    unsigned short cCtrys;      /* number of countries in file */
    unsigned short cCodePages;  /* number of codepages in file */
    struct _Langindex *LangIndex;   /* sorted list of languages avail. */
    struct _CtryIndex *CtryIndex;   /* sorted list of countries avail. */
    struct _CPIndex *CPIndex;   /* sorted list of codepages avail. */
    unsigned short cLangDict;   /* size of language dictionary */
    unsigned short cCtryDict;   /* size of country dictionary */
    struct _LangName *LangDict; /* Language-name dictionary (sorted) */
    struct _CtryName *CtryDict; /* Country-name dictionary (sorted) */
/*  void *CPDict;           // not needed?? */
/* CONSIDER--the remainder could be a separate internal structure... */
    struct _language *lpLang;   /* offset to language defs */
    struct _country *lpCtry;    /* offset to country defs */
    struct _codepage *lpCodepage;   /* offset to codepage defs */
    unsigned long revision;     /* rev of file header */
    long cDataArea;         /* size of data area (bytes) */
    char *lpDataArea;       /* misc data area; */
};

struct _Langindex {
    unsigned long code;
    struct _language *index;
    };

struct _Ctryindex {
    unsigned short code;
    struct _country *index;
    };

struct _CPIndex {
    unsigned short code;
    struct _codepage *index;
    };

struct _LangNames {
    unsigned char *name;
    unsigned long code;
    };

struct _CtryNames {
    unsigned char *name;
    unsigned short code;
    };

/*
 * Structures used within language, country, and codepage entries.
 */

/* valid types for 'format' field of _charmap structure */
enum    _CM_FORMAT
    {
    _CM_CHARMAP = (-1), /* recursive: array of charmap */
    _CM_DIRECT  =  0,   /* direct 1:1 mapping--no data */
    _CM_BYTES,      /* array of abs. bytes */
    _CM_WORDS,      /* array of abs. words */
    _CM_CONST,      /* maps directly to single data value */
    _CM_RELBYTES,       /* array of relative byte offsets */
    _CM_RELWORDS,       /* array of relative word offsets */
    _CM_RELCONST,       /* single relative data offset */
    _CM_BYTPAIRS,       /* array of pairs of bytes */
    _CM_WRDPAIRS,       /* array of pairs of words */
    _CM_STRPAIRS,       /* array of pairs of char strings */
    _CM_WCSPAIRS,       /* array of pairs of wchar_t strings */
    _CM_BITMAP      /* special n:n:n packed bit-map scheme */
                /* currently optimized for 8:4:4 */
    };

struct _charmap {       /* must be 32-bit aligned */
    short cp;       /* code_page of table (0=unicode ???)  */
    short format;       /* enum types listed above */
    unsigned short start;   /* offset (if any) to first char in range */
    unsigned short len; /* array size of union - 1 */
    union {
      short data;       /* used for const-type data */
      void * pdata;     /* used for all other data */
    };
};

struct _day_month {
    unsigned short hdr_size; /* size of fix-length portion (bytes) */
    short cp;       /* code_page of table (0=unicode ???) */
    short format;       /* 0/1 = standard info char/wchar_t;2+ reserved */
    unsigned short date_fmt; /* defaults; country will override */
    unsigned short time_fmt; /* defaults; country will override */
    short aux_type;     /* aux. info structure type; 0 = none */
    void * aux_info;    /* auxilary information structure pointer */
    utext **LongDateFmt;    /* word-for-windows format date */
    utext *wday[_CDAYS];
    utext *wday_abbr[_CDAYS];
    utext *month[_CMONTHS];
    utext *month_abbr[_CMONTHS];
    utext *date_sep;
    utext *time_sep;
    utext *ampm[2];
    utext *bcad[2];
};


/*
 * Language
 */

struct _language {
    short hdr_size;     /* size of fix-length portion (bytes) */
    short version;      /* 0 = use OS if available (default) */
    unsigned short lang;    /* standard language number */
    unsigned short sublang; /* language variant, (0, 1, 2...) */
    unsigned char shortname[_SHORT_LANG_LEN]; /* std. abbrev. lang name */
    unsigned char * name;   /* Language name (in english) */
    wchar_t * uniname;  /* optional Language name (in unicode) */
    short def_ctry;     /* default country number */
    short def_cp;       /* default code page */
    struct _charmap **colltab; /* list of collation table structures */
    struct _day_month **dmtab;
    short ctables;      /* number of collation tables (typ. 0 or 1) */
};

/*
 * Country
 */

struct _country {
    short hdr_size;     /* size of fix-length portion (bytes) */
    short version;      /* 0 = use OS if available (default) */
    unsigned char shortname[_SHORT_CTRY_LEN];  /* std. abbrev. country name */
    unsigned char * name;   /* Country name (in english) */
    wchar_t *uniname;   /* optional Country name (in unicode) */
    short def_cp;       /* optional default cp; 0 = use lang's def_cp */
    short nm_cp;        /* native code page of lconv data */
    unsigned short date_fmt; /* if defined, overrides language default */
    unsigned short time_fmt; /* if defined, overrides language default */
    unsigned char *LongDateFmt; /* word-for-windows format date */
/* consider splitting lconv components into CP-dependent and independent groups */
/* ANSI numeric/monetary format info (lconv struct) listed explicitly: */

    utext *decimal_point;
    utext *thousands_sep;
    utext *grouping;
    utext *int_curr_symbol;
    utext *currency_symbol;
    utext *mon_decimal_point;
    utext *mon_thousands_sep;
    utext *mon_grouping;
    utext *positive_sign;
    utext *negative_sign;
    char int_frac_digits;
    char frac_digits;
    char p_cs_precedes;
    char p_sep_by_space;
    char n_cs_precedes;
    char n_sep_by_space;
    char p_sign_posn;
    char n_sign_posn;

    utext * list_sep;
    short iMeasure;     /* Measurement system: 0 = Metric; 1 = English */
//  <more to be added?> /* other misc ctry-dependent data */
};

/*
 * Codepage
 */

struct _codepage {
    short hdr_size;     /* size of fix-length portion (bytes) */
    short version;      /* 0 = use OS if available (default) */
    short cp;       /* 850, 437, etc.  (0 = unicode ??) */
    char size;      /* max. wordlength in CP, (1,2, etc.) */
    char format;        /* 0 = unicode; 1 = native ;2+ reserved */
    union {
      struct {
      struct _charmap *cptouc; /* cp-to-unicode conversion table */
      struct _charmap *uctocp; /* optional unicode-to-cp conversion table */
                /* NULL ptr means build reverse from cptouc */
      };
      struct {
        struct _charmap *colltab; /* def. collation table if no unicode */
        struct _charmap *ctype;   /* NULL ptr means convert from unicode */
      };
    };
    struct _charmap *toupper; /* casemap table */
    struct _charmap *tolower; /* optional--use reverse of toupper if NULL. */
    unsigned char *MBCSEv;  /* lead-byte table for MBCS support */
    void * AuxCPInfo;   /* misc. info (room for future expansion) */
};
