/***
*nlsapi.h - NLS Retrieval API
*
*   Copyright (c) 1990-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   Defines a variety of structures needed for nls api locale support.
*   Describes how the data looks in memory  (i.e., after it's
*       been read in from locale.sys or retrieved from the OS).
*   [Internal]
*
*******************************************************************************/

#include <locale.h>
#include <lcfile.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Constants
 */

/* Categories/Types */


enum _NLS_type
{
    _NLS_none = 0,
    _NLS_langdata,
    _NLS_ctrydata,
    _NLS_CPdata,
    _NLS_lconvfmt,
    _NLS_datetimefmt,
    _NLS_datetimestr,

    // Charmap types
    _NLS_uctomb,
    _NLS_mbtouc,
    _NLS_toupper,
    _NLS_tolower,
    _NLS_ctype,
    _NLS_collate,       /* case-sensitive */

/* the following are not finalized: */
    _NLS_icollate,      /* case-insensitive */
    _NLS_ncollate,      /* strip accents */
    _NLS_scollate       /* symbols ???? */
};


/* data types for _NLS_GetLocale */

enum _LC_Type
{
_LCID,
_LCSTR
};


/* these are arbitrary for now--further research needed!  */
#define _MAX_LANG_LEN       32
#define _MAX_CTRY_LEN       32
#define _MAX_CP_LEN     8

typedef unsigned short RID;
#define RESOURCE_STUFF  1

typedef union
{
    unsigned char   ch;
    wchar_t     wch;
} uchar;

#define SELF_BASED  __based((__segment)__self)
#define ustring uchar SELF_BASED *

/***
 * Data Definitions
 ***/

struct _LC_ID {
    unsigned long   lang;
    unsigned short  ctry;
    unsigned short  cp;
    };

struct _LC_Strings {            /* make sure 32-bit aligned */
    char lang[_MAX_LANG_LEN];   /* 32?  40? */
    char ctry[_MAX_CTRY_LEN];   /* 32?  40? */
    char cp[_MAX_CP_LEN];       /* 6 ? more? */  /* short instead ?? */
    };

struct _FmtHdr {
    unsigned char fixedlen; /* serves as a version number */
    unsigned char /* enum _NLS_type */ type;
    unsigned short cp;  /* codepage of the data */
    };

struct _NLS_langdataRec {
    struct _FmtHdr hdr;
// UNDONE: bad alignment!
    short   data_version;   /* version of data not format */

    long lang;
    char shortname[_SHORT_LANG_LEN]; /* ASCII NAME (actually a key) */
    char SELF_BASED * name;     /* full English name (ASCII) */
    ustring  native_name;       /* optional native language name */

    unsigned short def_ctry;
    unsigned short def_cp;

#ifdef RESOURCE_STUFF
// UNDONE: how do we tell the number of tables???
    RID SELF_BASED *  colltabs; // _NLS_collate
    RID SELF_BASED *  datetimefmt;
    RID SELF_BASED *  datetimestr;
#endif

    // MUST BE LESS THAN 64 BYTES
    };


struct _NLS_ctrydataRec {
    struct _FmtHdr hdr;
    char shortname[_SHORT_CTRY_LEN]; /* ASCII NAME (actually a key) */
    char SELF_BASED * name; /* full English name (ASCII) */
    ustring native_name;    /* optional native language name */
    ustring list_sep;
    unsigned short def_cp;  /* if no codepage specified */
    short   data_rev;   /* version of data not format */

#ifdef RESOURCE_STUFF
    RID     lconvfmtRec;
#endif
    // MUST BE LESS THAN 64 BYTES
    };


struct _NLS_CPdataRec {
    struct _FmtHdr hdr;
    short   data_rev;   /* version of data not format */
    unsigned char max_char_size;    /* 1 or 2 */
    unsigned char format;   /* reserved - tbd - (alignment) */
    unsigned char SELF_BASED * leadbyte;    /* ends in '\0\0' */

#ifdef RESOURCE_STUFF
    RID     toupper;        // _NLS_toupper
    RID     tolower;        // _NLS_tolower
    RID     ctype;          // _NLS_ctype
    RID     colltab;        // _NLS_collate
                    // default collate (if no unicode)
    // conversion to/from unicode
    RID     uctomb;         // _NLS_uctomb
    RID     mbtouc;         // _NLS_mbtouc
#endif
    // MUST BE LESS THAN 64 BYTES
    };


struct _NLS_lconvfmtRec {
    struct _FmtHdr hdr;

    // literal copy of 'lconv' with 'char*' replaced by 'ustring'
    ustring decimal_point;
    ustring thousands_sep;
    ustring grouping;
    ustring int_curr_symbol;
    ustring currency_symbol;
    ustring mon_decimal_point;
    ustring mon_thousands_sep;
    ustring mon_grouping;
    ustring positive_sign;
    ustring negative_sign;
    char int_frac_digits;
    char frac_digits;
    char p_cs_precedes;
    char p_sep_by_space;
    char n_cs_precedes;
    char n_sep_by_space;
    char p_sign_posn;
    char n_sign_posn;
    };

struct _NLS_datetimefmtRec {
    struct _FmtHdr hdr;
    unsigned short date_fmt;
    unsigned short time_fmt;
    ustring date_sep;
    ustring time_sep;
    ustring ampm[2];
    };

struct _NLS_datetimestrRec {
    struct _FmtHdr hdr;

    // based arrays of based strings - ending in NULL
    ustring SELF_BASED * LongDateFmt;
    ustring SELF_BASED * wday;
    ustring SELF_BASED * wday_abbr;
    ustring SELF_BASED * month;
    ustring SELF_BASED * month_abbr;
    ustring bcad[2];
    };

/* Same record used for collate, ctype, toupper, tolower, uctomb and mbtouc */
struct _NLS_charmapRec {
    struct _FmtHdr hdr;
    struct _charmap *map;

    unsigned long total_size;   // for all resources contained in this
                //   charmap.

    short format;       // 1=bytes; 2=words, 3/4=packed byte/words
                // 0 = 1-1 mapping, no data
                // 5 = array of RID to _NLS_charmapRec
    unsigned short start;   // offset (if any) to first char in range
    unsigned short len;     // array size of data - 1
// CONSIDER:  is alignment necessary???  Is pointer far/huge?
    void SELF_BASED * data;
};


/* NLS API Prototypes */
// just cache the lang/country/codepage locally (<256 bytes)


/***
*_NLS_GetInfo() - get NLS info from locale.sys or OS
*
*Purpose: Get various NLS information from either locale.sys or the OS.
*
*Entry:
*   struct _LC_ID locale;   fully qualified locale id record (lang/ctry/cp)
*   enum _NLS_type type;    request type (defines 'data' record type also)
*   void huge * data;   ptr to struct to be filled, according to 'type'
*   long size;      total size of 'data' buffer, in bytes.
*
*   Note: data buffer must not cross a segment boundary unless 'size' is
*   >= 64K.  This lets us use _huge* arithmetic only for very large objects.
*
*Exit:
*   data filled with requested NLS info, if possible
*   returns size of buffer used/required.  If return value > size then
*   indicates size required.  At most 'size' bytes are actually written.
*   Special case is if size = 0, or if data* is a NULL ptr, just returns
*   size required.  It is recommend that a separate call be made to
*   _NLS_GetInfoSize (or to this routine) to determine the buffer size
*   required before actually trying to get the data.  Sizes may vary
*   dramatically for different locales and different info request types.
*
*Exceptions:
*   returns <0 if error
*   UNDONE: define particular error codes
*
*******************************************************************************/

long _NLS_GetInfo(struct _LC_ID locale, enum _NLS_type type,
             void huge * data, long size);

/***
*_NLS_GetInfoSize() - get memory required for NLS info from locale.sys or OS
*
*Purpose: Get memory requirements for subsequent _NLS_GetInfo call.
*
*Entry:
*   struct _LC_ID locale;   fully qualified locale id record (lang/ctry/cp)
*   enum _NLS_type type;    request type
*
*Exit:
*   returns size (in bytes) of buffer required by _NLS_GetInfo to return
*   requested information.  Memory must be contiguous and must not cross
*   a segment boundary unless size >=64K.  (This restriction allows use to
*   use huge* arithmetic only for very large objects.)
*
*Exceptions:
*   returns <0 if error
*   UNDONE: define particular error codes
*
*******************************************************************************/

long _NLS_GetInfoSize(struct _LC_ID locale, enum _NLS_type type);
    // zero or negative values are errors - details TBD

/***
* _NLS_GetLocale() - get full locale id/name
*
*Purpose: Given a full or partial locale string or id, return fully qualified
*   locale names and/or id, expanding as necessary.
*
*Entry:
*   type =  _LCID   ??? 0: input is struct _LC_ID *
*        =  _LCSTR  ??? 1: input is struct _LC_Strings *
*   input = (struct _LC_ID *) {valid _LC_ID if type _LCID}
*   input = (struct _LC_Strings *) {valid _LC_Strings if type _LCSTR }
*           (where valid means at least 1st field defined)
*   outid = {*_LC_ID buffer, or NULL ptr for no id output}
*   outstr = {*_LC_Strings buffer, or NULL ptr for no str output}
*
*Exit:
*   if either outid or outstr are not NULL, they are filled in with
*   fully qualified locale codes/strings, if possible.
*   returns(>=0 indicates success; <0 indicates failure
*   CONSIDER (any more specific useful returns, off hand?)
*
*Exceptions:
*   input not valid or not recognized in system/locale.sys
*   UNDONE: define particular error codes
*
*******************************************************************************/
int _NLS_GetLocale(unsigned short type, const void *input, struct _LC_ID *outid, struct _LC_Strings *outstr);


#ifdef __cplusplus
}
#endif
