	page	,132
	title	addseg - Add memory to a heap segment
;***
;addseg.asm - Add memory to a heap segment
;
;	Copyright (c) 1988-1992, Microsoft Corporation. All rights reserved.
;
;Purpose:
;
;*******************************************************************************

include	version.inc
.xlist
include	cmacros.inc
include heap.inc
.list

sBegin	code
	assumes cs,code
	assumes ds,data

page
;***
;addseg - Add memory to a heap segment
;
;Purpose:
;	Add supplied memory to the designated heap segment.
;
;Entry:
;	ds:bx = heap segment descriptor
;	si = offset to memory to be added
;	cx = size of memory block to be added
;
;	NOTES:
;	(1) Pointer si is assumed to be even
;	(2) Size cx is assumed to be even
;
;Exit:
;	carry clear = success
;	carry set   = failure
;
;Uses:
;	ax, cx, dx, si, di
;Preserves:
;	ds:bx, es
;
;Exceptions:
;
;*******************************************************************************

cProc	_addseg,<PUBLIC,NEAR>,<>

cBegin	<nogen>

;
; *** NOTES:
;(1) CLEAR FREE BIT IN HEAP SEG DESCRIPTOR ??
;

;
; Make sure size is big enough, but not too big
;
	cmp	cx,4		; need at least 4 bytes
	jb	error_rtn	; error
	dec	cx
	add	si,cx		; last byte of block must be within segment
	jc	error_rtn
	sub	si,cx		; restore si
	inc	cx		; restore cx

;
; See where the new segment lies with regard to the existing heap
;
	cld			; store forward
	cmp	si,[bx].start	; block < start ??
	jb	below		; yes, new block is below start
	cmp	si,[bx].last	; block > last ??
	ja	above		; yes, new block is above last
	;fall thru		; no, start < block < last

;
; --- Case 1: start of heap < new block < end of heap
;

within:

; Walk the heap and see where the new block fits in.

	mov	di,si		; di = address of new block
	mov	si,[bx].start	; si = start of heap

next_block:
	mov	dx,si		; dx = last header inspected
	lodsw			; get header
	; *** OPTIONAL VALIDITY CHECK
	cmp	ax,_HEAP_END	; end of heap ??
	je	error_rtn	; *** SHOULD BE "JE _HEAP_TOAST"
	; *** END VALIDITY CHECK
	and	al,not 1	; mask off free/inuse bit
	add	si,ax		; bump pointer to next block
	cmp	si,di		; current pointer < new block ??
	jb	next_block	; yes, onto next block
	; *** OPTIONAL VALIDITY CHECK ***
	je	error_rtn	; error, user gave us a pointer we already own
	; *** END VALIDITY CHECK
	; make sure the allocation block is NOT free and that the new block
	; lies entirely within it.
	xchg	bx,dx		; temporarily swap bx and dx
	test	word ptr [bx],1 ; check the free/inuse bit
	xchg	bx,dx		; swap bx and dx back
	jnz	error_rtn	; error! allocation block is free
	add	di,cx		; di = address of 1st byte after new block
	cmp	si,di		; is there any overlap?
	jb	error_rtn	; error! new block overlaps next block.
	sub	di,cx
	; fall thru, dx < new block < si

; Found the slot where the new block fits
; ax = size of <dx> block
; cx = size of new block
; dx = heap block containing the new block
; si = heap block after the new one
; di = new block

	xchg	si,di		; si = new block / di = next block
	call	_before 	; link new block in with subsequent block
				; (preserves dx, si)

	mov	di,dx		; di = header of block containing new block
	; *** OPTIONAL VALIDITY CHECK
	test	byte ptr [di],1 ; this block better be "in use"
	jnz	error_rtn	; error - block is free
	; *** END CHECK

	call	_after		; link new block in with previous block

	or	ax,ax		; si still point to the new block ??
	jnz	update_rover	; yes, update rover
	mov	si,di		; si = new block
update_rover:
	cmp	si,[bx].rover	; si > rover ??
	jae	good_rtn	; nope, all done
	mov	[bx].rover, si	; yes, update it
	jmp	short good_rtn	; good return

;
; --- Case 2: new block < start of heap
; si = new block address
; cx = size of new block
;

below:
	; make sure the supplied block doesn't overlap the heap segment
	; descriptor or the first allocation block.
	mov	dx,si
	add	dx,cx		; dx = address of 1st byte after block
	cmp	dx,bx		; is the new block below the descriptor
	jbe	desc_ok 	; yes
	mov	di,bx
	add	di,size _heap_seg_desc
				; di = address of 1st byte after descriptor
	cmp	si,di
	jb	error_rtn	; error! new block overlaps descriptor
	; fall thru, the new block is above the descriptor
desc_ok:
	mov	di,[bx].start	; di = old starting address
	cmp	dx,di		; does the new block overlap?
	ja	error_rtn	; yes, error!
	; fall thru, new block doesn't overlap
	call	_before 	; hook the new block in before the heap

	mov	[bx].start,si	; update starting address and rover
	mov	[bx].rover,si

	jmp	short good_rtn	; good return

;
; --- Error return
;

error_rtn:
	stc			; carry set = error
	jmp	short done

;
; --- Case 3: end of heap < new block
; si = new block address
; cx = size of new block
;

above:
	;Fill in new block as end of heap
	dec	cx
	dec	cx		; make room for _HEAP_END flag
	mov	di,si		; di = si = new block address
	add	di,cx		; di = end of new block
	mov	[di],_HEAP_END	; new end-of-heap flag
	push	di		; save ptr to end-of-heap flag for later
	dec	cx		; make room for header (free block)
	mov	[si],cx 	; store new header

	; link the new block to the end of the heap
	mov	di,[bx].last	; di = current end of heap
	call	_after		; return value in ax

	; update segment descriptor
	pop	ax		; ax = address of end-of-heap
	mov	[bx].last,ax	; update the last entry pointer
	add	ax,2		; ax = new size of heap seg
	mov	[bx].segsize,ax ; store it
	;fall thru		; good return

;
; --- Good return
;

good_rtn:
	clc			; carry clear = good return
	;fall thru

;
; Common return
;

done:
	ret

cEnd	<nogen>


page
;***
; _before - Place a new block before a given heap entry
;
;Purpose:
;	Place a user supplied block before a given heap entry.
;
;Entry:
;	si = pointer to new block
;	di = pointer to next heap entry
;	cx = size of new block
;
;	NOTES:
;	(1) It is the caller's responsibility to ensure that si < di
;	(2) ALSO SEE _AFTER NOTES
;
;Exit:
;	<none>
;
;Uses:
;	cx
;Preserves:
;	ax, bx, dx, si
;Exceptions:
;
;*******************************************************************************

cProc	_before,<LOCAL,NEAR>,<>

cBegin	<nogen>

; See if the two blocks are adjacent or not

	sub	di,si		; di = heap entry - new block
	sub	di,cx		; di = heap entry - end of new block
	jz	joined_before	; if blocks are adjacent, jump down

	;new block is not connected to heap, put intermediate block in place
	dec	cx
	dec	cx		; make room for connecting header
	add	si,cx		; point to after new block
	mov	[si],di 	; hook intermediate block to heap (in use)
	sub	si,cx		; get original header back

joined_before:
	dec	cx		; new block size marked as free
	mov	[si],cx 	; hook new block to heap (free)

	ret

cEnd	<nogen>


page
;***
; _after - Place a new block after a given heap entry
;
;Purpose:
;	Place a user supplied block after a given heap entry.
;
;Entry:
;	si = pointer to new block
;	di = pointer to header of existing entry in which new
;	     block will reside (may be pointer to _HEAP_END).
;
;	NOTES:
;	(1) It is the caller's responsibility to ensure that si > di
;	(2) It is assumed that the new block <si> is already filled in
;	(i.e., has a header that links it forward to the heap).
;	(3) <di> points to the block into which the new block goes.
;	That is, <di> points to either the old end-of-heap or a block
;	in the middle of the heap that will be partitioned into the new
;	block and possible an intermediate one.
;	(4) If calling both _before and _after on a new block, always
;	call _before first (this relates to #2 above).
;
;Exit:
;	ax = !0 = <si> still points to new block
;	ax =  0 = <di> points to the new block
;		  (i.e., the <si> block was merged with previous header)
;Uses:
;	dx
;Preserves:
;	bx, cx, si, di
;Exceptions:
;
;*******************************************************************************

cProc	_after,<LOCAL,NEAR>,<>

cBegin	<nogen>

; See if the two blocks are adjacent or not

	; assume blocks are adjacent
	mov	dx,[si] 	; header of new block (for later)
	inc	dx
	inc	dx		; add two bytes for header

	mov	ax,di		; ax = di = end of heap entry
	add	ax,2		; ax = block after di
	sub	ax,si		; any intermediate memory ??
	jz	joined_after	; if blocks are adjacent, jump down

	;new block is not connected to heap, put intermediate block in place
	neg	ax		; make ax = length of new block
	mov	dx,ax		; dx = length of new block (in use)
				; ax = non-zero

joined_after:
	mov	[di],dx 	; hook old heap to new block

	ret

cEnd	<nogen>

sEnd	code

	end
