	page	,132
	title	minseg - Shrink or free a heap segment
;***
;minseg.asm - Shrink or free a heap segment
;
;	Copyright (c) 1988-1992, Microsoft Corporation. All rights reserved.
;
;Purpose:
;	Contains the _minseg() routine that tries to shrink or free (if
;	possible) the supplied heap segment.
;
;******************************************************************************

.xlist
include version.inc
include cmacros.inc
include heap.inc
.list

externNP _findlast			; find last block in a heap segment
externNP _incseg			; resize a segment
externNP _unlinkseg			; unlink and free a heap segment

sBegin	code
	assumes cs,code
	assumes ds,nothing

page
;***
;_minseg - Shrink/free a heap segment.
;
;Purpose:
;	Shrink a heap segment by resizing the segment to return free memory
;	at the end of the heap segment, if any, back to the OS. If the heap
;	segment is freeable and consists of a single free allocation block,
;	_unlinkseg is called to unlink the segment and free (i.e., return it
;	to the OS).
;
;Entry:
;	ds:bx = pointer to heap segment descriptor
;	es:di = pointer to heap list descriptor (used only if the segment can
;		unlinked and freed)
;
;Exit:
;	No Error:
;		cf = 0 (clear)
;		ax = 1, if the segment was not freed (ds:bx is preserved)
;		   = 0, if the segment is freed (ds:bx = pointer to heap list
;			descriptor)
;	Error:
;		cf = 1 (set)
;		ax = 1, if inconsistency was discovered within the heap
;			segment (ds:bx is preserved)
;		   = 0, if error was returned by _unlinkseg (ds:bx = pointer
;			heap list descriptor)
;
;Uses:
;	ax, bx, cx, dx, si, di, es, ds
;
;Notes:
;	(1) _searchseg() should be called to coalesce the heap segment before
;	    before his function is called.
;	(2) _minseg nominally expects es:di to point to a heap list descriptor
;	    on entry. However, this pointer is never referenced unless the
;	    heap segment can be unlinked and freed (in this case, es:di is
;	    simply handed off to _unlinkseg). Thus, _minseg can safely
;	    operate on the near heap segment even though there is no such
;	    thing as a near heap list discriptor.
;	(3) We have some special near heap code to make sure we free up
;	    unused memory between the end of the near heap and the end of
;	    DGROUP.
;
;******************************************************************************

cProc	_minseg,<PUBLIC,NEAR>,<>

cBegin	<nogen>

; ***** Quick checks for special cases.
;
; Exit Paths:
;	min_done (can't modify heap)
;	min_done (_unlinkseg is called to free the segment completely)
;	fall thru
;		cx = number of bytes to keep beyond header of last block
;		di = pointer to last allocation block or ptr to _HEAP_END

;	Do we have permission to modify the size of this heap segment?

	mov	ax,[bx].flags
	test	al,_HEAP_MODIFY
	jz	min_done		; can't modify, quit

;	Is the last block in the heap segment free?

	push	ax
	call	_findlast		; get the last allocation block
	pop	ax
	mov	dx,[si] 		; dx = header of last block
	test	dl,1			; is it free?
	jz	check_near		;   no, do special near heap check


;
;	Does the heap segment consist of a single (free) allocation block and
;	is it the heap segment freeable?
;

	mov	cx,2			; reserve space for end-of-heap marker
	mov	dx,di			; es:dx = pointer to heap list desc
	mov	di,[bx].start
	xchg	si,di			; (swap si and di for later convenience)
	cmp	si,di			; last = first?
	jne	calc_newsize		;   no, go calculate the new seg size
	inc	cx			; yes, reserve space for one allocation
	inc	cx			; block
	test	al,_HEAP_FREE		; is this heap segment freeable?
	jz	calc_newsize		;   no, go shrink the segment

	mov	di,dx			; es:di = pointer to heap list desc
	call	_unlinkseg		; unlink and free the segment
	xor	ax,ax			; return value of 0
	jmp	short donef		; return to caller (the carry flag and
	;------------------		; ax have appropriate values)

;
;	Last heap block is in use.  If we're working on the near heap, see if
;	there's any space between the end of the heap and the top of DGROUP
;

check_near:
	test	al,_HEAP_NEAR		; near heap ??
	jz	min_done		;   nope, can't minimize

	mov	di,[bx].segsize 	; di = current segsize
	dec	di			; di = segsize - 1
	cmp	di,[bx-2]		; current segsize >= _asizds ??
	jae	min_done		;   yes, can't minimize

	mov	di,[bx].last		; di = ptr to _HEAP_END
	xor	cx,cx			; no extra bytes needed
	;fall thru

; ***** End of quick checks for special cases.

; calc_newsize
; Calculate the proposed new segment size ( N*16, where (N-1)*16 < di + cx
; <= N*16 ).
;
; Entry:
;	cx = number of bytes to keep beyond header of last block
;	di = pointer to last or pointer to _HEAP_END
;
;	[NOTE:	The only time di can point to _HEAP_END is if (a) we are
;	working on the near heap, and (b) the last block in the pre-shrunk
;	heap is "in-use".]
;
; Exit Paths:
;	min_done (can't shrink segment)
;	fall thru to do_incseg
;		ax = proposed new segment size
;		di = pointer to last or pointer to _HEAP_END
;
; Uses:
;	ax, cx

calc_newsize:
	mov	ax,di
	add	cx,0Fh
	add	ax,cx
	jc	min_done		; cannot shrink segment by even one
					; paragraph, quit
	and	al,not 0Fh		; ax = new segment size

;
; Call _incseg to shrink the segment.
;
; Entry:
;	ax = proposed segment size
;	di = pointer to last or pointer to _HEAP_END
;
; Exit Paths:
;	bad_heap (error)
;	fall thru
;		ax = new segment size
;		di = pointer to last or pointer to _HEAP_END
;
; Notes:
;	It is assumed that _incseg preserves di. If this is changed, then
;	di must be saved across the call. Otherwise, minseg will break!
;

do_incseg:
	call	_incseg 		; resize the segment
	jc	bad_heap		; error
	;fall thru			; success

; ***** Update end-of-heap marker and fields of the heap segment descriptor.
;
; Entry:
;	ax = new segment size
;	di = pointer to last or pointer to _HEAP_END
;
; Exit Paths:
;	fall thru to min_done
;		ax = 1
;	bad_heap (error occurred)
;		ax = -1

	mov	[bx].segsize,ax 	; update segsize field of descriptor

;	Set new end-of-heap marker and update .last field.

	sub	ax,2			; deduct 2 for end-of-heap marker
	mov	si,ax			; si points to new end-of-heap marker
	mov	[si],_HEAP_END		; seg new end-of-heap marker
	mov	[bx].last,si		; update .last with pointer to eoh

;	Update the header of the last allocation block if required.

	cmp	si,di			; is last block now eoh marker?
	je	chk_rover		;   yes, no need to update header
	sub	ax,di			; update header
	dec	ax			; ax = size of last block + 1
	mov	[di],ax 		; update header for last

;	Check rover. If it was set to (the original value of) last, reset it
;	to be start.

chk_rover:
	mov	si,[bx].rover
	cmp	si,di			; was rover set to last allocation block?
	jbe	min_done		;   no, we are done
	mov	si,[bx].start
	mov	[bx].rover,si

; ***** End of update of heap descriptor fields

min_done:
	clc				; clear carry to indicate no error
	jmp	short done

bad_heap:
	stc				; set carry flag to indicate error

done:
	mov	ax,1			; non-zero return because segment
					; was not freed
donef:
	ret
	cEnd	<nogen>

sEnd	code
	end
