	page	,132
	title	resize - C runtime heap allocation resize
;***
;resize.asm - resize an allocation block in a heap segment
;
;	Copyright (c) 1988-1992, Microsoft Corporation. All rights reserved.
;
;Purpose:
;	defines _resize() - resize an allocation block
;
;*******************************************************************************

.xlist
include	version.inc
include	cmacros.inc
include heap.inc
.list

externNP _growseg

sBegin	code
assumes	cs,code
assumes ds,nothing

page
;***
;_resize - resize an allocation block in a heap segment
;
;Purpose:
;	This routine resizes an allocation block to the specified new
;	size. If not successful (expansion only), the block is expanded
;	in size as much as possible. The free/inuse status of allocation
;	block is not altered (actually, it is altered, but it is also
;	restored).
;
;	In the case of expansion, _resize first tries to coalesce subsequent
;	free blocks with the specified block (if possible). If the resulting
;	block is not big enough and if it is the last block in the heap
;	segment, then _growseg is called in an attempt to expand the heap
;	segment enough to allow sufficient expansion of the allocation block.
;
;Entry:
;	ds:bx = pointer to heap segment descriptor
;	ds:si = pointer to the heap block to modify + 2
;	cx    = new size
;
;Exit:
;	carry flag clear if successful
;	carry flag set otherwise
;
;Uses:
;	ax, cx, dx, es
;
;Preserves:
;	bx, si, di, ds
;
;Exceptions:
;
;*******************************************************************************

cProc	_resize,<PUBLIC,NEAR>,<si,di>

	localW	FailFlag

cBegin
	cld			; clear direction flag
	mov	[FailFlag],0	; initialize flag (to no failure)
	inc	cx
	jz	max_expand
	and	cl,not 1	; cx = newsize rounded up to nearest even
	jmp	short expand_prep

max_expand:
	dec	cx		; cx = FFFF

expand_prep:
	dec	si
	dec	si
	mov	di,si		; si = di = pointer to heap block
	lodsw			; ax = header of block,
				; si = pointer to heap block + 2
	push	ax		; save header (for free flag value)
	and	al,not 1
	mov	ds:[di],ax	; temporarily mark block inuse
	mov	dx,ax

;
; coal_loop
; Try to expand the allocation block by coalescing in proceeding free
; blocks.
;
; Entry:
;	ax = header of block (marked inuse)
;	dx = ax
;	si = pointer to block + 2
;	di = pointer to block

coal_loop:
	cmp	dx,cx		; big enough?
	jae	expand_succeed	;   yes
	add	si,ax		; si = next block
	jc	heap_toast	; bad heap!
	lodsw			; ax = header of next block,
				; si = pointer to next block + 2
	test	al,1		; free?
	jz	try_grow	;   no, can't coalesce any more
	add	dx,ax		; coalesce...
	inc	dx
	mov	ds:[di],dx
	dec	ax		; ax = size of block just added on
	jmp	short coal_loop

;
; try_grow
; coal_loop has failed to expand the block enough. If the block is the last
; allocation block in this heap segment, try using _growseg to expand the
; segment sufficiently to allow the block to expand. If successful, jump
; back to coal_loop to expand the block.
;
; Entry:
;	block is marked inuse
;	ax = header of next block (not free)
;	dx = header of block
;	si = pointer to next block + 2
;	di = pointer to block
;

try_grow:
	cmp	ax,_HEAP_END	; at end-of-heap?
	jne	expand_fail	;   no, fail
	test	byte ptr ds:[bx].flags,_HEAP_MODIFY
				; permission to modify the heap segment size?
	jz	expand_fail	;   no, fail
	lea	ax,[di+4]	; ax = block offset + header size + eoh size
	add	ax,cx		; new size + ax >= 64K?
	jnc	try_grow1	;   new size ok, try to grow the segment
	mov	[FailFlag],1	; note that we must ultimately return failure
	sub	cx,ax		; cx now adjusted to give a segment size of
				; exactly 64K
try_grow1:
	inc	word ptr ds:[di] ;mark block as free
	cmp	di,ds:[bx].rover ;di < rover?
	jnb	try_grow2	 ;  no
	mov	ds:[bx].rover,di ;yes, reset it

try_grow2:
	push	dx		; must explicitly save dx and si
	push	si
	call	_growseg	; _growseg preserves cx and di!
	pop	si
	pop	dx
	jnc	coal_again	; the pops above don't the affect carry flag
	mov	[FailFlag],1	; make sure failure is recorded
	cmp	cx,10h		; have we already tried < 1 paragraph?
	jb	expand_fail	;   yes, time to fail
	shr	cx,1		; divide by 2
	adc	cx,0		; add on lost bit
	jmp	short try_grow2 ; try growing again

coal_again:
	dec	word ptr ds:[di] ;mark block inuse again
	lea	si,[di+2]
	mov	ax,dx
	jmp	short coal_loop ; try coalescing again

;
; expand_fail
; The block has been expanded as far as possible and it wasn't enough. Set
; ax to indicate failure, adjust si and jump to the exit code.
;
; Entry:
;	dx = header of block
;	si = pointer to next block + 2
;	di = pointer to block
;

expand_fail:
	mov	ax,1		; ax = 1 to indicate failure
	dec	si		; si = pointer to next block (this is what
	dec	si		; expand_exit expects)
	jmp	short expand_exit

;
; heap_toast
; A bad heap block has been encountered and this heap segment is probably
; now trashed. Set errno and return failure.
;
;	*** WHAT DO WE SET ERRNO TO AND HOW???? FOR NOW JUST CLEAN THE
;	*** STACK UP, SET THE CARRY FLAG AND GET OUT.
;

heap_toast:
	pop	dx		; clean up stack
	stc
	jmp	short done

;
; expand_exact
; We have an exact fit. Set si to point to the next block (this is what
; expand_exit is expecting) and branch to expand_exit.
;

expand_exact:
	inc	si
	inc	si
	add	si,cx
	jmp	short expand_exit

;
; expand_succeed
; If necessary, divide the block into two allocation blocks. The first is
; the resized original block, with size exactly equal to cx. The second is
; whatever is leftover and is marked free.
;
; Entry:
;	dx = header of block
;	di = pointer to block
;	zf = set, if we have an exact fit
;	   = clear, otherwise
;

expand_succeed:
	mov	ax,[FailFlag]	; stash FailFlag in ax for now
	mov	si,di
	je	expand_exact	 ; exact fit, go finish up
	or	ax,ax		; have we failed to satisfy user request?
	jnz	expand_exit	;   yes, give user whatever expansion we
				;   were able to obtain
	mov	ds:[di],cx	; shrink block to be an exact fit
	inc	si
	inc	si
	add	si,cx		; si = pointer to new (remainder block)
	jc	heap_toast	; bad heap!
	sub	dx,cx		; dx = size of new block + 2
	dec	dx		; dx = size of new block + free flag
	mov	ds:[si],dx	; set header for new block
	dec	dx		; dx = size of new block

;
; expand_exit
; Restore the original setting of the block's free flag, check the rover
; pointer and reset it if necessary, and set the carry flag to indicate
; success (set) or failure (clear).
;
; Entry:
;	ax = 0, if success
;	     1, if failure
;	si = pointer to block, if no resizing has occurred
;	   = pointer to next block, if any resizing has occurred
;	di = pointer to block
;

expand_exit:
	pop	cx		; retrieve original header from tos
	xchg	ax,cx		; ax = header, cx = flag value
	and	ax,1		; ax = original free/inuse flag value
	and	word ptr ds:[di],not 1 ; restore block's original free/inuse
	or	word ptr ds:[di],ax    ; status

check_rover:
	mov	dx,ds:[bx].rover ;get rover pointer
	cmp	di,dx		; di < rover?
	jnb	set_carry	;   no, rover is okay, go set up return
	cmp	si,dx		; si > rover?
	jna	set_carry	;   no, rover is okay, go set the carry flag
	mov	ds:[bx].rover,di ;we have absorbed the block pointed to by
				; rover, reset it
set_carry:
	neg	cx		; clears carry if cx is 0, else sets carry
done:
	cEnd

sEnd	code
	end
