	page	,132
	title	searchsg - Search a heap segment
;***
;searchsg.asm - Search a heap segment
;
;	Copyright (c) 1988-1992, Microsoft Corporation. All rights reserved.
;
;Purpose:
;	Contains the _searchseg() routine that searches a single
;	heap segment looking for a block of specified size.
;
;*******************************************************************************

.xlist
include version.inc
include cmacros.inc
include heap.inc
.list


sBegin	code

	assumes cs,code

page
;***
; _searchseg - Search a heap segment
;
;Purpose:
;
; This routine scans through the supplied heap segment looking for a
; block big enough to satisfy the size request.  Contiguous free blocks
; are coalesced as necessary.
;
; The search is done in two parts:
;
;	(1) First, we search from the current rover pointer to the end
;	of the heap segment.
;
;	(2) Second, we search from the start of the heap segment to the
;	rover pointer (being careful to overlap so that all free blocks
;	are coalesced as much as possible).
;
;Entry:
;	cx = requested size
;	     [If requested size = _HEAP_COALESCE, then we simply coalesce
;	     the heap segment and return with carry set]
;	ds:bx = address of heap descriptor
;
;Exit:
;	Success:
;		carry = clear
;		dx:ax = address of block
;	Failure:
;		carry = set
;
;Uses:
;	ax, dx, si, di
;
;Preserved:
;	ds:bx, cx
;
;Exceptions:
;	If we detect an inconsistent heap, we return NULL and set errno
;	to ***???***.
;
;*******************************************************************************

cProc	_searchseg,<PUBLIC, NEAR>,<>

cBegin	<nogen> 			; nogen

	inc	cx
	and	cl,not 1		; round size up to even number

;
; first_pass -
; On the first pass through the heap, we will search from the current
; rover pointer to the end of the heap.  Set the 'end-of-search' pointer
; such that the search will not end until we encounter the _HEAP_END flag.
;
; Entry:
;			BX = heap descriptor
;			CX = requested size
;
; Exit Paths:
;		check_block
;			SI = pointer where the heap search should start
;			DI = 0 (pointer to last free block examinted)
;			BX = pointer to _HEAP_END (where search should end)
;			CX = requested size
;			TOS = heap descriptor
;

first_pass:
	push	bx			; save heap descriptor
	cld				; go forward when loading
	mov	si,[bx].rover		; si = pointer to start of search
	mov	bx,[bx].last		; bx = pointer to end-of-heap marker
	xor	di,di			; zero out di
	jmp	short check_block	; jump into loop


; next_pass -
; We have completed a pass through the heap.
;	(1) Update .last pointer if necessary.	The only way this could happen
;	is if we coalesced the last block with one before it.  Thus, we only
;	need a pointer to the last free block we examined to determine if
;	.last must be updated.
;	(3) Determine which pass has been completed by checking the lower
;	order bit of the end-of-search pointer.
;	(3) If we just finished the second pass, return an error.
;	(4) Else, start up a heap search from the start of the heap to the
;	rover pointer. Use rover - 1 as the value of the end-of-search pointer
;	since it works are well as rover does and it makes it easy to
;	determine which pass we have completed.
;
; Entry:
;			BX = end-of-search pointer = pointer to end-of-heap
;			     marker (after first pass) or rover - 1 (after
;			     second pass)
;			CX = requested size
;			DI = 0 or pointer to last free block examined + 2
;			TOS = heap descriptor
;
; Exit Paths:
;		failure1 (end of search, block not found)
;			TOS = heap descriptor
;		failure2 (end of search, block not found)
;			BX = heap descriptor
;		check_block (start up the second heap search)
;			SI = pointer where the heap search should start
;			DI = 0 (pointer to last free block examinted)
;			BX = end-of-search pointer = rover - 1
;			CX = requested size
;			TOS = heap descriptor
;

next_pass:
	mov	ax,bx			; ax = end-of-search pointer
	pop	bx			; bx = heap descriptor

; see if we need another pass through the heap

check_pass:
	test	al,1			; test low order bit on eos pointer
	jnz	failure2		; bit was set, we're done

; set up for second pass through the heap

	push	bx			; save heap descriptor
	mov	si,[bx].start		; si = start search ptr = start
	mov	bx,[bx].rover		; bx = end-of-search ptr = rover
	cmp	bx,si			; start == rover ??
	je	failure1		; if so, no need to try another pass
	dec	bx			; end-of-search pointer = rover - 1
	xor	di,di			; zero out di

	jmp	short check_block	; jump into loop

;
; ---------- Start of segment search loop ----------
;
; The following registers are not changed by any code in the segment
; search loop:
;			BX = end of search pointer
;			CX = requested size
;			TOS = heap descriptor
;

;
; next_block -
; See if we're at the end of the segment search loop.  If not, move
; on to the next block.
;
; Entry:
;			AX = header of last block examined (not free)
;			SI = pointer to last block examined (not free) + 2
;			DI = 0 or pointer to last free block examined + 2
;
; Exit Paths:
;		next_pass (first pass through heap failed)
;			DI = 0 or pointer to last free block examined + 2
;
;		check_block (info for next block is loaded into AX and SI)
;			AX = header for (new) block to be checked
;			SI = pointer to (new) block to be checked + 2
;			DI = 0 or pointer to last free block examined + 2
;

EVEN
next_block:
	lea	dx,[si-2]		; get current ptr
	cmp	dx,bx			; current ptr >= end-of-search ptr ?
	jae	next_pass		; yes, done with this pass

	add	si,ax			; bump pointer to next block
	jc	heap_toast		; *** inconsistent heap ***
	;fall thru

;
; check_block -
; Check to see if the current block is free.
;
; Entry:
;			SI = pointer to current block
;			DI = 0 or pointer to last free block examined + 2
;
; Exit Paths:
;		next_block (if current block is not free)
;			AX = current block (not free) header
;			SI = pointer to current block + 2
;			DI = 0 or pointer to last free block examined + 2
;		coal_loop (if current block is free)
;			AX = current block (free) header
;			SI, DI = pointers to current block + 2
;

check_block:
	lodsw				; ax = get the first block header
	test	al,1			; is it free?
	jz	next_block		; no, go try the next block
	mov	di,si
	;fall thru

;
; coal_loop -
; Check to see if current block is big enough. If not, try to
; coalesce proceeding free block with the current block. Loop until we get
; get a big enough block or cannot coalesce anymore.
;
; Entry:
;			AX = current block header
;			SI, DI = pointers to current block + 2
;
; Exit Paths:
;		blockfound (current block is big enough)
;			AX = size of current block
;			SI, DI = pointers to current block + 2
;		next_block (failure)
;			DX = size of current block
;			DI = pointer to current block + 2
;			AX = header of next block (not free)
;			SI = pointer to next block (not free) + 2
;

EVEN
coal_loop:
	dec	ax			; subtract free flag
	cmp	ax,cx			; is it big enough?
	jae	blockfound		;   yep, go finish up
	add	si,ax			; point to header of next block
	jc	heap_toast		; *** inconsistent heap ***
	mov	dx,ax			; save size of original block in DX
	lodsw				; get header of next block into AX
	test	al,1			; is it free?
	jz	next_block		;   no, go try the next block
	add	ax,dx			; coalesce
	add	ax,2			; add in header space
	mov	si,di			; reset SI to block + 2
	mov	[si-2],ax		; update block size
	jmp	short coal_loop

;
; ---------- End of segment search loop ----------
;

;
; heap_toast -
; We detected an internal consistency error in the heap.
; Set errno and return an error to the user.
;

heap_toast:

	; xor	dx,dx			; _doserrno = 0
	; mov	ax,E????		; bad heap
	; jmp	_dosretfnull
	;
	;*** NEED A ROUTINE THAT SETS ERRNO AND RETURNS NULL
	;(_dosret/_dosretf always return -1; maybe we need _dosretnull
	; and _dosretfnull).

	mov	ax,ax	   ; place where we can place a breakpoint for now...
	; fall thru for now

;
; failure1, failure2 -
; We could not find a big enough block in the heap.
;
;    (1) Set the rover pointer to the beginning of the heap since
;	 the rover block may have been coalesced (i.e., the rover
;	 pointer may no longer point to valid memory).
;
;	*** I THINK THIS IS WHAT THE OLD HEAP IS DOING.
;	*** DO WE WANT TO DO THIS OR CAN WE SOMEHOW MOVE THE
;	*** ROVER BACK TO THE MERGED BLOCK, IF NECESSARY.
;
;    (2) Return carry set to caller.
;
; Entry:
;		failure1
;			TOS = heap descriptor
;
;		failure2
;			BX = heap descriptor
;			CX = requested size
;
; Exit Paths:
;		Return to caller (via done)
;			Carry = set
;			DS:BX = heap descriptor
;			CX = requested size
;

failure1:
	pop	bx			; bx = heap descriptor
failure2:
	mov	ax,[bx].start		; set rover = start
	mov	[bx].rover,ax
	stc				; carry set = error
	jmp	short done

;
; blockfound -
; Found a block that fullfills the caller's request.
;	(1) Transform the block found into an allocated block that
;	exactly satisfies the allocation request (in CX) and a leftover,
;	free block.
;	(2) Update the rover and end pointers, as needed.
;	(3) Return pointer to the allocated block to the caller.
;
; Entry:
;			AX = size of (free) block (AX >= CX)
;			SI, DI = pointers to block + 2
;			CX = requested size
;			ZF = 1 iff the block is an exact fit (i.e., AX == CX)
;			     0 otherwise
;			TOS = heap descriptor
;
; Exit Paths:
;		Return to caller
;			Carry = clear
;			DX:AX = pointer to allocated block
;			DS:BX = heap descriptor
;			CX = requested size
;

blockfound:
	pop	bx			; get heap descriptor

	mov	[si-2],cx		; update header
	je	set_rover		; jump if block size = request size

	add	di,cx			; point to leftover piece
	sub	ax,cx			; compute header for leftover piece
	dec	ax			; as a free block
	mov	[di],ax 		; make it a free block
	sub	di,cx			; update rover

set_rover:
	add	di,cx			; point to leftover piece
set_rover1:
	mov	[bx].rover,di		; update rover
	mov	ax,si			; return pointer to allocated block
	mov	dx,ds

	clc				; carry clear = success

;
; Common exit
;

done:

	ret				; explicit return
cEnd	<nogen> 			; nogen

sEnd	code

	end
