/***
*nheap.c - Windows near heap routines
*
*   Copyright (c) 1987-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   Provides heap routine interfaces for windows/C libs.
*   Contains the "essential" routines.
*
*   NOTE:  Currently, the C runtime libs assume that the DGROUP
*   selector will NEVER change value.  To ensure this, lock down
*   DGROUP whenever issueing a local allocation call.
*
*******************************************************************************/

#include <windows.h>
#include <malloc.h>
#include <sizeptr.h>
#include <stddef.h>

/*---------------------------------------------------------------------------*/
/*                                       */
/*  _nmalloc()                                   */
/*                                       */
/*---------------------------------------------------------------------------*/

#ifdef _NOTCXX

void _near * _nmalloc(n)
size_t n;

{
  HANDLE h;

  LockData(0);      /* Always lock/unlock DGROUP */

  h = (HANDLE)LocalAlloc(LMEM_NODISCARD | LMEM_FIXED, n ? n : 1);

  UnlockData(0);

  return((void _near *)h);

}


#else   /* !_NOTCXX */

int (__cdecl * _pnhNearHeap)( size_t ) = NULL;

void _near * _nmalloc(n)
size_t n;

{
  HANDLE h;

  n = (n ? n : 1);  /* if n == 0, n = 1 */

  for (;;) {

    LockData(0);          /* Always lock/unlock DGROUP */

    h = (HANDLE)LocalAlloc(LMEM_NODISCARD | LMEM_FIXED, n);

    UnlockData(0);

    if ((h != 0) || (_pnhNearHeap == NULL) || (*_pnhNearHeap)(n) == 0)
        return((void _near *)h);

  }

}

#endif  /* _NOTCXX */

/*---------------------------------------------------------------------------*/
/*                                       */
/*  _nfree()                                     */
/*                                       */
/*---------------------------------------------------------------------------*/

void _nfree(p)
void _near * p;

{
  if (p != (void _near *) NULL)
    LocalFree((HANDLE)p);
}

/*---------------------------------------------------------------------------*/
/*                                       */
/*  _nrealloc()                                    */
/*                                       */
/*---------------------------------------------------------------------------*/

void _near * _nrealloc(p, n)
void _near * p;
size_t  n;

{
  HANDLE h;

  /* if ptr is NULL, call malloc */
  if (p == (void _near *) NULL)
    return(_nmalloc(n));

  /* if ptr is !NULL and size is 0, call free and return NULL */
  if (n == 0) {
    _nfree(p);
    return((void _near *) NULL);
    }

  LockData(0);      /* Always lock/unlock DGROUP */

  h = ((HANDLE)LocalReAlloc((HANDLE)p, n ? n : 1,
                   LMEM_MOVEABLE | LMEM_NODISCARD | LMEM_ZEROINIT));
  UnlockData(0);

  return((void _near *)h);
}

/*---------------------------------------------------------------------------*/
/*                                       */
/*  _nmsize()                                    */
/*                                       */
/*---------------------------------------------------------------------------*/

size_t _nmsize(p)
void _near * p;

{
  return((size_t)LocalSize((HANDLE)p));
}
