/***
*whseg.c - Halloc/Hfree support for windows
*
*   Copyright (c) 1990-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   Support routines for halloc/hfree on windows.
*   We need this table because we have to remember the handles
*   associated with huge windows segments.  Even though handles
*   equal segments in Win/16, they do not in other windows
*   environments.
*
*   [The "wh" stands for "Windows Huge".]
*
*******************************************************************************/


#include <malloc.h>
#include <stddef.h>
#include <string.h>

/*
 * Structure of each windows segment/handle entry
 */

struct _whseg {
    unsigned int segment;
    unsigned int handle;
    };
typedef struct _whseg WHSEG;


/*
 * Pointer to the windows segment/handle table,
 * Current size of table (in bytes),
 * Table increment constant (in entries).
 */

WHSEG * _whtable = NULL;
_whtabsiz = 0;

#define  WHTABINC 10


/***
*_whclear() - Clear a segment's table entry
*
*Purpose:
*
*   Remove a segment's table entry from the table and return
*   corresponding handle.
*
*Entry:
*   unsigned seg = segment value
*
*Exit:
*   !0 = unsigned handle = handle value corresponding to the segment
*    0 = error (segment not found)
*
*Exceptions:
*
*******************************************************************************/

unsigned near  _whclear (seg)
unsigned seg;
{

    WHSEG * p;
    WHSEG * end;

    /* look for segment in table */
    for (p=_whtable, end=_whtable+(_whtabsiz/sizeof(WHSEG));
         p < end; p++)
        {
        if (p->segment == seg) {
            p->segment = 0;
            return(p->handle);
            }
        }

    /* not found */
    return(0);

}



/***
*_whset() - Set a segment's entry
*
*Purpose:
*
*   Add a segment and it's handle to the table.
*   Grow table if necessary.
*
*Entry:
*   unsigned seg = segment value
*   unsigned hand = handle corresponding to segment
*
*Exit:
*   !0 = success = segment
*    0 = error (entry not added to table)
*
*Exceptions:
*
*******************************************************************************/

unsigned near  _whset (seg, hand)
unsigned seg;
unsigned hand;
{

    WHSEG * p;
    WHSEG * end;
    unsigned newsize;

    /*
     * look for an empty slot in the table
     */

    for (p=_whtable, end=_whtable+(_whtabsiz/sizeof(WHSEG));
         p < end; p++)
        {
        if (p->segment == 0) {
            p->segment = seg;
            p->handle = hand;
            return(seg);    // success
            }
        }

    /*
     * table is full, try for more memory
     */

    newsize = _whtabsiz+(WHTABINC*sizeof(WHSEG));
    if ((p = realloc(_whtable,newsize)) == NULL)
        return(0);      // error

    /*
     * set up new data
     */

    _whtable = p;           // new table ptr

    p += (_whtabsiz/sizeof(WHSEG)); // init first new entry
    p->segment = seg;
    p->handle = hand;

    _whtabsiz = newsize;        // update size

    p++;                // zero out rest of entries
    memset( p, 0, ((WHTABINC-1)*(sizeof(WHSEG))) );

    return(seg);            // success

}
