	page	,132
	title	open - open a file
;***
;open.asm - file open
;
;	Copyright (c) 1985-1992, Microsoft Corporation. All rights reserved.
;
;Purpose:
;	defines open() - open or create a file
;
;*******************************************************************************

.xlist
include version.inc
include cmacros.inc
include msdos.inc
include errno.inc		; errno.h  equiv.
include fcntl.inc		; fcntl.h  equiv.
.list


sBegin	data

	assumes ds,data

externW _fmode			; default file mode flag (8000H=binary)
externB _osfile 		; file handle flags, etc.
ifndef _WINDOWS
externB _osmajor		; OS version number
endif
externW _nfile			; number of handles supported by C run-time
externW _umaskval		; current umask value

sEnd

sBegin	code

extrn	__dosretax:near

	assumes cs,code
	assumes ds,data

page
;***
;int _sopen(path, oflag, shflag, pmode) - opne a file with sharing
;
;Purpose:
;	Opens the file with possible file sharing.
;	shflag defines the sharing flags:
;	  SH_COMPAT -	set compatability mode
;	  SH_DENYRW -	deny read and write access to the file
;	  SH_DENYWR -	deny write access to the file
;	  SH_DENYRD -	deny read access to the file
;	  SH_DENYNO -	permit read and write access
;
;	Other flags are the same as open().
;
;	SOPEN is the routine used when file sharing is desired.  If the DOS is
;	not version 3.0 or higher, we simply ignore the sharing mode.  The
;	sharing mode is set into BH (0 if not DOS 3.0 or later) and then we
;	use a sneaky trick to share code with open - since the sopen arguments
;	the same as the open arguments with the exception of the sharing mode
;	which is stuck just before the optional mode field, we simply copy the
;	mode value (garbage if no mode was given) into what used to be the
;	sharing mode argument field (which is now in BH) and jump into OPEN().
;
;   WARNING:
;	This routine must be kept synchronized with Real- and Protected-Mode
;	versions of "OPEN.ASM" as far as the local stack variables.
;
;Entry:
;	char *path -	file to open
;	int oflag -	open flag
;	int shflag -	sharing flag
;	int pmode -	permission mode (needed only when creating file)
;
;Exit:
;	returns file handle for the opened file
;	returns -1 and sets errno if fails.
;
;Uses:
;
;Exceptions:
;
;*******************************************************************************

cProc	_sopen,<PUBLIC>,<>

	parmDP	opath
	parmW	oflag
	parmW	oshflag
	parmW	omode

	localB	RdBuf		; for ^Z read ahead
	localB	ShMode		; sharing mode, non-zero if sharing requested
	localB	Exists		; set to 1 if file previously existed
	localB	TextDev 	; FTEXT bit set if opened in TEXT mode
				; FDEV bit set if filename is a device
cBegin

ifndef _WINDOWS 		; Win 3.x only runs on DOS 3.x or greater
	xor	bh,bh		; default to no sharing mode on DOS 2.X
	cmp	_osmajor,3	; 3.x or above?
	jb	defmode 	; no, ignore sharing modes
endif

	mov	bh,byte ptr [oshflag] ; sharing mode
defmode:

	mov	ax,omode
	mov	oshflag,ax	; move mode up to look like open param list
	jmp	short __copensub ; let open do the work

cEnd	nogen


page
;***
;int _open(path, flag, pmode) - open or create a file
;
;Purpose:
;	Opens the file and prepares for subsequent reading or writing.
;	the flag argument specifies how to open the file:
;	  O_APPEND -	reposition file ptr to end before every write
;	  O_BINARY -	open in binary mode
;	  O_CREAT -	create a new file; no effect if file already exists
;	  O_EXCL -	return error if file exists, only use with O_CREAT
;	  O_RDONLY -	open for reading only
;	  O_RDWR -	open for reading and writing
;	  O_TEXT -	open in text mode
;	  O_TRUNC -	open and truncate to 0 length (must have write permission)
;	  O_WRONLY -	open for writing only
;	exactly one of O_RDONLY, O_WRONLY, O_RDWR must be given
;
;	The pmode argument is only required when O_CREAT is specified.	Its
;	flag settings:
;	  S_IWRITE -	writing permitted
;	  S_IREAD -	reading permitted
;	  S_IREAD | S_IWRITE - both reading and writing permitted
;	The current file-permission maks is applied to pmode before
;	setting the permission (see umask).
;
;	The oflag and mode parameter have different meanings under DOS. See
;	the A_xxx attributes in msdos.inc
;
;	The SOPEN routine jumps directly into this function at __copensub.
;	"sopen()" will have set BH to the file sharing mode requested.
;
;	The declaration of local stack variables in SOPEN.ASM must be kept
;	synchronized with both the Real and Prot-Mode versions of this routine.
;
;Entry:
;	char *path - file name
;	int flag - flags for open()
;	int pmode - permission mode for new files
;
;Exit:
;	returns file handle of open file if successful
;	returns -1 (and sets errno) if fails
;
;Uses:
;
;Exceptions:
;
;*******************************************************************************

cProc	_open,<PUBLIC>,<>

	parmdp	opath
	parmw	oflag
	parmw	omode

	localB	RdBuf		; for ^Z read ahead
	localB	ShMode		; sharing mode, non-zero if sharing requested
	localB	Exists		; set to 1 if file previously existed
	localB	TextDev 	; FTEXT bit set if opened in TEXT mode
				; FDEV bit set if filename is a device
cBegin
	xor	bh,bh		; indicating no file sharing

__copensub:
	mov	[ShMode],bh	; save sharing mode (0 if none)
	mov	ax,[oflag]	; load open mode flag
	mov	cx,ax		; save open flags

; check if the file/device is to be opened in TEXT mode
; and save the information in [TextDev] for later use

	mov	[TextDev],0	; Assume binary until proven otherwise
	test	ax,O_BINARY
	jnz	NotTextMode	; not TEXT mode

	test	ax,O_TEXT	; forced to TEXT mode ?
	jnz	IsTextMode	; yup, set TEXT mode flag

				; mode not explicitly set
	test	byte ptr [_fmode+1],80H
				; binary default?
	jnz	NotTextMode	; not TEXT mode
IsTextMode:
	mov	[TextDev],FTEXT ; set TEXT mode flag
NotTextMode:

; we will first open the file to see if it exists already or not.

if	sizeD
	push	ds		; save ds
	lds	dx,[opath]	; get path name
else
	mov	dx,[opath]	; get path name
endif

	and	al,3		; mask to 0-2 only
	or	al,bh		; sharing mode
	callos	open		; see if file or device exists

if	sizeD
	pop	ds		; restore ds
endif

	jnc	FileExists	; file exists

	cmp	ax,E_nofile	; is the error "no such file"?
	jne	SetCarry	; no, just fail now
	test	cx,O_CREAT	; file doesn't exist, permission to create?
	jz	SetCarry	; no, give error
	jmp	NewFile 	; yes, see if we can create it

SetCarry:
	stc			; set carry to indicate error
	jmp	__dosretax	; return error value


FileExists:

; We come here if the file already exists and was opened ok.  The first thing
; we do is check to see if an exclusive file was requested (O_CREAT and O_EXCL
; both set).  If so, we fail immediately.  Otherwise we check to see if this
; is a device and if it is opened in TEXT mode and save this information

	xchg	ax,bx		; save (bx) = file handle
	mov	ax,cx		; reload saved open flags
	and	ax,O_CREAT+O_EXCL
	cmp	ax,O_CREAT+O_EXCL ; both O_CREAT and O_EXCL set?
	jne	OpenOk		; no, continue on

	callos	close		; yes, close file
	mov	ax,EEXIST shl 8 + 0 ; return file-exists error; no dos error
	jmp	SetCarry

OpenOk:

; check if this is a device and save the information in [TextDev] for later.
; At this point we have the following items in registers
;		BX = file handle
;		CX = oflag

	mov	[Exists],1	; mark that file is not new
	mov	ax,DOS_ioctl shl 8 + 0 ; function 0 (get dev info)
	callos
	test	dl,80H		; test if file is a device
	jz	ModeSet 	; no, go on
	or	[TextDev],FDEV	; yes, set device flag

ModeSet:

; ok, now we have accumulated the information we need for later.  If we have
; a device, then we are done, except for the finishing touches.  If we have
; a file, then we must do some further work, depending on the access flags
; specified.
; At this point we have the following items in registers
;		AX = oflag
;		BX = file handle

	test	[TextDev],FDEV	; is it a device?
	jz	IsFile		; no, go on
	jmp	WeDone		; yes, we are done

IsFile:

; ok, its a file - now lets check for all the ugly cases to handle

	mov	ax,[oflag]
	test	ax,O_TRUNC	; truncating file ?
	jz	ChkCtrlZ	; no, check for ^Z terminated file

; We are truncating the entire file.  If the file is opened for writing, we
; can simply do a write, but if the file is opened for read-only, then we must
; get the existing files attributes into CX for use in the creat call, close
; the file, and then go creat it.

	test	ax,O_RDWR+O_WRONLY ; truncating and opening for writing?
	jz	HardWay 	; nope, do it the hard way

	xor	cx,cx		; its writable, just write 0 bytes
	callos	write
	jmp	WeDone		; should be done

HardWay:

; OK, this guy has asked for the incredible (O_TRUNC|O_RDONLY access on an
; existing file, so we will give it to him, albeit slowly.  Close the file,
; get the file attributes, and go through the creat process.

	callos	close		; close file

if	sizeD
	push	ds		; save ds
	lds	dx,[opath]	; get path name
else
	mov	dx,[opath]	; get path name
endif

	mov	ax,DOS_filemode shl 8 + 0 ; get existing file attribute into CX
	callos

if	sizeD
	pop	ds		; restore ds
endif

ifdef _WINDOWS
	jmp	CreatFile	; go use creat to open the file
else
	jmp	short CreatFile ; go use creat to open the file
endif

ChkCtrlZ:

; We can only play the truncate-the-^Z game on files which are opened
; for read/write access (O_RDWR) because of file sharing problems.
; The file must also be opened in TEXT mode for obvious reasons.
; At this point we have the following items in registers
;		AX = oflag
;		BX = file handle

	test	[TextDev],FTEXT ; TEXT mode?
	jnz	ChkRdWr
	jmp	WeDone		; no, don't worry about ^Z's

ChkRdWr:
	test	ax,O_RDWR	; read/write access given?
	jnz	TryTrunc	; yup, lets check for ^Z's
	jmp	WeDone		; no, don't worry about ^Z's

TryTrunc:

; We are opening a file in TEXT mode for read/writing.	We must detect if
; the file is ^Z terminated and remove the ^Z, otherwise seeking to the
; end-of-file and writing (or using APPEND mode) will cause funny results

	mov	cx,-1
	mov	dx,cx		; -1L
	mov	ax,DOS_lseek shl 8 + 2 ; move to end - 1
	callos			; do the seek
	neg	cx		; set CX to 1 (was -1 from lseek call)
if sizeD OR ?WIN
	push	ds		; save ds
	push	ss
	pop	ds		; ds == ss
endif
	lea	dx,[RdBuf]
	callos	read		; read one byte to see if it's ^Z
if sizeD OR ?WIN
	pop	ds		; restore ds
endif
	or	ax,ax
	jz	Rewind		; read 0 bytes -> 0 length file

	cmp	[RdBuf],1aH	; ^Z terminated ?
	jne	Rewind		; no, clean up

	neg	cx		; yes, lseek back prior to ^Z
	mov	dx,cx		; -1L
	mov	ax,DOS_lseek shl 8 + 2 ; go back one byte
	callos
	xor	cx,cx		; write 0 bytes, truncates file
	callos	write

Rewind:

; we have done our check for ^Z terminated files, reposition to the beginning
; of the file and go finish up

	xor	cx,cx
	mov	dx,cx
	mov	ax,DOS_lseek shl 8 + 0 ; seek to beginning of file
	callos

ifdef _WINDOWS
	jmp	WeDone
else
	jmp	short WeDone
endif

; This ends the code for the case where the file to be opened already exists

NewFile:

; We get here only if we are creating a brand new file (note that there
; is an entry point further down, CreatFile, which is used for existing
; files which are being truncated and have read-only access specified.
; The only reason we allow this craziness is because Xenix does.

	mov	[Exists],0	; clear file exists flag

; load the requested attributes/permissions into CX for the call to creat

	mov	cx,[omode]	; sys level, translate to XENIX style modes
	call	__cXENIXtoDOSmode ; convert to DOS mode
	mov	[omode],cx	; store it back in omode

; here we must do some gymnastics to handle the case where we are creating
; a new file (or truncating an existing one) with either a sharing mode or
; with read (write) only access specified.  In these cases, we want to use
; the creat system call to do the work, but we have two problems:
;
;	1. creat does not allow us to set sharing modes
;	2. creat always opens a file for both read and write access
;
; so the open routine must first call creat to create the file and then close
; it and re-open it with open in order to get the file sharing mode and/or
; the access mode set correctly.  Unfortunately, if a READ-ONLY permission
; is specified for a new file, creat will set this, so when we try to re-open
; it for writing (if requested), it fails.  This could also happen if the file
; existed and O_TRUNC was specified too.  As a result we must detect this
; situation and creat the file with WRITE permission, close it, re-open it, and
; use the change file attributes system call to set the correct permission
; on the file.	Since the new permission does not affect the open file
; handle, the user can still write to the file as requested, but after
; file is closed it will be read-only, also as requested.

	test	[ShMode],-1	; file sharing mode specified ?
	jnz	NoRdOnly	; yes, clear read only flag before creat

	test	[oflag],O_RDWR	; no sharing, read and write requested ?
	jnz	CreatFile	; yes, no problems, creat it

NoRdOnly:
	and	cl,not A_ro	; clear read-only attribute

CreatFile:

; ok, we're all set up to either truncate an existing file or create a new
; one.	At this point we have the following values in registers
;		CX = file attributes (DOS form)

if	sizeD
	push	ds
	lds	dx,[opath]	; path name
else
	mov	dx,[opath]	; path name
endif

	callos	create		; creat new file or truncate existing one

if	sizeD
	pop	ds		; restore ds
endif

	jnc	ChkReopen	; created ok, see if must close and reopen

ErrRet:
	jmp	__dosretax	; return error

ChkReopen:
	xchg	ax,bx		; (bx) = file handle

	test	[ShMode],-1	; sharing mode requested ?
	jnz	CloseFile	; yes, must close and reopen

	test	[oflag],O_RDWR	; no sharing, read/write access ?
	jnz	WeDone		; yes, we're done, finish up

CloseFile:
	callos	close		; close file again

; now we can try to reopen with all the proper access and/or sharing modes
; requested.  This could fail in some cases so we have to look out for this.

	mov	al,byte ptr [oflag] ; move open flags to al
	and	al,3		; mask to 0-2 only
	or	al,[ShMode]	; sharing mode

if	sizeD
	push	ds
	lds	dx,[opath]	; path name
else
	mov	dx,[opath]	; path name
endif

	callos	open		; reopen file

if	sizeD
	pop	ds		; restore DS
endif

	jc	ErrRet		; reopen failed, return error

	xchg	ax,bx		; set BX = file handle

; if read-only permission was originally requested, and the file didn't
; previously exist, set it now

	test	[Exists],1	; did file previously exist
	jnz	WeDone		; yes, don't change permissions

	test	[omode],A_ro	; new file, was read-only permission specified ?
	jz	WeDone		; no, don't need to change anything

; note that the code below is known not to work on network systems where, if
; share is installed, the DOS will close the file before or after (not sure)
; the chmod operation

	or	cl,A_ro 	; yes, set read only flag in attribute byte

if	sizeD
	push	ds		; save ds
	lds	dx,[opath]	; load path name
else
	mov	dx,[opath]	; load path name
endif

	mov	ax,DOS_filemode shl 8 + 1
	callos			; set read only attribute

if	sizeD
	pop	ds		; restore ds
endif

	jc	ErrRet		; return error if it occurs

; This ends the section of code used to handle the case where the file
; did not already exist

WeDone:

; Now we do the final cleanup - setting up the __osfile entry and returning.
; At this point we have the following items in registers
;		BX = file handle

	test	[TextDev],FDEV	; device?
	jnz	SetRestDev	; yes, cannot get attributes
				; and must ignore O_APPEND mode

if	sizeD
	push	ds		; save ds
	lds	dx,[opath]	; load pathname pointer
else
	mov	dx,[opath]	; load pathname pointer
endif

	mov	ax,DOS_filemode shl 8 + 0 ; get file attributes
	callos

if	sizeD
	pop	ds		; restore ds
endif

	mov	ax,cx		; reload saved open flags
;
; At this point use CL to hold file mode flags to be stored in __osfile[bx]
;
	xor	cl,cl		; clear file mode flags
	and	ax,A_ro 	; check read-only bit of open flags
	jz	SetAppend

SetRdOnly:
	mov	cl,FRDONLY	; yes, set read only bit in flag accumulator

SetAppend:
	test	[oflag],O_APPEND ; is append mode set?
	jz	SetRest 	; no, set rest of flags
	or	cl,FAPPEND	; yes, set FAPPEND bit

SetRest:
	cmp	bx,[__nfile]	; ensure that file handle is in the
	jb	fh_okay 	; range supported by the C run-time

	callos	close		; close the handle

	mov	ax,EMFILE shl 8 + 0
	jmp	SetCarry	; set Carry Flag and then Jump to __DOSRETAX

fh_okay:
	or	cl,[TextDev]	; set TEXT mode flag and DEVice flag
	or	cl,FOPEN	; mark file as open
	mov	_osfile[bx],cl	; set file modes

	mov	ax,bx		; set up file handle for return
cEnd

SetRestDev:
	xor	cl,cl		; clear file mode flags
	jmp	SetRest 	; then set the ones that still apply

page
;***
;_cXENIXtoDOSmode - convert XENIX mode to DOS attribute byte
;
;Purpose:
;	Converts XENIX mode flag to a DOS attribute byte.
;
;Entry:
;	CX - Xenix mode
;
;Exit:
;	CX - DOS attribute byte
;
;Uses:
;	AX, CX (assumes that BP and SP are unaltered)
;
;Exceptions:
;
;*******************************************************************************

cProc	_cXENIXtoDOSmode,<NEAR,PUBLIC>

cBegin	nogen
	mov	ax,[_umaskval]	; get current program umask
	not	ax
	and	ax,cx		; apply umask
	xor	cx,cx
	test	al,80H		; write permitted
	jnz	wrtok		; yes

	or	cl,A_ro 	; read only

wrtok:
	ret
cEnd	nogen

sEnd
	end
