/***
*fdopen.c - open a file descriptor as stream
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines fdopen() - opens a file descriptor as a stream, thus allowing
*   buffering, etc.
*
*******************************************************************************/

#include <stdio.h>
#include <file2.h>
#include <register.h>
#include <assert.h>
#include <internal.h>


/***
*FILE *_fdopen(filedes, mode) - open a file descriptor as a stream
*
*Purpose:
*   associates a stream with a file handle, thus allowing buffering, etc.
*   The mode must be specified and must be compatible with the mode
*   the file was opened with in the low level open.
*
*   [NOTE: It is valid for this module to assign a value directly to _flag2
*   instead of simply twiddling bits since we are initializing the buffer
*   data base.]
*
*Entry:
*   int filedes - handle referring to open file
*   char *mode - file mode to use ("r", "w", "a", etc.)
*
*Exit:
*   returns stream pointer and sets FILE struct fields if successful
*   returns NULL if fails
*
*Exceptions:
*
*******************************************************************************/

FILE * 
_fdopen (filedes, mode)
const REG2 char *mode;
int filedes;
{
    REG1 FILE *stream;
    FILE2 *stream2;
    int whileflag, tbflag, cnflag;

    assert(mode != NULL);
#ifdef _QWIN
    assert(filedes < _wnfile);
#else
    assert(filedes < _nfile);
#endif

#ifdef _QWIN
    if (filedes < 0 || filedes >= _wnfile)
        return(NULL);
#else
    if (filedes < 0 || filedes >= _nfile)
        return(NULL);
#endif

    /* Find a free stream; stream is returned 'locked'. */

    if ((stream = _getstream()) == NULL)
        return(NULL);

    stream2 = &(_iob2_(stream));


    /* First character must be 'r', 'w', or 'a'. */

    switch (*mode) {
    case 'r':
        stream->_flag = _IOREAD;
        break;
    case 'w':
    case 'a':
        stream->_flag = _IOWRT;
        break;
    default:
        stream = NULL;  /* error */
        goto done;
        break;
    }

    /* There can be up to three more optional characters:
       (1) A single '+' character,
       (2) One of 't' and 'b' and
       (3) One of 'c' and 'n'.

       Note that currently, the 't' and 'b' flags are syntax checked
       but ignored.  'c' and 'n', however, are correctly supported.
    */

    whileflag=1;
    tbflag=cnflag=0;
    stream2->_flag2 = _commode;

    while(*++mode && whileflag)
        switch(*mode) {

        case '+':
            if (stream->_flag == _IORW)
                whileflag=0;
            else
                stream->_flag = (char)_IORW;
            break;

        case 'b':
        case 't':
            if (tbflag)
                whileflag=0;
            else
                tbflag=1;
            break;

        case 'c':
            if (cnflag)
                whileflag = 0;
            else {
                cnflag = 1;
                stream2->_flag2 |= _IOCOMMIT;
            }
            break;

        case 'n':
            if (cnflag)
                whileflag = 0;
            else {
                cnflag = 1;
                stream2->_flag2 &= ~_IOCOMMIT;
            }
            break;

        default:
            whileflag=0;
            break;
        }

    _cflush++;  /* force library pre-termination procedure */
    stream->_file = (char)filedes;

/* Common return */

done:
    return(stream);
}
