/***
*fflush.c - flush a stream buffer
*
*   Copyright (c) 1985-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   defines fflush() - flush the buffer on a stream
*       flushall() - flush all stream buffers
*
*******************************************************************************/

#include <stdio.h>
#include <register.h>
#include <file2.h>
#include <io.h>
#include <internal.h>
#include <dos.h>

/* Values passed to flsall() to distinguish between flushall() and
 * fflush(NULL) behavior
 */
#define FLUSHALL    1
#define FFLUSHNULL  0

/* Core routine for fflush(NULL) and flushall()
 */
static int near pascal flsall(int);

/***
*int fflush(stream) - flush the buffer on a stream
*
*Purpose:
*   if file open for writing and buffered, flush the buffer. if problems
*   flushing the buffer, set the stream flag to error
*   Always flushes the stdio stream and forces a commit to disk if file
*   was opened in commit mode.
*
*Entry:
*   FILE *stream - stream to flush
*
*Exit:
*   returns 0 if flushed successfully, or no buffer to flush
*   returns EOF and sets file error flag if fails.
*   FILE struct entries affected: _ptr, _cnt, _flag.
*
*Exceptions:
*
*******************************************************************************/


int  fflush (
    FILE *str
    )
{


    REG1 FILE2 _NEAR_ *stream2;

    /* if stream is NULL, flush all streams */
    if ( str == NULL ) {
        return(flsall(FFLUSHNULL));
    }

    if (_flush(str) != 0) {
        /* _flush failed, don't attempt to commit */
        return(EOF);
    }

    /* lowio commit to ensure data is written to disk */
    stream2 = &(_iob2_((FILE _NEAR_ *)FP_OFF(str)));
    if (stream2->_flag2 & _IOCOMMIT) {
        return (_commit(_fileno(str)) ? EOF : 0);
    }
}


/***
*int _flush(stream) - flush the buffer on a single stream
*
*Purpose:
*   if file open for writing and buffered, flush the buffer. if problems
*   flushing the buffer, set the stream flag to error
*   Multi-thread version assumes stream lock is held by caller
*
*Entry:
*   FILE *stream - stream to flush
*
*Exit:
*   returns 0 if flushed successfully, or no buffer to flush
*   returns EOF and sets file error flag if fails.
*   FILE struct entries affected: _ptr, _cnt, _flag.
*
*Exceptions:
*
*******************************************************************************/

int near _flush (
    FILE *str
    )
{
    REG1 FILE _NEAR_ *stream;
    REG2 int rc = 0; /* assume good return */
    REG3 int nchar;

    /* Init near pointer to stream */
    stream = (FILE _NEAR_ *) FP_OFF(str);

    if ((stream->_flag & (_IOREAD | _IOWRT)) == _IOWRT && bigbuf(stream)
        && (nchar = stream->_ptr - stream->_base) > 0)
    {
        if ( _write(_fileno(stream), stream->_base, nchar) == nchar ) {
            /* if this is a read/write file, clear _IOWRT so that
             * next operation can be a read
             */
            if ( _IORW & stream->_flag )
                stream->_flag &= ~_IOWRT;
        }
        else {
            stream->_flag |= _IOERR;
            rc = EOF;
        }
    }

    stream->_ptr = stream->_base;
    stream->_cnt = 0;

    return(rc);
}


/***
*int _flushall() - flush all output buffers
*
*Purpose:
*   flushes all the output buffers to the file, clears all input buffers.
*
*Entry:
*   None.
*
*Exit:
*   returns number of open streams
*
*Exceptions:
*
*******************************************************************************/

int  _flushall (
    void
    )
{
    return(flsall(FLUSHALL));
}


/***
*static int near pascal flsall(flushflag) - flush all output buffers
*
*Purpose:
*   Flushes all the output buffers to the file and, if FLUSHALL is passed,
*   clears all input buffers. Core routine for both fflush(NULL) and
*   flushall().
*
*   MTHREAD Note: All the locking/unlocking required for both fflush(NULL)
*   and flushall() is performed in this routine.
*
*Entry:
*   int flushflag - flag indicating the exact semantics, there are two
*           legal values: FLUSHALL and FFLUSHNULL
*
*Exit:
*   if flushflag == FFLUSHNULL then flsbuf returns:
*       0, if successful
*       EOF, if an error occurs while flushing one of the streams
*
*   if flushflag == FLUSHALL then flsbuf returns the number of streams
*   successfully flushed
*
*Exceptions:
*
*******************************************************************************/

static int near pascal flsall (
    int flushflag
    )
{
    REG1 FILE _NEAR_ *stream = _iob;
    REG2 int count = 0;
    int errcode = 0;


    for (; stream <= _lastiob; stream++) {


        if ( (flushflag == FLUSHALL) && inuse(stream) ) {
            /* FLUSHALL functionality: fflush the read or write
             * stream and, if successful, update the count of
             * flushed streams
             */
            if ( fflush(stream) != EOF )
                /* update count of successfully flushed
                 * streams
                 */
                count++;
        }
        else if ( (flushflag == FFLUSHNULL) &&
        (stream->_flag & _IOWRT) ) {
            /* FFLUSHNULL functionality: fflush the write stream
             * and kept track of the error, if one occurs
             */
            if ( fflush(stream) == EOF )
                errcode = EOF;
        }


    }


    if ( flushflag == FLUSHALL )
        return(count);
    else
        return(errcode);
}
