/***
*fwopen.c - open a window stream
*
*   Copyright (c) 1990-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines _fwopen() which opens a window for stream i/o.
*
*******************************************************************************/

#include <assert.h>
#include <fcntl.h>
#include <file2.h>
#include <io.h>
#include <stdio.h>
#include <stdlib.h>
#include <internal.h>


/***
*_fwopen - open a windows stream
*
*Purpose:
*   Opens a new windows stream with the specified window characteristics
*   and access modes.
*
*Entry:
*   struct * wopeninfo = pointer to QWIN open structure, may be NULL
*   struct * wsizeinfo = pointer to QWIN size structure, may be NULL
*   char *mode - mode of file access
*
*Exit:
*   success= pointer to stream
*   error = NULL
*
*Exceptions:
*
*******************************************************************************/

FILE *  _fwopen (wopeninfo, wsizeinfo, mode)
struct _wopeninfo  * wopeninfo;
struct _wsizeinfo  * wsizeinfo;
const char * mode;
{

    FILE * stream;
    FILE * retval = NULL;       // set up for error
    FILE2 * stream2;
    int fh;
    char streamflag;
    char stream2flag = _commode;    // init to default
    int commodeset;
    int modeflag;
    int whileflag;

    struct _wopeninfo  * wp;
    struct _wopeninfo wopeninfo2;
    char title[12] = "Stream "; // template for title
    char *numstring = &title[7];    // points after "Stream " in title

    assert(mode != NULL);
    assert(*mode != '\0');

    /*
     * force library pre-termination procedure
     */

    _cflush++;

    /*
     * Get a free stream
     * [NOTE: _getstream() returns a locked stream.]
     */

    if ((stream = _getstream()) == NULL)
        return(NULL);


    /*
     *
     * Parse the user's specification string as set flags in
     *
     * (1) modeflag - lowio flags word
     * (2) streamflag - stream handle flags word.
     * (3) stream2flag - stream handle (FILE2) flags word.
     *
     * [Don't set _O_CREATE/_O_TRUNC/_O_APPEND/_O_EXCL since
     * they don't make sense in windows and _wopen() will return
     * errors for some of these.]
     */

    /* First mode character must be 'r', 'w', or 'a'. */

    switch (*mode) {
    case 'r':
        modeflag = _O_RDONLY;
        streamflag = _IOREAD;
        break;
    case 'w':
        modeflag = _O_WRONLY;
        streamflag = _IOWRT;
        break;
    case 'a':
        modeflag = _O_WRONLY;
        streamflag = _IOWRT;
        break;
    default:
        return(NULL);
        break;
    }

    /* There can be up to three more optional mode characters:
       (1) A single '+' character,
       (2) One of 't' and 'b' and
       (3) One of 'c' and 'n'.
    */

    whileflag=1;
    commodeset=0;

    while(*++mode && whileflag)
        switch(*mode) {

        case '+':
            if (modeflag & _O_RDWR)
                whileflag=0;
            else {
                modeflag |= _O_RDWR;
                modeflag &= ~(_O_RDONLY | _O_WRONLY);
                streamflag = _IORW;
            }
            break;

        case 't':
            if (modeflag & (_O_TEXT | _O_BINARY))
                whileflag=0;
            else
                modeflag |= _O_TEXT;
            break;

        case 'b':
            if (modeflag & (_O_TEXT | _O_BINARY))
                whileflag=0;
            else
                modeflag |= _O_BINARY;
            break;

        case 'c':
            if (commodeset)
                whileflag=0;
            else {
                commodeset = 1;
                stream2flag |= _IOCOMMIT;
            }
            break;

        case 'n':
            if (commodeset)
                whileflag=0;
            else {
                commodeset = 1;
                stream2flag &= ~_IOCOMMIT;
            }
            break;

        default:
            whileflag=0;
            break;
        }

    /*
     * windows structure
     */

    if (wopeninfo == NULL) {

        /*
         * Set up a default windows structure
         */

        wopeninfo2._version = _QWINVER;
        wopeninfo2._wbufsize = _WINBUFDEF;
        wopeninfo2._title = (char far *) title;

        /*
         * Set title equal to index into _iob table
         * [Note: We depend on title string being "Stream " above!]
         */

        _itoa (_iob_index(stream), numstring, 10);

        wp = &wopeninfo2;
        }

    else    {
        wp = wopeninfo;
        }

    /*
     * open the stream
     */

    if ( (fh = _wopen (wp, wsizeinfo, modeflag)) == -1)
        goto done;

    /*
     * Init stream
     */

    stream2 = &(_iob2_(stream));

    stream->_flag = streamflag;
    stream->_cnt = 0;
    stream->_base = stream->_ptr = NULL;
    stream->_file = (char)fh;

    stream2->_flag2 = stream2flag;
    stream2->__tmpnum = 0;

    retval = stream;    // good return

    /*
     * unlock stream and return.
     */

done:
    return(retval);
}
