/***
*setvbuf.c - set buffer size for a stream
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines setvbuf() - set the buffering mode and size for a stream.
*
*******************************************************************************/

#include <stdio.h>
#include <register.h>
#include <file2.h>
#include <assert.h>
#include <malloc.h>
#include <internal.h>
#include <dos.h>
#include <limits.h>


/***
*int setvbuf(stream, buffer, type, size) - set buffering for a file
*
*Purpose:
*   Controls buffering and buffer size for the specified stream.  The
*   array pointed to by buf is used as a buffer, unless NULL, in which case
*   will allocate a buffer automatically.  type specifies the type
*   of buffering: _IONBF = no buffer, _IOFBF = buffered
*
*   [NOTE: It is valid for this module to assign a value directly to _flag2
*   instead of simply twiddling bits since we are initializing the buffer
*   data base.]
*
*Entry:
*   FILE *stream - stream to control buffer on
*   char *buffer - pointer to buffer to use (NULL means auto allocate)
*   int type - type of buffering (_IONBF or _IOFBF)
*   size_t size - size of buffer
*
*Exit:
*   return 0 if successful
*   returns non-zero if fails
*
*Exceptions:
*
*******************************************************************************/

int  setvbuf (str, buffer, type, size )
FILE *str;
char *buffer;
int type;
size_t size;
{
    REG1 FILE _NEAR_ *stream;
    REG2 FILE2 _NEAR_ *stream2;
    int retval=0;   /* assume good return */

    assert(str != NULL);

    /* (1) Make sure type is one of the three legal values.
       (2) If we are buffering, make sure size is greater than 0.
    */

    if ( type != _IONBF )
        if ( ((size==0) || (size>INT_MAX))
            || ((type!=_IOFBF) && (type!=_IOLBF)) )
            return(-1);

    /* Init stream pointers */
    stream = (FILE _NEAR_ *) FP_OFF(str);
    stream2 = &(_iob2_(stream));

    /* Lock the file */

    /* Flush the current buffer and free it, if it is ours. */
    _flush(stream);
    _freebuf(stream);

    /* CASE 1: No Buffering. */

    if (type & _IONBF) {
        stream->_flag |= _IONBF;
        stream2->_flag2 = 0;
        buffer = &(stream2->_charbuf);
        size = 1;
    }

    /* CASE 2: Default Buffering -- Allocate a buffer for the user. */

    else if ( !buffer ) {
        if ((buffer=malloc(size)) == NULL) {
            retval = -1;
            goto done;
        }
        stream->_flag &= ~(_IONBF);
        stream->_flag |= (_IOMYBUF);
        stream2->_flag2 = 0;
    }

    /* CASE 3: User Buffering -- Use the buffer supplied by the user. */
    else {
        _cflush++;  /* force library pre-termination procedure */
        stream->_flag &= ~(_IOMYBUF | _IONBF);
        stream2->_flag2 = _IOYOURBUF;
    }

    /* Common return for all cases. */

    stream2->_bufsiz = size;
    stream->_ptr = stream->_base = buffer;
    stream->_cnt = 0;
done:
    return(retval);
}
