/***
*tempnam.c - generate unique file name
*
*   Copyright (c) 1986-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*
*******************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <errno.h>
#include <malloc.h>
#include <string.h>
#include <register.h>
#include <internal.h>

#ifdef _MBCS
#include <mbstring.h>
#endif


/***
*char *_tempnam(dir, prefix) - create unique file name
*
*Purpose:
*   Create a file name that is unique in the specified directory.
*   The semantics of directory specification is as follows:
*   Use the directory specified by the TMP environment variable
*   if that exists, else use the dir argument if non-NULL, else
*   use P_tmpdir if that directory exists, else use the current
*   working directory), else return NULL.
*
*Entry:
*   char *dir - directory to be used for temp file if TMP env var
*       not set
*   char *prefix - user provided prefix for temp file name
*
*Exit:
*   returns ptr to constructed file name if successful
*   returns NULL if unsuccessful
*
*Exceptions:
*
*******************************************************************************/

char *  _tempnam( dir, pfx )
char *dir;
char *pfx;
{
    REG1 char *ptr;
    REG2 unsigned int pfxlength=0;
    char *s;
    char *pfin;
    unsigned int first;


    /* try TMP path */
    if ( ( ptr = getenv( "TMP" ) ) && ( _access( ptr, 0 ) != -1 ) )
    dir = ptr;


    /* TMP path not available, use alternatives */
    else if (!( dir != NULL && ( _access( dir, 0 ) != -1 ) ) )
    /* do not "simplify" this depends on side effects!! */
    {
    if ( _access( P_tmpdir, 0 ) != -1 )
        dir = P_tmpdir;
    else
        dir = ".";
    }

    if (pfx)
    pfxlength = strlen(pfx);
    if ( ( s = malloc(strlen(dir) + pfxlength + 8 ) ) == NULL )
    /* the 8 above allows for a backslash, 6 char temp string and
       a null terminator */
    {
        goto done2;     /* error return */
    }
    *s = '\0';
    strcat( s, dir );
    pfin = &(dir[ strlen( dir ) - 1 ]);
#ifdef _MBCS
    if (*pfin == '\\') {
    if (pfin != _mbsrchr(s,'\\'))
        /* *pfin is second byte of a double-byte char */
        strcat( s, "\\" );
    }
    else if (*pfin != '/')
    strcat( s, "\\" );
#else
    if ( ( *pfin != '\\' ) && ( *pfin != '/' ) )
    {
    strcat( s, "\\" );
    }
#endif
    if ( pfx != NULL )
    {
    strcat( s, pfx );
    }
    ptr = &s[strlen( s )];

    /*
    Re-initialize _tempoff if necessary.  If we don't re-init _tempoff, we
    can get into an infinate loop (e.g., (a) _tempoff is a big number on entry,
    (b) prefix is a long string (e.g., 8 chars) and all tempfiles with that
    prefix exist, (c) _tempoff will never equal first and we'll loop forever).

    [NOTE: To avoid a conflict that causes the same bug as that discussed
    above, tempnam() uses _tempoff; tmpnam() uses _tmpoff]
    */


    if (_old_pfxlen < pfxlength)
    _tempoff = 1;
    _old_pfxlen = pfxlength;

    first = _tempoff;

    do {
    if ( ++_tempoff == first ) {
        free(s);
        s = NULL;
        goto done1;     /* error return */
    }
    _itoa( _tempoff, ptr, 10 );
    if ( strlen( ptr ) + pfxlength > 8 )
    {
        *ptr = '\0';
        _tempoff = 0;
    }
    }
    while ( (_access( s, 0 ) == 0 ) || (errno == EACCES) );

    /* Common return */
done1:
done2:
    return(s);
}


