/***
*input.c - C formatted input, used by scanf, etc.
*
*   Copyright (c) 1987-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   defines _input() to do formatted input; called from scanf(),
*   etc. functions.  This module defines cscanf() instead when
*   CPRFLAG is defined -- see makefile in lattice directory.
*
*******************************************************************************/

#define ALLOW_RANGE /* allow "%[a-z]"-style scansets */

#include <sizeptr.h>
#include <stdio.h>
#include <string.h>
#include <register.h>
#include <assert.h>
#include <ctype.h>
#include <cvt.h>
#include <conio.h>
#include <stdarg.h>
#include <fltintrn.h>


#define HEXTODEC(chr)   _hextodec(chr)
#define LEFT_BRACKET    ('[' | ('a' - 'A')) /* 'lowercase' version */

static int near pascal _hextodec(int);


#ifdef CPRFLAG

#define INC()       (++charcount, _inc())
#define UN_INC(chr) (--charcount, _ungetch(chr))
#define EAT_WHITE() _whiteout(&charcount)

static int near pascal _inc(void);
static int near pascal _whiteout(int *);

#else

#define INC()       (++charcount, _inc(stream))
#define UN_INC(chr) (--charcount, _un_inc(chr, stream))
#define EAT_WHITE() _whiteout(&charcount, stream)

static int near pascal _inc(FILE *);
static void near pascal _un_inc(int, FILE *);
static int near pascal _whiteout(int *, FILE *);

#endif


#ifdef CPRFLAG
static int input(const unsigned char *, va_list);


/***
*int _cscanf(format, arglist) - read formatted input direct from console
*
*Purpose:
*   Reads formatted data like scanf, but uses console I/O functions.
*
*Entry:
*   char *format - format string to determine data formats
*   arglist - list of POINTERS to where to put data
*
*Exit:
*   returns number of successfully matched data items (from input)
*
*Exceptions:
*
*******************************************************************************/


int 
_cscanf (const char *format, ...)

{
va_list arglist;

va_start(arglist, format);

assert(format != NULL);

return input(format,arglist);   /* get the input */
}

#endif  /* CPRFLAG */


#define ASCII       32       /* # of bytes needed to hold 256 bits */

#define SCAN_SHORT     0     /* also for FLOAT */
#define SCAN_LONG      1     /* also for DOUBLE */
#define SCAN_L_DOUBLE  2     /* only for LONG DOUBLE */

#define SCAN_NEAR    0
#define SCAN_FAR     1

#ifdef ALLOW_RANGE

static char sbrackset[] = " \t-\r]"; /* use range-style list */

#else

static char sbrackset[] = " \t\n\v\f\r]"; /* chars defined by isspace() */

#endif

static char cbrackset[] = "]";



/***
*int _input(stream, format, arglist), static int input(format, arglist)
*
*Purpose:
*   get input items (data items or literal matches) from the input stream
*   and assign them if appropriate to the items thru the arglist. this
*   function is intended for internal library use only, not for the user
*
*   The _input entry point is for the normal scanf() functions
*   The input entry point is used when compiling for cscanf() [CPRFLAF defined]
*   and is a static function called only by cscanf() -- reads from console.
*
*Entry:
*   FILE *stream - file to read from
*   char *format - format string to determine the data to read
*   arglist - list of pointer to data items
*
*Exit:
*   returns number of items assigned and fills in data items
*   returns EOF if error or EOF found on stream before 1st data item matched
*
*Exceptions:
*
*******************************************************************************/

#pragma check_stack(on) /* large auto vars -- put stack checking on */

#ifdef CPRFLAG

static int
input (format, arglist)
const unsigned char *format;
va_list arglist;

#else

int
_input (stream, format, arglist)
FILE *stream;
const unsigned char *format;
va_list arglist;

#endif

{
    char table[ASCII];          /* which chars allowed for %[], %s   */
    char floatstring[CVTBUFSIZE + 1];   /* ASCII buffer for floats       */

    unsigned long number;       /* temp hold-value           */
    void far *pointer;          /* points to user data receptacle    */
    void far *start;            /* indicate non-empty string         */

    unsigned char *scanptr;     /* for building "table" data         */

REG2 int ch;
    int charcount;          /* total number of chars read        */
REG1 int comchr;            /* holds designator type         */
    int count;              /* return value.  # of assignments   */
    unsigned int hold_seg;      /* for LONG (far) %p objects         */
    int started;            /* indicate good number          */
    int width;              /* width of field            */
    int widthset;           /* user has specified width      */

#ifdef SIZED
    char coerceshort;           /* only used for %hp in large model  */
#else
    char farone;            /* 0 = NEAR, 1 = FAR for small model */
#endif

#ifdef ALLOW_RANGE
    unsigned char rngch;        /* used while scanning range         */
#endif

    char done_flag;         /* general purpose loop monitor      */
    unsigned char last;         /* also for %[a-z]           */
    char longone;           /* 0 = SHORT, 1 = LONG, 2 = L_DOUBLE */
    char negative;          /* flag for '-' detected         */
    unsigned char prevchar;     /* for %[a-z]                */
    char reject;            /* %[^ABC] instead of %[ABC]         */
    char suppress;          /* don't assign anything             */
    char match;             /* flag: !0 if any fields matched    */


    assert(format != NULL);

#ifndef CPRFLAG
    assert(stream != NULL);
#endif

    /*
    count = # fields assigned
    charcount = # chars read
    match = flag indicating if any fields were matched

    [Note that we need both count and match.  For example, a field
    may match a format but have assignments suppressed.  In this case,
    match will get set, but 'count' will still equal 0.  We need to
    distinguish 'match vs no-match' when terminating due to EOF.]
    */

    count = charcount = match = 0;

    while (*format) {

    if (isspace(*format)) {

        UN_INC(EAT_WHITE()); /* put first non-space char back */

        while (isspace(*++format)); /* NULL */

    }

    if ('%' == *format) {

        number = width = widthset = started = done_flag =
        suppress = longone = negative = reject = prevchar =
#ifdef SIZED
        coerceshort = 0;
#else
        farone = 0;
#endif


        while (!done_flag) {

        if (isdigit(comchr = *++format)) {
            ++widthset;
            width = MUL10(width) + (comchr - '0');
        } else
            switch (comchr) {
            case 'F':
#ifndef SIZED
                ++farone;
#endif
            case 'N':   /* no way to push NEAR in large model */
                break;  /* NEAR is default in small model */
            case 'h':
                /* longone is already 0 */
#ifdef SIZED
                ++coerceshort;
#endif
                break;
            case 'L':
                ++longone;
                    /* NOBREAK */
            case 'l':
                ++longone;
                break;
            case '*':
                ++suppress;
                break;

            default:
                ++done_flag;
                break;
            }
        }

        if (!suppress) {

#ifndef SIZED
        if (farone) {
            pointer = va_arg(arglist, void far *);
        } else {
            pointer = va_arg(arglist, void *);
        }

#else /* all pointers pushed as FAR in large model */

        pointer = va_arg(arglist, void *);

#endif
        }

        done_flag = 0;

        /* switch to lowercase to allow %E,%G, and to
           keep the switch table small */

        if ('n' != (comchr = (*format | ('a' - 'A'))))
        if ('c' != comchr && LEFT_BRACKET != comchr)
            ch = EAT_WHITE();
        else
            ch = INC();

        if (!widthset || width) {

        switch(comchr) {

            case 'c':
            if (!widthset) {
                ++widthset;
                ++width;
            }
            scanptr = cbrackset;
            --reject; /* set reject to 255 */
            goto scanit2;

            case 's':
            scanptr = sbrackset;
            --reject; /* set reject to 255 */
            goto scanit2;

            case LEFT_BRACKET :   /* scanset */
            scanptr = (char *)(++format);

            if ('^' == *scanptr) {
                ++scanptr;
                --reject; /* set reject to 255 */
            }


scanit2:
            memset(table, 0, ASCII);

#ifdef ALLOW_RANGE

            if (LEFT_BRACKET == comchr)
                if (']' == *scanptr) {
                prevchar = ']';
                ++scanptr;

                table[ ']' >> 3] = 1 << (']' & 7);

                }

            while (']' != *scanptr) {

                rngch = *scanptr++;

                if ('-' != rngch ||
                 !prevchar ||       /* first char */
                 ']' == *scanptr) /* last char */

                table[(prevchar = rngch) >> 3] |= 1 << (rngch & 7);

                else {  /* handle a-z type set */

                rngch = *scanptr++; /* get end of range */

                if (prevchar < rngch)  /* %[a-z] */
                    last = rngch;
                else {          /* %[z-a] */
                    last = prevchar;
                    prevchar = rngch;
                }
                for (rngch = prevchar; rngch <= last; ++rngch)
                    table[rngch >> 3] |= 1 << (rngch & 7);

                prevchar = 0;

                }
            }


#else
            if (LEFT_BRACKET == comchr)
                if (']' == *scanptr) {
                ++scanptr;
                table[(prevchar = ']') >> 3] |= 1 << (']' & 7);
                }

            while (']' != *scanptr)
                table[scanptr >> 3] |= 1 << (scanptr & 7);

#endif
            if (!*scanptr)
                goto error_return;      /* trunc'd format string */

            /* scanset completed.  Now read string */

            if (LEFT_BRACKET == comchr)
                format = scanptr;

            start = pointer;

            /*
             * execute the format directive. that is, scan input
             * characters until the directive is fulfilled, eof
             * is reached, or a non-matching character is
             * encountered.
             *
             * it is important not to get the next character
             * unless that character needs to be tested! other-
             * wise, reads from line-buffered devices (e.g.,
             * scanf()) would require an extra, spurious, newline
             * if the first newline completes the current format
             * directive.
             */
            UN_INC(ch);

            while ( !widthset || width-- ) {

                ch = INC();

                if (
#ifndef CPRFLAG
                  (EOF != ch) &&
#endif
                  ((table[ch >> 3] ^ reject) & (1 << (ch & 7)))
                ) {
                if (!suppress) {
                    *(char far *)pointer = (char)ch;
                    ++(char far *)pointer;
                }
                else {
                    /* just indicate a match */
                    ++(char far *)start;
                }
                }
                else  {
                UN_INC(ch);
                break;
                }
            }

            /* make sure something has been matched and, if
               assignment is not suppressed, null-terminate
               output string if comchr != c */

            if (start != pointer) {
                if (!suppress) {
                ++count;
                if ('c' != comchr) /* null-terminate strings */
                    *(char far *)pointer = '\0';
                } else /*NULL*/;
            }
            else
                goto error_return;

            break;

            case 'i':   /* could be d, o, or x */

            comchr = 'd'; /* use as default */

            case 'x':

            if ('-' == ch) {
                ++negative;

                goto x_incwidth;

            } else if ('+' == ch) {
x_incwidth:
                if (!--width && widthset)
                ++done_flag;
                else
                ch = INC();
            }

            if ('0' == ch) {

                if ('x' == ((char)(ch = INC())) || 'X' == (char)ch) {
                ch = INC();
                comchr = 'x';
                } else {
                ++started;
                if ('x' != comchr)
                    comchr = 'o';
                else {
                    /* scanning a hex number that starts */
                    /* with a 0. push back the character */
                    /* currently in ch and restore the 0 */
                    UN_INC(ch);
                    ch = '0';
                }
                }
            }
            goto getnum;

            /* NOTREACHED */

            case 'p':

#ifndef SIZED   /* in large data model, ALL pointers are LONG (far) */
            if (longone) {
#else
            if (!coerceshort) {
                ++longone;
#endif
                comchr = 'F';   /* indicates long */
            }

            case 'o':
            case 'u':
            case 'd':

            if ('-' == ch) {
                ++negative;

                goto d_incwidth;

            } else if ('+' == ch) {
d_incwidth:
                if (!--width && widthset)
                ++done_flag;
                else
                ch = INC();
            }

getnum:
            while (!done_flag) {

                if ('x' == comchr || 'p' == comchr || 'F' == comchr)

                if (isxdigit(ch)) {

                    number = (number << 4);
                    ch = HEXTODEC(ch);

                } else if ('F' == comchr)

                    if (started)
                    if (':' == ch) {
                        hold_seg = (unsigned int)number;
                        number = 0;
                        started = -1;
                        comchr = 'p'; /* switch to offset */
                        ch = '0';     /* don't add ':' */
                    } else {
                        started = 0;
                        ++done_flag;
                    }
                    else
                    ++done_flag;
                else

                    ++done_flag;

                else if (isdigit(ch))

                if ('o' == comchr)
                    if ('8' > ch)
                    number = (number << 3);
                    else {
                     ++done_flag;
                    }
                else /* 'd' == comchr */
                    number = MUL10(number);

                else
                ++done_flag;

                if (!done_flag) {
                ++started;
                number += ch - '0';

                if (widthset && !--width)
                    ++done_flag;
                else
                    ch = INC();
                } else
                UN_INC(ch);

            } /* end of WHILE loop */

            if ('p' == comchr && longone)
                number = (number & 0x0000FFFF) | ((unsigned long)hold_seg) << 16;

            if (negative)
                number = -number;

            if ('F' == comchr)  /* expected a ':' in long pointer */
                started = 0;

            if (started)
                if (!suppress) {

                ++count;
assign_num:
                if (longone)
                    *(long far *)pointer = (unsigned long)number;
                else
                    *(int far *)pointer = (unsigned int)number;

                } else /*NULL*/;
            else
                goto error_return;

            break;

            case 'n':   /* char count, don't inc return value */
            number = charcount;
            goto assign_num; /* found in number code above */


            case 'e':
            case 'f':
            case 'g': /* scan a float */

            scanptr = floatstring;

            if ('-' == ch) {
                *scanptr++ = '-';
                goto f_incwidth;

            } else if ('+' == ch) {
f_incwidth:
                --width;
                ch = INC();
            }

            if (!widthset || width > CVTBUFSIZE)          /* must watch width */
                width = CVTBUFSIZE;


            /* now get integral part */

            while (isdigit(ch) && width--) {
                ++started;
                *scanptr++ = (char)ch;
                ch = INC();
            }

            /* now check for decimal */

            if ('.' == ch && width--) {
                *scanptr++ = '.';
                ch = INC();

                while (isdigit(ch) && width--) {
                ++started;
                *scanptr++ = (char)ch;
                ch = INC();
                }
            }

            /* now check for exponent */

            if (started && ('e' == (char)ch || 'E' == (char)ch) && width--) {
                *scanptr++ = 'e';

                if ('-' == (ch = INC())) {

                *scanptr++ = '-';
                goto f_incwidth2;

                } else if ('+' == ch) {
f_incwidth2:
                if (!width--)
                    ++width;
                else
                    ch = INC();
                }


                while (isdigit(ch) && width--) {
                ++started;
                *scanptr++ = (char)ch;
                ch = INC();
                }

            }

            UN_INC(ch);

            if (started)
                if (!suppress) {
                ++count;
                *scanptr = '\0';
                _fassign( longone, pointer , floatstring);
                } else /*NULL */;
            else
                goto error_return;

            break;


            default:    /* either found '%' or something else */

            if ((int)*format != ch) {
                UN_INC(ch);
                goto error_return;
                }
            else
                match--; /* % found, compensate for inc below */

            if (!suppress)

#ifndef SIZED
                if (farone)
                arglist = (va_list)((void far **)arglist - 1);
                else
                arglist = (va_list)((void **)arglist - 1);
#else /* only one size pointer in large data model programs */
                arglist = (va_list)((void **)arglist - 1);
#endif

        } /* SWITCH */

        match++;    /* matched a format field - set flag */

        } /* WHILE (width) */

        else {  /* zero-width field in format string */
        UN_INC(ch);  /* check for input error */
        goto error_return;
        }

        ++format;  /* skip to next char */

    } else  /*  ('%' != *format) */
        {

        if ((int)*format++ != (ch = INC()))
        {
        UN_INC(ch);
        goto error_return;
        }
        if (_isleadbyte(ch))
        {
        if ((int)*format++ != (ch=INC()))
            {
            UN_INC(ch);
            UN_INC(ch);
            goto error_return;
            }

/* CONSIDER: is this needed? */
/*      --charcount;  */ /* only count as one character read */
        }
        }


#ifndef CPRFLAG
    if ( (EOF == ch) && ((*format != '%') || (*(format + 1) != 'n')) )
        break;
#endif

    }  /* WHILE (*format) */

error_return:

#ifndef CPRFLAG
    if (EOF == ch)
    /* If any fields were matched or assigned, return count */
    return ( (count || match) ? count : EOF);
    else
#endif
    return count;

}

#pragma check_stack(off)    /* don't bother checking for one local */

/* _hextodec() returns a value of 0-15 and expects a char 0-9, a-f, A-F */
/* _inc() is the one place where we put the actual getc code. */
/* _whiteout() returns the first non-blank character, as defined by isspace() */

static int near pascal _hextodec(chr)
int chr;
{
   return isdigit(chr) ? chr : (chr & ~('a' - 'A')) - 'A' + 10 + '0';
}


#ifdef CPRFLAG

static int near pascal _inc(void)
{
    return(_getche());
}

static int near pascal _whiteout(counter)
REG1 int *counter;
{
    REG2 int ch;

    while(isspace(ch = (++*counter, _inc())));
    return ch;
}

#else

static int near pascal _inc(fileptr)
REG1 FILE *fileptr;
{
    return(getc(fileptr));
}

static void near pascal _un_inc(chr, fileptr)
int chr;
FILE *fileptr;
{
    if (EOF != chr)
    ungetc(chr, fileptr);
}

static int near pascal _whiteout(counter, fileptr)
REG1 int *counter;
REG3 FILE *fileptr;
{
    REG2 int ch;

    while(isspace(ch = (++*counter, _inc(fileptr))));
    return ch;
}

#endif
