/***
*_open.c - open a stream, with string mode
*
*   Copyright (c) 1985-1992, Microsoft Corporation.  All rights reserved.
*
*Purpose:
*   defines _openfile() - opens a stream, with string arguments for mode
*
*******************************************************************************/

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <register.h>
#include <file2.h>
#include <assert.h>
#include <internal.h>
#include <dos.h>

#define CMASK   0644    /* rw-r--r-- */


/***
*FILE *_openfile(filename, mode, shflag, stream) - open a file with string
*   mode and file sharing flag.
*
*Purpose:
*   parse the string, looking for exactly one of {rwa}, at most one '+',
*   at most one of {tb} and at most one of {cn}. pass the result on as
*   an int containing flags of what was found. open a file with proper
*   mode if permissions allow. buffer not allocated until first i/o call
*   is issued. intended for use inside library only
*
*   [NOTE: It is valid for this module to assign a value directly to
*   _flag2 instead of simply twiddling bits since we are initializing
*   the buffer data base.]
*
*Entry:
*   char *filename - file to open
*   char *mode - mode to use (see above)
*   int shflag - file sharing flag
*   FILE *stream - stream to use for file
*
*Exit:
*   set stream's fields, and causes system file management by system calls
*   returns stream or NULL if fails
*
*Exceptions:
*
*******************************************************************************/

FILE *
_openfile (filename, mode, shflag, str)
const char *filename;
REG3 const char *mode;
int shflag;
FILE *str;
{
    REG2 int modeflag;
    char streamflag;
    char stream2flag = _commode;
    int commodeset = 0;
    int whileflag;
    int filedes;
    REG1 FILE _NEAR_ *stream;
    FILE2 _NEAR_ *stream2;

    assert(filename != NULL);
    assert(mode != NULL);
    assert(str != NULL);

    /* Parse the user's specification string as set flags in
       (1) modeflag - system call flags word
       (2) streamflag - stream handle flags word.
       (3) stream2flag - stream handle (FILE2) flags word. */

    /* First mode character must be 'r', 'w', or 'a'. */

    switch (*mode) {
    case 'r':
        modeflag = _O_RDONLY;
        streamflag = _IOREAD;
        break;
    case 'w':
        modeflag = _O_WRONLY | _O_CREAT | _O_TRUNC;
        streamflag = _IOWRT;
        break;
    case 'a':
        modeflag = _O_WRONLY | _O_CREAT | _O_APPEND;
        streamflag = _IOWRT;
        break;
    default:
        return(NULL);
        break;
    }

    /* There can be up to three more optional mode characters:
       (1) A single '+' character,
       (2) One of 't' and 'b' and
       (3) One of 'c' and 'n'.
    */

    whileflag=1;

    while(*++mode && whileflag)
        switch(*mode) {

        case '+':
            if (modeflag & _O_RDWR)
                whileflag=0;
            else {
                modeflag |= _O_RDWR;
                modeflag &= ~(_O_RDONLY | _O_WRONLY);
                streamflag = (char)_IORW;
            }
            break;
        case 't':
            if (modeflag & (_O_TEXT | _O_BINARY))
                whileflag=0;
            else
                modeflag |= _O_TEXT;
            break;

        case 'b':
            if (modeflag & (_O_TEXT | _O_BINARY))
                whileflag=0;
            else
                modeflag |= _O_BINARY;
            break;

        case 'c':
            if (commodeset)
                whileflag=0;
            else {
                commodeset = 1;
                stream2flag |= _IOCOMMIT;
            }
            break;

        case 'n':
            if (commodeset)
                whileflag=0;
            else {
                commodeset = 1;
                stream2flag &= ~_IOCOMMIT;
            }
            break;
        default:
            whileflag=0;
            break;
        }

    /* Try to open the file.  Note that if neither 't' nor 'b' is
       specified, sopen will use the default. */

    if ((filedes = _sopen(filename, modeflag, shflag, CMASK)) < 0)
        return(NULL);

    /* Set up the stream data base. */

    _cflush++;  /* force library pre-termination procedure */

    /* Init pointers */
    stream = (FILE _NEAR_ *) FP_OFF(str);
    stream2 = &(_iob2_(stream));

    stream->_flag = streamflag;
    stream->_cnt = 0;
    stream->_base = stream->_ptr = NULL;
    stream->_file = (char)filedes;

    stream2->_flag2 = stream2flag;
    stream2->__tmpnum = 0;

    return(stream);
}
