	page	,132
	title	memcpy - Copy source memory bytes to destination
;***
;memcpy.asm - contains memcpy and memmove routines
;
;	Copyright (c) 1986-1992, Microsoft Corporation.  All right reserved.
;
;Purpose:
;	memcpy() copies a source memory buffer to a destination buffer.
;	Overlapping buffers are not treated specially, so propogation may occur.
;	memmove() copies a source memory buffer to a destination buffer.
;	Overlapping buffers are treated specially, to avoid propogation.
;
;*******************************************************************************

	.xlist
	include version.inc
	include cmacros.inc
	.list

sBegin	code
	assumes cs,code
	assumes ds,data

page
;***
;memcpy - Copy source buffer to destination buffer
;
;Purpose:
;	memcpy() copies a source memory buffer to a destination memory buffer.
;	This routine does NOT recognize overlapping buffers, and thus can lead
;	to propogation.
;	For cases where propogation must be avoided, memmove() must be used.
;
;	Algorithm:
;
;	void * memcpy(void * dst, void * src, size_t count)
;	{
;		void * ret = dst;
;
;		/*
;		 * copy from lower addresses to higher addresses
;		 */
;		while (count--)
;			*dst++ = *src++;
;
;		return(ret);
;	}
;
;memmove - Copy source buffer to destination buffer
;
;Purpose:
;	memmove() copies a source memory buffer to a destination memory buffer.
;	This routine recognize overlapping buffers to avoid propogation.
;	For cases where propogation is not a problem, memcpy() can be used.
;
;   Algorithm:
;
;	void * memmove(void * dst, void * src, size_t count)
;	{
;		void * ret = dst;
;
;		if (dst <= src || dst >= (src + count)) {
;			/*
;			 * Non-Overlapping Buffers
;			 * copy from lower addresses to higher addresses
;			 */
;			while (count--)
;				*dst++ = *src++;
;			}
;		else {
;			/*
;			 * Overlapping Buffers
;			 * copy from higher addresses to lower addresses
;			 */
;			dst += count - 1;
;			src += count - 1;
;
;			while (count--)
;				*dst-- = *src--;
;			}
;
;		return(ret);
;	}
;
;
;Entry:
;	void *dst = pointer to destination buffer
;	const void *src = pointer to source buffer
;	size_t count = number of bytes to copy
;
;Exit:
;	Returns a pointer to the destination buffer in AX/DX:AX
;
;Uses:
;	BX, CX, DX
;
;Exceptions:
;*******************************************************************************

ifdef	MEM_MOVE
	ifdef	MODELINDEP

	cProc	_fmemmove,<PUBLIC>,<>

	else

	cProc	memmove,<PUBLIC>,<>

	endif
else
	ifdef	MODELINDEP

	cProc	_fmemcpy,<PUBLIC>,<>

	else

	cProc	memcpy,<PUBLIC>,<>

	endif
endif

	parmDP	dst		; destination pointer
	parmDP	src		; source pointer
	parmW	count		; number of bytes to copy

cBegin
	mov	dx,di		; Preserve SI and DI
	mov	bx,si

if	sizeD
	push	ds		; Preserve DS
	lds	si,src		; DS:SI = src
	les	di,dst		; ES:DI = dst
else
	mov	ax,ds
	mov	es,ax
	mov	si,src		; DS:SI = src
	mov	di,dst		; ES:DI = dst
endif

	mov	ax,di		; save dst in AX for return value
	mov	cx,count	; cx = number of bytes to move
	jcxz	done		; if cx = 0 Then nothing to copy

ifdef	MEM_MOVE
;
; Check for overlapping buffers:
;	If segments are different, assume no overlap
;		Do normal (Upwards) Copy
;	Else If (dst <= src) Or (dst >= src + Count) Then
;		Do normal (Upwards) Copy
;	Else
;		Do Downwards Copy to avoid propogation
;
if	sizeD
	mov	ax,es		; compare the segments
	cmp	ax,word ptr (src+2)
	jne	CopyUp
endif
	cmp	di,si		; src <= dst ?
	jbe	CopyUp

	mov	ax,si
	add	ax,cx
	cmp	di,ax		; dst >= (src + count) ?
	jae	CopyUp
;
; Copy Down to avoid propogation in overlapping buffers
;
	mov	ax,di		; AX = return value (offset part)

	add	si,cx
	add	di,cx
	dec	si		; DS:SI = src + count - 1
	dec	di		; ES:DI = dst + count - 1
	std			; Set Direction Flag = Down
	rep	movsb
	cld			; Set Direction Flag = Up
	jmp	short done

CopyUp:
	mov	ax,di		; AX = return value (offset part)
endif ; MEM_MOVE
;
; There are 4 situations as far as word alignment of "src" and "dst":
;	1. src and dst are both even	(best case)
;	2. src is even and dst is odd
;	3. src is odd and dst is even
;	4. src and dst are both odd	(worst case)
;
; Case #4 is much faster if a single byte is copied before the
; REP MOVSW instruction.  Cases #2 and #3 are effectively unaffected
; by such an operation.  To maximum the speed of this operation,
; only DST is checked for alignment.  For cases #2 and #4, the first
; byte will be copied before the REP MOVSW.
;
	test	al,1		; fast check for dst being odd address
	jz	move

	movsb			; move a byte to improve alignment
	dec	cx
;
; Now the bulk of the copy is done using REP MOVSW.  This is much
; faster than a REP MOVSB if the src and dst addresses are both
; word aligned and the processor has a 16-bit bus.  Depending on
; the initial alignment and the size of the region moved, there
; may be an extra byte left over to be moved.  This is handled
; by the REP MOVSB, which moves either 0 or 1 bytes.
;
move:
	shr	cx,1		; Shift CX for count of words
	rep	movsw		; CF set if one byte left over
	adc	cx,cx		; CX = 1 or 0, depending on Carry Flag
	rep	movsb		; possible final byte
;
; Return the "dst" address in AX/DX:AX
;
done:
	mov	si,bx
	mov	di,dx		; Restore SI and DI

if	sizeD
	pop	ds		;restore ds
	mov	dx,es		;segment part of dest address
endif
	pop	bp
	ret

cEnd	<nogen>

sEnd

	end
