/***
*clock.c - Contains the clock runtime
*
*   Copyright (c) 1987-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   The clock runtime returns the processor time used by
*   the current process.
*
*******************************************************************************/

#include <time.h>
#include <sys\timeb.h>
#include <sys\types.h>

static struct _timeb near itime = {
    0 }
; /* Initialize to force near */


/***
*clock_t clock() - Return the processor time used by this process.
*
*Purpose:
*   This routine calculates how much time the calling process
*   has used.  At startup time, startup calls _inittime which stores
*   the initial time.  The clock routine calculates the difference
*   between the current time and the initial time.
*
*   Clock must reference _cinitime so that _cinitim.asm gets linked in.
*   That routine, in turn, puts _inittime in the startup initialization
*   routine table.
*
*Entry:
*   No parameters.
*   itime is a static structure of type _timeb.
*
*Exit:
*   If successful, clock returns the number of CLK_TCKs (milliseconds)
*   that have elapsed.  If unsuccessful, clock returns -1.
*
*Exceptions:
*   None.
*
*******************************************************************************/

clock_t  clock()

{
    struct _timeb now;
    clock_t elapsed;
#ifdef _WINDOWS
    extern int _near _citime;
#else
    extern int _citime;
#endif

    /* Reference _citime so _inittime gets executed at startup time.*/

    _citime=0;

    /* Calculate the difference between the initial time and now. */

    _ftime(&now);
    elapsed = (now.time - itime.time) * CLOCKS_PER_SEC;
    elapsed += (int)now.millitm - (int)itime.millitm;
    return(elapsed);

}


/***
*void _inittime() - Initialize the time location
*
*Purpose:
*   This routine stores the process's time of startup.
*   It is only linked in if the user issues a clock runtime
*   call.
*
*Entry:
*   No arguments.
*
*Exit:
*   No return value.
*
*Exceptions:
*   None.
*
*******************************************************************************/

void _inittime()

{
    _ftime(&itime);
}
