/***
*localtim.c - Convert long time value to time structure
*
*   Copyright (c) 1985-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   Converts calendar time stored as a time_t value to a structure of type
*   struct tm expressed as local time.
*
*******************************************************************************/

#include <time.h>
#include <ctime.h>
#include <register.h>
#include <stddef.h>
#include <internal.h>

/***
*struct tm *localtime(time) - convert time_t value to tm structure
*
*Purpose:
*   Breaks down the time value into local date and time information,
*   correcting for time zone and daylight savings time (if any) and returns
*   a pointer to a structure containing the information.
*
* NOTES:
*   (1) gmtime must be called before _isindst to ensure that the tb time
*       structure is initialized.
*   (2) gmtime and localtime use a single statically allocated buffer; each
*       call to one of these routines destroys the contents of the previous
*       call.
*
*Entry:
*   time_t *time - pointer to a long time value
*
*Exit:
*   *struct tm - localtime returns a pointer to a time structure.
*   if an error occurs, returns NULL.
*
*Exceptions:
*   none
*
*******************************************************************************/

struct tm *  localtime (
    const time_t *timp
    )
{
    REG1 struct tm *tb;
    _WINSTATIC long ltime;

    /*
     * Check for illegal time_t value
     */
    if ( (long)(*timp) < 0L )
        return( NULL );

    __tzset();

    if ( (ltime = (long)*timp - _timezone) >= 0L ) {
        /*
         * General case: ltime is in range.
         */
        tb = gmtime( (time_t *)&ltime );

        /*
         * Check and adjust for Daylight Saving Time.
         */
        if ( _daylight && _isindst( tb ) ) {
            if ( (ltime += 3600L) < 0L )
                return( NULL );
            tb = gmtime( (time_t *)&ltime );
            tb->tm_isdst = 1;
        }
    }
    else if ( (ltime + _DAY_SEC) >= 0L ) {
        /*
         * Special case 1: local rep. for some time in the first day
         * the Epoch (01-01-70, UTC).
         */
        ltime += _DAY_SEC;
        tb = gmtime( (time_t *)&ltime );
        tb->tm_year = 69;   /* 1969 */
        tb->tm_mon = 11;    /* December */
        tb->tm_mday = 31;   /* 31st */
        tb->tm_wday = 3;    /* it was a Weds */
        tb->tm_yday = 364;  /* last day of 1969 */
    }
    else if ( (ltime - _DAY_SEC) >= 0L ) {
        /*
         * Special case 2: local rep. for some time in the last
         * 24 hours of the Epoch. The last of the Epoch is such
         * that we can simply increment the various day fields
         * (see comments below).
         *
         * For the record, the last representable second of the
         * Epoch is:
         *      03:14:07, 01-19-1938 (UTC)
         */
        ltime -= _DAY_SEC;
        tb = gmtime( (time_t *)&ltime );
        tb->tm_mday++;      /* not the end of a month */
        tb->tm_wday++;      /* not the end of a week */
        tb->tm_yday++;      /* not the end of a year */
    }

    return(tb);
}
