/***
*mktime.c - Normalize user time block structure
*
*   Copyright (c) 1987-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   Mktime converts the user's time structure (possibly incomplete)
*   into a fully defined structure with "normalized" values, and
*   converts it into a time_t value.
*
*******************************************************************************/

#include <stddef.h>
#include <ctime.h>
#include <time.h>
#include <internal.h>
#include <stdio.h>

/*
 * ChkAdd evaluates to TRUE if dest = src1 + src2 has overflowed
 */
#define ChkAdd(dest, src1, src2)    ( ((src1 >= 0L) && (src2 >= 0L) \
    && (dest < 0L)) || ((src1 < 0L) && (src2 < 0L) && (dest >= 0L)) )

/*
 * ChkMul evaluates to TRUE if dest = src1 * src2 has overflowed
 */
#define ChkMul(dest, src1, src2)    ( dest / src1 != src2 )

/***
*time_t mktime(tb) - Normalize user time block structure
*
*Purpose:
*   Mktime converts a broken-down time, in a tm structure, into calendar
*   time with the same encoding as that of the values returned by the
*   "time" function.  Three practical uses of this routine are:
*       (1) To have mktime fill in the tm_wday and tm_yday
*           values for the user.
*       (2) To pass in a time structure with "out of bounds"
*           values and have mktime.c "normalize" it (e.g., pass
*           in 1/35/87 and get back 2/4/87).
*       (3) Convert a broken-down time into a calendar time (in time_t
*           format).
*Entry:
*   struct tm *tb - pointer to a tm time structure to normalize, convert
*
*Exit:
*   Normal Return:
*   Mktime returns the specified calender time encoded as a value
*   of the type time_t (unsigned long).
*
*   Error Return:
*   If the calendar time cannot be represented mktime returns -1.
*
*Exceptions:
*   None.
*
*******************************************************************************/

time_t  mktime (
    struct tm *tb
    )
{
    struct tm *tbtemp;         /* temporary time structure */
    long tmptm1, tmptm2;           /* temps for hour and min. values */
    long tmzone;               /* diff between local and GMT */
    _WINSTATIC long tmptm3;        /* temp for second values */

    /*
     * First, make sure tb->tm_year reasonably within range.
     */
    if ( ((tmptm1 = tb->tm_year) < _BASE_YEAR - 1) || (tmptm1 > _MAX_YEAR
      + 1) )
        goto err_mktime;

    /*
     * Adjust month value so it is in the range 0 - 11.  This is because
     * we don't know how many days are in months 12, 13, 14, etc.
     */
    tmptm1 += tb->tm_mon/12;
    tb->tm_mon %= 12;
    if ( tb->tm_mon < 0 ) {
        tb->tm_mon += 12;
        tmptm1--;
    }

    /*
     * Check that year count is still in range.
     */
    if ( (tmptm1 < _BASE_YEAR - 1) || (tmptm1 > _MAX_YEAR + 1) )
        goto err_mktime;

    /*
     * Calculate days elapsed minus one, in the given year, to the given
     * month. Note the leap year adjustment assumes tmptm1 > 0 (thus,
     * BASEYEAR needs to be > 1 to ensure this).
     */
    tmptm2 = _days[tb->tm_mon];
    if ( !(tmptm1 & 3) && (tb->tm_mon > 1) )
        tmptm2++;

    /*
     * Get time zone correction.
     */
    __tzset();
    tmzone = _timezone;

    /*
     * Calculate elapsed hours. No possibility of overflow in 16-bit world
     * (ints are 16 bits, longs are 32 bits).
     */
    tmptm3 = (  /* 365 days for each elapsed year */
            ((tmptm1 - _BASE_YEAR) * 365L)

            /* 1 day per elapsed leap year */
          + ((tmptm1 - 1L) >> 2) - _LEAP_YEAR_ADJUST

            /* see above */
          + tmptm2

            /* day of month */
          + (long)tb->tm_mday   )

            /* convert to hours */
          * 24L

            /* hours since midnight */
          + (long)tb->tm_hour

            /* hours to UTC */
          + tmzone / 3600L;

    /*
     * Reduce tmzone mod hours (the hours have been added in above).
     */
    tmzone %= 3600L;

    /*
     * Calculate elapsed minutes. Must guard against overflow from here
     * on.
     */
    tmptm1 = tmptm3 * 60L;
    if ( ChkMul(tmptm1, tmptm3, 60L) )
        goto err_mktime;

    tmptm2 = tmptm1 + (long)tb->tm_min;
    if ( ChkAdd(tmptm2, tmptm1, (long)tb->tm_min) )
        goto err_mktime;

    tmptm3 = tmptm2 + tmzone / 60L;
    if ( ChkAdd(tmptm3, tmptm2, tmzone / 60L) )
        goto err_mktime;

    /*
     * Reduce tmzone mod minutes (the minutes have added in above).
     */
    tmzone %= 60L;

    /*
     * Calculate elapsed seconds.
     */
    tmptm1 = tmptm3 * 60L;
    if ( ChkMul(tmptm1, tmptm3, 60L) )
        goto err_mktime;

    tmptm2 = tmptm1 + (long)tb->tm_sec;
    if ( ChkAdd(tmptm2, tmptm1, (long)tb->tm_sec) )
        goto err_mktime;

    tmptm3 = tmptm2 + tmzone;
    if ( ChkAdd(tmptm3, tmptm2, tmzone) )
        goto err_mktime;

    /*
     * Check that the number of elapsed seconds is >= 0.
     */
    if ( tmptm3 < 0L )
        goto err_mktime;

    /*
     * Convert this second count back into a time block structure.
     */
    if ( (tbtemp = localtime(&tmptm3)) == NULL )
        goto err_mktime;

    /*
     * We now must compensate for DST. The ANSI rules are to use the
     * passed-in daylight flag if non-negative, otherwise compute if we
     * have DST. Recall that tbtemp has the time without DST compensation,
     * but has computed if we are in DST or not.
     */
    if ( (tb->tm_isdst > 0) || ((tb->tm_isdst < 0) && (tbtemp->tm_isdst
      > 0)) ) {
        /* compensate for DST */
        tmptm3 -= 3600L;
        if ( tmptm3 < 0L )
        goto err_mktime;
        tbtemp = localtime(&tmptm3);
    }

    /*
     * Update *tb and return the calculated time_t value
     */
    *tb = *tbtemp;
    return ( (time_t)tmptm3 );

    /*
     * All error paths end up here.
     */
err_mktime:
    return ( (time_t)(-1L) );
}
