/***
*tzset.c - set timezone information and see if we're in daylight time
*
*   Copyright (c) 1985-1992, Microsoft Corporation. All rights reserved.
*
*Purpose:
*   defines tzset() - set timezone and daylight saving time vars
*
*******************************************************************************/

#ifdef _WINDLL
#include <windows.h>
#undef min  /* windows declarations conflict with ours and generate */
#undef max  /* warnings, even though semantically equivalent */
#endif

#include <string.h>
#include <ctype.h>
#include <ctime.h>
#include <register.h>
#include <time.h>
#include <stdlib.h>
#include <internal.h>

/***
*void tzset() - sets timezone information and calc if in daylight time
*
*Purpose:
*   Sets the timezone information from the TZ environment variable
*   and then sets timezone, daylight, and tzname.  If we're in daylight
*   time is automatically calculated.
*
*Entry:
*   None, reads TZ environment variable.
*
*Exit:
*   sets daylight, timezone, and tzname global vars, no return value
*
*Exceptions:
*
*******************************************************************************/

#define _tzset_lk _tzset


void __tzset(void)
{
static int near first_time;


    if ( ! first_time )
    {
        _tzset_lk();
        first_time++;
    }


}



void _tzset(void)

{
    REG1 char *TZ;
    REG2 int negdiff = 0;
#ifdef _WINDLL
    char static tzstring[16];
    char far * dosenv;
#endif

#ifdef _WINDLL
    /* Since there's no windll initialization, get env string at runtime if
    necessary. */

    if (!(TZ = getenv("TZ")) || !*TZ) {

        dosenv = GetDOSEnvironment();
        if ((dosenv == (char far *)NULL) || (!*dosenv))
        return;

        /* scan through the environment looking for a "TZ=" string.  (The
           environment is a series of null terminated strings with a double
           null at the end.) */

        for (; (*dosenv) ; (dosenv += (_fstrlen(dosenv)+1)) )
        if (_fstrncmp(dosenv, (char far *) "TZ=", 3) == 0)
            break;

        if (!*dosenv)
        return; /* didn't find TZ string */

        dosenv+=3;      /* bump pointer to TZ string */

        /* copy string to near memory and set TZ appropriately. */

        TZ = tzstring;
        _fstrncpy (tzstring, dosenv, 16);
    }

#else
    if (!(TZ = getenv("TZ")) || !*TZ)
        return;
#endif
    strncpy(_tzname[0], TZ, 3);

    /*
     * time difference is of the form:
     *
     *  [+|-]hh[:mm[:ss]]
     *
     * check minus sign first.
     */
    if ( *(TZ += 3) == '-' ) {
        negdiff++;
        TZ++;
    }

    /*
     * process, then skip over, the hours
     */
    _timezone = atol(TZ) * 3600L;

    while ( (*TZ == '+') || ((*TZ >= '0') && (*TZ <= '9')) ) TZ++;

    /*
     * check if minutes were specified
     */
    if ( *TZ == ':' ) {
        /*
         * process, then skip over, the minutes
         */
        _timezone += atol(++TZ) * 60L;
        while ( (*TZ >= '0') && (*TZ <= '9') ) TZ++;

        /*
         * check if seconds were specified
         */
        if ( *TZ == ':' ) {
        /*
         * process, then skip over, the seconds
         */
        _timezone += atol(++TZ);
        while ( (*TZ >= '0') && (*TZ <= '9') ) TZ++;
        }
    }

    if ( negdiff )
        _timezone = -_timezone;

    /*
     * finally, check for a DST zone suffix
     */
    if ( _daylight = *TZ )
        strncpy(_tzname[1], TZ, 3);
    else
        *_tzname[1] = '\0';

}


/***
*int _isindst(tb) - determine if broken-down time falls in DST
*
*Purpose:
*   Determine if the given broken-down time falls within DST. Only
*   modern, summer-time DST is handled (USA, post 1967).
*
*   This is the rule for years before 1987:
*   a time is in DST iff it is on or after 02:00:00 on the last Sunday
*   in April and before 01:00:00 on the last Sunday in October.
*   This is the rule for years starting with 1987:
*   a time is in DST iff it is on or after 02:00:00 on the first Sunday
*   in April and before 01:00:00 on the last Sunday in October.
*
*Entry:
*   struct tm *tb - structure holding broken-down time value
*
*Exit:
*   1, if time represented is in DST
*   0, otherwise
*
*******************************************************************************/

int _isindst (
    REG1 struct tm *tb
    )
{
    int mdays;
    REG2 int yr;
    int critsun;

    /*
     * Handle easy cases.
     *
     * Modern DST was put into effect by Congress in 1967. Also, if the
     * month is before April or after October, it cannot be DST.
     */
    if ( (tb->tm_year < 67) || (tb->tm_mon < 3) || (tb->tm_mon > 9) )
        return(0);

    /*
     * If the month is after April and before October, it must be DST.
     */
    if ( (tb->tm_mon > 3) && (tb->tm_mon < 9) )
        return(1);

    /*
     * Now for the hard part.  Month is April or October; see if date
     * falls between appropriate Sundays.
     */

    /*
     * The objective for years before 1987 (after 1986) is to determine
     * if the day is on or after 2:00 am on the last (first) Sunday in
     * April, or before 1:00 am on the last Sunday in October.
     *
     * We know the year-day (0..365) of the current time structure. We must
     * determine the year-day of the last (first) Sunday in this month,
     * April or October, and then do the comparison.
     *
     * To determine the year-day of the last Sunday, we do the following:
     *    1. Get the year-day of the last day of the current month (Apr
     *       or Oct)
     *    2. Determine the week-day number of #1,
     *       which is defined as 0 = Sun, 1 = Mon, ... 6 = Sat
     *    3. Subtract #2 from #1
     *
     * To determine the year-day of the first Sunday, we do the following:
     *    1. Get the year-day of the 7th day of the current month
     *       (April)
     *    2. Determine the week-day number of #1,
     *       which is defined as 0 = Sun, 1 = Mon, ... 6 = Sat
     *    3. Subtract #2 from #1
     */

    /*
     * First we get #1. The year-days for each month are stored in _days[]
     * they're all off by -1
     */
    if ( ((yr = tb->tm_year) > 86) && (tb->tm_mon == 3) )
        mdays = 7 + _days[tb->tm_mon];
    else
        mdays = _days[tb->tm_mon+1];

    /*
     * if this is a leap-year, add an extra day
     */
    if ( !(yr & 3) )
        mdays++;

    /*
     * mdays now has #1
     */

    /* Now get #2. We know the week-day number of January 1, 1970, which is
     * defined as the constant _BASE_DOW. Add to this the number of elapsed
     * days since then, take the result mod 7, and we have our day number.
     *
     * To obtain #3, we just subtract this from mdays.
     */

    critsun = mdays - ((mdays + 365 * (yr - 70) + ((yr - 1) >> 2) -
          _LEAP_YEAR_ADJUST + _BASE_DOW) % 7);

    /* Now we know 1 and 3; we're golden: */

    return ( (tb->tm_mon == 3)
         ? ((tb->tm_yday > critsun) ||
            ((tb->tm_yday == critsun) && (tb->tm_hour >= 2)))
         : ((tb->tm_yday < critsun) ||
            ((tb->tm_yday == critsun) && (tb->tm_hour < 1))) );
}
