{ 6/24/88: A program to play John Conway's Life in three dimensions.
  Idea, algorithm, and explanation from Scientific American, February 1987,
  Computer Recreations column.

  Implementation copyright(c) 1988 L.G. Eckelkamp.  Please feel free to modify
  the source code as you see fit.  I don't suppose anything can be more like
  public domain than LIFE variations, so do whatever makes you happy with it.
  This was originally written using Turbo Pascal 4.0 and it compiles just fine
  using 5.0 without any changes.

  "The first number indicates the fewest living neighbors a cell must have to
  keep from being undernourished; the second indicates the most  it can have
  before it is overcrowded.  The third and fourth numbers govern the fate of
  dead cells: third is the fewest living neighbors a cell must have to come
  alive; the foursth is the most it can have to come alive.  According to
  this notation, Conway's Life becomes Life 2333.
    "In Life 4555, a living cell dies if it has fewer than four or more than
  five living neighbors.  A dead cell comes to life if it has exactly five
  living neighbors.
    "Two large, 3-D arrays called CELLS and NEWCELLS are assigned three indices,
  I, J, and K, which correspond to the three coordinates of the cellular space.
  The content of each array element indicates whether the corresponding cell is
  alive or dead.  Let 1 signify life and 0 signify death.
    "Three nested loops...are needed to compute the status of each cell in each
  generation.  The outer loop used the I index to compute a succession of planes
  that sweeps through the space.  Within this loop are two others using J and K
  respectively.  The J loop computes successive rows within each plane and the
  K loop computes successive cells in a row.  One can use the following generic
  form as a guide:

      for i = 1 to 30
        for j = 1 to 30
          for K = 1 to 30
            compute neighbors
            decide status
            display live cubes

    "Inside the innermost loop there are three basic tasks for the program to
  do.  The task Compute Neighbors requires the program to examine the 26 neighbors
  of each cell and total the number currently alive.  This can be done with
  three miniature loops or by listing all possible coordinates of the 26 cells.
  In loop form the procedure might use the following algorithm:

      tot := 0;
      for L := I-1 to I+1
        for M := J-1 to J+1
          for N := K-1 to K+1
            if cells(l,m,n) = 1 then
              tot := tot+1
      tot := tot-cells(i,j,k)

  The last line of the procedure ensures that the status of cells(i,j,k) does
  not add to the total.
    "Having decided on the total TOT of living neighbors, the program must next
  decide the new status of the current cell, cells(i,j,k).  The task Decide Status
  is merely a matter of checking the size of tot in relation to the status of
  cells(i,j,k):

      if cells(i,j,k) = 0
        then if tot = 5
          then newcells(i,j,k) := 1
          else newcells(i,j,k) := 0
      if cells(i,j,k) = 1
        then if tot<4 or tot>5
          then newcells(i,j,k) := 0
          else newcells(i,j,k) := 1

     "One can change the algorithm to fit Life 5766 or make it general enough
   to fit any three-dimensional rule whatever...
     "A general version of the foregoing status computation might use four
   variables Bays calls el, eu, fl, and fu.  The letters e and f stand for en-
   vironment and fertility and l and u for lower and upper.  Thus el and eu are the
   lower and upper bounds for the continued life of a cell in its environment;
   the cell will stay alive if the number of living cubes surrounding it is
   greater than or equal to el but less than or equal to eu.  By the same token
   fl and fu are the conditions of fertility for a dead cell.  Its rebirth is
   guaranteed if the number of living cubes surrounding it is greater than or
   equal to fl but less than or equal to fu.  The general algorithm is therefore

      if cells(i,j,k) = 0
        then if tot < fl or tot > fu
          then newcells(i,j,k) := 0
          else newcells(i,j,k) := 1
      if cells(i,j,k) = 1
        then if tot < el or tot > eu
          then newcells(i,j,k) := 0
          else newcells(i,j,k) := 1

   At this point in either version of a three dimensional Life program the
contents of newcells can be moved into cells by means of the appropriate
triple loop. This frees up newcells for the next generation of living cubes.
}

program ThreeDimensionalLife;
uses  Graph, Drivers;

const
     MAXDIM = 20;         {15 gets color on every plane}
     ARRAYDIM = 21;       {MAXDIM + 1; don't ask...range errors}
type
    cols    = 0..ARRAYDIM;
    rows    = 0..ARRAYDIM;
    plane   = 0..ARRAYDIM;

var
    i : cols;
    j : rows;
    k : plane;
    cell : array[cols,rows,plane] of byte;
    newcell : array[cols,rows,plane] of byte;
    tot : shortint;
    generation : integer;
    grDriver, grMode, grError : integer;



procedure InitializeCells;
{loads up the first array}

var UserChoice : longint;
    Ch         : char;

begin
  repeat
    write('Would you like to seed the population? (Y/N) ');
    readln(ch);
    Ch := UpCase(ch);
  until Ch in ['Y','N'];
    case Ch of
      'Y' : begin
              ClearDevice;
              writeln;
              writeln('Seed integer range is  -2147483648 to 2147483647');
              writeln;
              write('Enter the random seed integer: ');
              readln(UserChoice);
              RandSeed := UserChoice;
            end;
      'N' : begin
            Randomize;
            end
     end; {case}
    writeln('Populating worlds...');

    {FillChar(cell, SizeOf(cell), Random(2));}
    {FillChar has some sort of problem used this way}

     for i := 0 to ARRAYDIM do begin
       for j := 0 to ARRAYDIM do begin
         for k := 0 to ARRAYDIM do begin
           cell[i,j,k] := Random(2);
         end;
       end;
     end;
     writeln('Universe now populated.');
end; {InitializeCells}

(*========================================================================*)
(*DecideStatus defines the nature of the LIFE simulation...4555, 5766, etc*)
(*========================================================================*)

procedure DecideStatus;

begin
  case cell[i,j,k] of
    0 : if (tot>=3) and (tot<=5) then newcell[i,j,k] := 1
           else newcell[i,j,k] := 0;
    1 : if (tot<4) or (tot>8) then newcell[i,j,k] := 0
           else newcell[i,j,k] := 1
  end; {case}

end; {procedure DecideStatus}

(*==========================================================================*)


procedure ShiftNewcellToCell;

begin
  move(newcell,cell,sizeof(newcell));   {block move suggested by Tom Vogl}
end; {procedure ShiftNewcellToCell}

(*********************************************************************)

procedure ComputeNeighbors(column, rrow, pplane: byte);
{looks at one specific cell and checks surrounding cells for life}
var
   l, m, n : byte;
begin
  tot := 0;
  for l := (i-1) to (i+1) do
      begin {i loop}
        for m := (j-1) to (j+1) do
        begin {j loop}
          for n := (k-1) to (k+1) do
          begin {k loop}
            if cell[l,m,n] = 1 then
            tot := tot + 1;
          end;  {k loop, innermost}
        end;  {j loop}
      end;  {i loop, outermost}
    if (cell[i,j,k] = 1) and (tot>=1) then       {tot>=1 to cover tot=0}
    tot := tot - 1;  {so current cell doesn't add itself}
end;  {ComputeNeighbors}



procedure DisplayInVGAMode; forward;

procedure NextGeneration;
var  HowMany       : integer;

begin
write('How many generations to calculate? : ');
readln(HowMany);
writeln('OK, you''ll have to be patient for a moment, I''m doing the best I can.');
generation := 0;

for generation := 1 to HowMany do begin
  for i := 1 to MAXDIM do begin
    for j := 1 to MAXDIM do begin
      for k := 1 to MAXDIM do begin
        ComputeNeighbors(i,j,k);
        DecideStatus;
      end;  {innermost loop}
    end;
  end;      {outermost loop}

  ShiftNewCellToCell;   {set up cell array for new display}

  DisplayInVGAMode;

  end;     {end of generation loop}
  writeln('complete...');
end; {NextGeneration}



procedure DisplayInVGAMode;
const
    DEPTH  = 5;
    RADIUS = 10;
var
    d1, x1, y1, x2, y2, xMax, yMax , width, height : integer;
    ILoopSize,JLoopSize : integer;
    str2  : string[2];
    str79 : string[79];
    strG  : string[4];

begin {DisplayInVGAMode}
    ClearDevice;
    xMax := GetMaxX; yMax := GetMaxY;
    Str(generation,strG);
    for k := MAXDIM downto 1 do begin
      SetViewPort(0,0,xMax,15, ClipOn);
      ClearViewPort;
      Str(k,str2);
      str79 := Concat('Generation: ',strG,'    Plane: ',str2);
      OutTextXY(0,0,str79);
      SetViewPort(0,0,xMax,yMax, ClipOn);
      width := (RADIUS*2)-(k div 2);        {... RADIUS*2...}
      height:= width;
      SetFillStyle(SolidFill, k);
      SetColor(Yellow);
      for i :=  MAXDIM downto 1 do begin
        ILoopSize := i*22;                 {22}
        y2 := height + ILoopSize;
        for j :=  1 to MAXDIM do begin
          JLoopSize := j*28;                {28}
          x2 := width + JLoopsize;
          case cell[i,j,k] of
                    0 : begin
                        end;
                    1 : begin
                        {Circle(x2,y2,(JLoopSize div 10));}
                        Bar3D(JLoopSize,ILoopSize,x2,y2,DEPTH, TopOn);
                        end
               end; {case}
        end;
      end;
      SetViewPort(0,0,xMax,10,ClipOn);  {resets for OutTextXY call above}

    end;  {k loop}
    SetViewPort(0,0,xMax,10,ClipOn);
    ClearViewPort;
    str79 := Concat(strG,' Generations Completed...');
    OutTextXY(0,0,str79);
end; {DisplayInVGAMode}

begin {program}
  if RegisterBGIDriver(@EGAVGADriverProc) < 0 then halt;
  grDriver := Detect;
  grMode := VGAHi;
  InitGraph(grDriver,grMode,'');
  grError := GraphResult;
  if grError <> grOK then
    writeln('Screw Up Alert : ',graphErrorMsg(grError)) else
      begin
        writeln('John Conway''s LIFE in 3-D...');
        writeln('as developed by Carter Bays and described by A.K. Dewdney.');
        writeln; writeln;
        writeln('Implementation (c) 1988 by L.G. Eckelkamp');
        writeln;
        write('           <ENTER> continues...');
        readln;
        ClearDevice;
        InitializeCells;
        NextGeneration;
      end;
   readln;
   CloseGraph;
end. {bwahahahahahahahha}