{ Charts.pas       --Defines the chart classes
                   --For use on IBM 8514 adapter define Use8514 and compile
                   --Written by Zack Urlocker
                     Copyright 1990, The Whitewater Group.
                     All rights reserved.
                   --mzu 06/01/90
}

unit Charts;

interface

uses
 PTypes, Dicts, Graph, Objects;  { Points, Rects, Dicts, BGI graphics, Streams }

type

ChartStream = object(DosStream)          { for storing charts }
  procedure registerTypes; virtual;
end;  { ChartStream }


ChartPtr = ^Chart;

Chart = object(Base)       { Chart is a formal class with descendants }
{ Object fields }
  Name : String;           { title string          }
  Items : DynamicDict;     { key->value pairs      }
  Color : Integer;         { color of the chart    }
  Scale : Point;           { scaling factor        }
  Lead : Point;            { lead before edges     }
  Space : Integer;         { space between items   }
  Loc : Rect;              { location of the chart }
  radius : Integer;        { useful for piecharts  }

{ Functions and procedures }
  constructor Init;        { so that inheritance works }
  destructor Done; virtual;{ to clean up memory }
  procedure Draw; virtual;
  procedure DrawFrame; virtual;
  procedure DrawLabels; virtual;
  procedure DrawData; virtual;
  procedure ReScale; virtual;
  procedure AdjustScale(max : Integer); virtual;
  procedure Erase;
  procedure Move(l, t, r, b : Integer);
  procedure Store(S : ChartStream);
  constructor Load(S : ChartStream);
  procedure Write(fName : String);
  constructor Read(fName : String);

  procedure Add(key : String; value : Integer);
  procedure Remove(key : String);
  procedure Update(key : String; value : Integer);
  function XLead : Integer;
  function YLead : Integer;
  function XScale : Integer;
  function YScale : Integer;
  function Left : Integer;
  function Top : Integer;
  function Right : Integer;
  function Bottom : Integer;
  procedure ResetLead; virtual;
  procedure ResetSpace; virtual;
  procedure ResetScale; virtual;
  procedure SetColor(c : Integer);
  procedure SetLead(x, y : Integer);
  procedure SetLoc(l, t, r, b : Integer);
  procedure SetName(s : String);
  procedure SetScale(x, y : Integer);
  procedure SetSpace(i : Integer);
  function GetColor : Integer;
  function GetName : String;
  function GetSpace : Integer;
end;  { Chart }

HBarChartPtr = ^HbarChart;

HBarChart = object(Chart)                { Horizontal bars }
{ Functions and procedures }
  procedure Store(S : ChartStream);
  constructor Load(S : ChartStream);
  procedure DrawLabels; virtual;
  procedure DrawData; virtual;
  procedure ResetLead; virtual;
  procedure AdjustScale(max : Integer); virtual;
end;  { HBarChart }


VBarChartPtr = ^VBarChart;

VBarChart = object(Chart)                { Vertical bars }
{ Functions and procedures }
  procedure Store(S : ChartStream);
  constructor Load(S : ChartStream);
  procedure DrawLabels; virtual;
  procedure DrawData; virtual;
  procedure ResetSpace; virtual;
  procedure AdjustScale(max : Integer); virtual;
end;  { VBarChart }


V3DBarChartPtr = ^V3DBarChart;           { Vertical 3D bars }

V3DBarChart = object(VBarChart)
{ Functions and procedures }
  procedure Store(S : ChartStream);
  constructor Load(S : ChartStream);
  procedure DrawData; virtual;
end;  { V3DBarChart }

PieChartPtr = ^PieChart;

PieChart = object(Chart)                 { Pie charts }
{ Functions and procedures }
  procedure Store(S : ChartStream);
  constructor Load(S : ChartStream);
  procedure DrawLabels; virtual;
  procedure DrawData; virtual;
  procedure AdjustScale(max : Integer); virtual;
  procedure CalcRadius;
end;  { PieChart }


implementation

var

OldExitProc : Pointer;  { Saves exit procedure address }

function Minimum(x, y : Integer): Integer; forward;
function Maximum(x, y : Integer): Integer; forward;

{ *********   Chart  ********* }

constructor Chart.Init;
begin
  Name := '';
  Color := White;
  ResetScale;
  ResetLead;
  ResetSpace;
  Loc.Init(0, 0, 0, 0);
  if not(Items.Init) then   { Constructor failure }
  begin
    Items.Done;
    Fail;
  end;
end;

destructor Chart.Done;
{ Dispose of the dictionary of items. }
begin
  Items.Done;
end;

procedure Chart.Draw;
begin
  if not Loc.Empty then
  begin
     ReScale;
     Graph.SetColor(White);
     DrawFrame;
     DrawLabels;
     SetFillStyle(1, Color);
     DrawData;
     SetViewPort(0, 0, GetMaxX, GetMaxY, ClipOff);
  end;
end;

procedure Chart.Erase;
begin
 SetViewPort(Loc.Left, Loc.Top, Loc.Right, Loc.Bottom, ClipOn);
 ClearViewPort;
 SetViewPort(0, 0, GetMaxX, GetMaxY, ClipOn);
end;

procedure Chart.Move(L, T, R, B : Integer);
{ Move the chart to a new location by erasing then redrawing. }
var  Image : Pointer;
     Size : Word;
begin
 Erase;
 SetLoc(L, T, R, B);
 Draw;
end;

procedure Chart.DrawFrame;
begin
  Rectangle(Loc.Left, Loc.Top, Loc.Right, Loc.Bottom);
  SetViewPort(Loc.Left+1, Loc.Top+1, Loc.Right-1, Loc.Bottom-1, ClipOn);
  SetTextJustify(CenterText, TopText);
  SetTextStyle(TriplexFont, HorizDir, 3);
  OutTextXY(Loc.Width div 2, 2, Name);
  SetTextStyle(DefaultFont, HorizDir, 1); { default }
  SetTextJustify(LeftText, TopText);  { restore default }
end;

procedure Chart.ReScale;
var Max : Integer;
begin
  Max := Items.MaxValue;
  If Max > 0 then
  begin
    resetLead;
    resetSpace;
    adjustScale(Max);
  end;
end;

{ Abstract methods that must be implemented in descendant classes. }
procedure Chart.DrawData;
begin
  abstract;
end;

procedure Chart.DrawLabels;
begin
  abstract;
end;

procedure Chart.AdjustScale(max:Integer);
begin
  abstract;
end;


{ File and stream I/O methods }

procedure Chart.Write(FName : String);
{ Store a chart onto a file by storing onto a stream. }
var
  S : ChartStream;
begin
  S.Init(FName, SCreate);
  if S.Status <> 0 then
    writeLn('** Cannot create file:', FName)
  else
    begin
      Store(S);
      if S.Status <> 0 then
        writeLn('** Disk write error:', FName);
    end;
  S.Done;
end;

constructor Chart.Read(FName : String);
{ Read a chart from a file. }
var S : ChartStream;
begin
  S.Init(FName, SOpenRead);
  if S.Status <> 0 then
    writeLn('** Cannot open file: ', FName)
  else
    begin
      Load(S);
      if S.Status <> 0 then
        writeLn('** Disk read error:', FName);
    end;
  S.Done;
end;

procedure Chart.Store(S : ChartStream);
{ Store a chart onto a stream. Not all object fields are stored.
  For example, scale, lead, space are set properly when you rescale.
  Strings are stored with the length byte first.  Must be read in
  the exact same order. }
var i,size,len,value,l,t,r,b: Integer;
    Key : String;
begin
   len := length(name);
   S.write(len, sizeOf(Integer));      { Length byte }
   S.write(name, len+1);
   S.write(color, sizeOf(Integer));
   l:=left;
   r:=right;
   t:=top;
   b:=bottom;
   S.write(l, sizeOf(Integer));
   S.write(t, sizeOf(Integer));
   S.write(r, sizeOf(Integer));
   S.write(b, sizeOf(Integer));
   Size := Items.Size;
   S.write(Size, sizeOf(Integer));
   items.reset;
   for I := 1 to Items.Size do           { store each key->value pair }
   begin
     Key := Items.CurrentKey;
     len := Length(key);
     Value := Items.CurrentValue;
     S.write(len, sizeOf(Integer));      { Length byte }
     S.write(Key, len+1);
     S.write(Value, sizeOf(Integer));
     Items.Next;
   end;
end;

constructor Chart.Load(S : ChartStream);
{ Load a chart from a stream. Must be read in same order written. }
var i,n,len,value,l,t,r,b : Integer;
    Key : string;
begin
 Items.init;
 S.Read(len, sizeOf(Integer));
 S.Read(name, len+1);
 S.read(color, sizeOf(Integer));
 S.read(l, sizeOf(Integer));
 S.read(t, sizeOf(Integer));
 S.read(r, sizeOf(Integer));
 S.read(b, sizeOf(Integer));
 S.Read(N, SizeOf(Integer));
 Loc.init(l,t,r,b);
 For i:=1 to n do                         { read each key->value pair }
   begin
     S.Read(len, sizeOf(Integer));
     S.Read(key, len+1);
     S.Read(value, sizeOf(Integer));
     Add(key, value);
   end;
end;

{ Pass through methods }

procedure Chart.Add(key : String; value : Integer);
begin
  Items.Add(key, value);
end;

procedure Chart.Remove(key : String);
begin
  Items.Remove(key);
end;

procedure Chart.Update(key : String; value : Integer);
begin
  Items.Update(key, value);
end;

procedure Chart.ResetLead;
begin
  Lead.Init(15, 30);
end;

procedure Chart.ResetSpace;
begin
  Space := 15;
end;

procedure Chart.ResetScale;
begin
  Scale.Init(10, 10);
end;

procedure Chart.SetColor(c : Integer);
begin
  Color := c;
end;

procedure Chart.SetLead(x, y : Integer);
begin
  Lead.SetX(x);
  Lead.SetY(y);
end;

procedure Chart.SetLoc(l, t, r, b : Integer);
begin
  Loc.SetLoc(l, t, r, b);
end;

procedure Chart.SetName(s : String);
begin
  Name := s;
end;

procedure Chart.SetScale(x, y : Integer);
begin
  Scale.SetX(x);
  Scale.SetY(y);
end;

{ Miscelaneous Chart access methods }

procedure Chart.SetSpace(i : Integer);
begin
  Space := i;
end;

function Chart.GetColor : Integer;
begin
  GetColor := Color;
end;

function Chart.GetName : String;
begin
  GetName := Name;
end;

function Chart.GetSpace : Integer;
begin
  GetSpace := Space;
end;


function Chart.XLead : Integer;
begin
  XLead := Lead.GetX;
end;

function Chart.YLead : Integer;
begin
  YLead := Lead.GetY;
end;

function Chart.XScale : Integer;
begin
  XScale := Scale.GetX;
end;

function Chart.YScale : Integer;
begin
  YScale := Scale.GetY;
end;

function Chart.Left : Integer;
begin
  Left := Loc.Left;
end;

function Chart.Top : Integer;
begin
  Top := Loc.Top;
end;

function Chart.Right : Integer;
begin
  Right := Loc.Right;
end;

function Chart.Bottom : Integer;
begin
  Bottom := Loc.Bottom;
end;


{ *********   HBarChart  ********* }

procedure HBarChart.Store(S : ChartStream);
begin
  Chart.store(S);
end;

constructor HBarChart.Load(S : ChartStream);
begin
  Chart.load(S);
end;

procedure HBarChart.DrawLabels;
var I : Integer;
    x, y : Integer;
begin
  x := 2;
  Items.Reset;
  for I:=1 to Items.Size do
  begin
    y := YLead + (I-1) * (Space + YScale);
    OutTextXY(x, y, Items.CurrentKey);
    Items.Next;
  end;
end;

procedure HBarChart.DrawData;
var I : Integer;
    l, t, r, b: Integer;
begin
  l := XLead;
  Items.Reset;
  for I := 1 to Items.Size do
  begin
    t := YLead + (I-1) * (YScale + Space);
    r := l + Items.CurrentValue * XScale;
    b := t + YScale;
    bar(l, t, r, b);
    Items.Next;
  end;
end;

procedure HBarChart.ResetLead;
begin
  Lead.Init(50, 30);
end;

procedure HBarChart.AdjustScale(max : Integer);
var x, y : Integer;
{ Note: integer division! Scale > 1 }
begin
  x := (Loc.Width - 2 * XLead) div max;
  setScale(x, 10);
end;


{ *********   VHBarChart  ********* }

procedure VBarChart.Store(S : ChartStream);
begin
  Chart.store(S);
end;

constructor VBarChart.Load(S : ChartStream);
begin
  Chart.load(S);
end;

procedure VBarChart.DrawLabels;
var I : Integer;
    x, y : Integer;
begin
  y := Loc.Height - 4 - TextHeight(Name);
  Items.Reset;
  for I:=1 to Items.Size do
  begin
      x := -5 + I * (Space + XScale);
      OutTextXY(x, y, Items.CurrentKey);
      Items.Next;
  end;
end;

procedure VBarChart.DrawData;
var I : Integer;
    l, t, r, b: Integer;
begin
  b := Loc.Height - YLead;
  Items.Reset;
  for I := 1 to Items.Size do
  begin
    t := b - Items.CurrentValue * YScale;
    l := I * (Space + XScale);
    r := l + XScale;
    bar(l, t, r, b);
    Items.Next;
  end;
end;

procedure VBarChart.ResetSpace;
begin
  Space := 30;
end;

procedure VBarChart.AdjustScale(max : Integer);
var y : Integer;
begin
  y := (Loc.Height - 2 * YLead) div max;
  setScale(10, y);
end;

{ *********   V3DBarChart *********}

procedure V3DBarChart.Store(S : ChartStream);
begin
  Chart.store(S);
end;

constructor V3DBarChart.Load(S : ChartStream);
begin
  Chart.load(S);
end;

procedure V3DBarChart.DrawData;
var I : Integer;
    l, t, r, b: Integer;
begin
  b := Loc.Height - YLead;
  Items.Reset;
  for I := 1 to Items.Size do
  begin
    t := b - Items.CurrentValue * YScale;
    l := I * (Space + XScale);
    r := l + XScale;
    bar3D(l, t, r, b, 5, TopOn);
    Items.Next;
  end;
end;


{ *********   PieChart  ********* }


procedure PieChart.Store(S : ChartStream);
begin
  Chart.store(S);
end;

constructor PieChart.Load(S : ChartStream);
begin
  Chart.load(S);
end;

procedure PieChart.DrawLabels;
var I : Integer;
    l, t, r, b: Integer;
    x, y : integer;
    OldPattern : FillPatternType;
begin
   CalcRadius;
   GetFillPattern(OldPattern);
   l :=  2 * (Radius + XLead);
   r := l + XScale;

   Items.Reset;
   for I := 1 to Items.Size do
   begin
     t := YLead + (I-1) * (YScale + Space);
     b := t + YScale;
     SetFillStyle(I , Color);             { Filled rect }
     bar(l, t, r, b);
     SetFillPattern(OldPattern, Color);   { Solid Frame }
     rectangle(l, t, r, b);
     OutTextXY(r + 20, t, Items.CurrentKey);
     Items.Next;
  end;
end;

procedure PieChart.DrawData;
var I : Integer;
    l, t, r, b: Integer;
    x, y : integer;
    Total : Integer;
    Angle, StartAngle, EndAngle : Word;
    OldPattern : FillPatternType;
begin
   CalcRadius;
   x := XLead + Radius;
   y := YLead + Radius;

   Total := 0;
   Items.Reset;
   for I := 1 to Items.Size do
   begin
     Total := Total + Items.CurrentValue;
     Items.Next;
   end;

   StartAngle := 0;
   GetFillPattern(OldPattern);

   Items.Reset;
   for I := 1 to Items.Size - 1 do
   begin
     SetFillStyle(I , Color);
     Angle := Trunc((Items.CurrentValue / Total) * 360);
     EndAngle := StartAngle + Angle;
     PieSlice(x, y, StartAngle, EndAngle, Radius);
     StartAngle := EndAngle;
     Items.Next;
   end;

   { Do the last slice separately to avoid truncation }
   SetFillStyle(I+1 , Color);
   PieSlice(x, y, StartAngle, 360, Radius);

   SetFillPattern(OldPattern, Color);
end;

procedure PieChart.AdjustScale(max : Integer);
var y : Integer;
begin
  y := (Loc.Height - YLead) div (Items.Size) - Space;
  y := Minimum(y, 30);
  y := Maximum(y, 10);
  setScale(y, y);
end;

procedure PieChart.CalcRadius;
begin
   if (Loc.Width - XLead) < (Loc.Height - YLead) then
     radius := (Loc.Width - XLead) div 2 - 5
   else
     radius := (Loc.Height - YLead) div 2 - 5;
end;

{ ********** ChartStream  ************ }

procedure ChartStream.registerTypes;
begin
  DosStream.registerTypes;
  register(typeOf(Chart), @Chart.store, @Chart.load);
  register(typeOf(HBarChart), @HBarChart.store, @HBarChart.load);
  register(typeOf(VBarChart), @VBarChart.store, @VBarChart.load);
  register(typeOf(PieChart), @PieChart.store, @PieChart.load);
  register(typeOf(V3DBarChart), @V3DBarChart.store, @V3DBarChart.load);
end;


{ ********** Miscelaneous Routines ********** }


function Minimum(x, y : Integer): Integer;
var ans : Integer;
begin
  if x < y then
     ans := x
  else
     ans := y;
  Minimum := ans;
end;

function Maximum(x, y : Integer): Integer;
var ans : Integer;
begin
  if x > y then
     ans := x
  else
     ans := y;
  Maximum := ans;
end;


{$F+}
procedure ChartExitProc;
{ Automatically close out graphics system. }
begin
  ExitProc := OldExitProc; { Restore exit procedure address }
  CloseGraph;              { Shut down the graphics system }
end;
{$F-}

{$F+}
function HeapErrProc(Size : Word) : Integer;
{ Custom Heap error function.  }
begin
  HeapErrProc := 1;
end;
{$F-}

procedure InitCharts;
{ Initialize graphics and report any errors that may occur }
var
  GraphDriver, GraphMode, ErrorCode : Integer;
begin
  OldExitProc := ExitProc;                { save previous exit proc }
  ExitProc := @ChartExitProc;             { insert our exit proc in chain }
  HeapError := @HeapErrProc;              { install heap error proc }

{$IFDEF Use8514}                          { check for Use8514 $DEFINE }
    GraphDriver := IBM8514;
    GraphMode := IBM8514Hi;
{$ELSE}
    GraphDriver := Detect;                { use autodetection }
{$ENDIF}
    InitGraph(GraphDriver, GraphMode, '');
    ErrorCode := GraphResult;             { preserve error return }
    if ErrorCode <> grOK then             { error? }
    begin
      Writeln('** Graphics Error: ', GraphErrorMsg(ErrorCode));
      WriteLn('** Progam halted.');
      Halt(1);
    end;
end;


{ initialization }

begin
 InitCharts;
end.