//  SOUND.CPP
//
//  C++ class to play digitized sounds through
//  the PC speaker.  This is a C++ wrapper for
//  John Ratcliff's routines presented in the
//  March 1993 issue of Dr. Dobbs Journal.
//
//  Kevin Dahlhausen        ap096@po.cwru.edu
//
//  Note:  Compile only in large memory model (for now.)

#include "sound.h"
#include <dos.h>
#include <io.h>
#include <fcntl.h>
#include <alloc.h>

extern "C" int  far CheckIn(void);   // Is sound driver available? 0 no, 1 yes.

const unsigned int SndBlkSize=65350;

Sound::Sound(char *SoundFile, int Frequency)
//  requires: Soundfile=name of soundfile to load, Frequency=playback rate
//  ensures:  Soundfile loaded and massaged
{
    SndInfo.freq=Frequency;
    SndInfo.next=0;
    ErrorStat=0;
    loadData(SoundFile);    // load raw data into memory
    if (!ErrorStat) massageData();
}

inline int Sound::error()
{ return ErrorStat; }

inline int Sound::driverLoaded()
// requires: none
// ensures: Sound::driverLoaded=0 if sound driver is not loaded
{
 return CheckIn();
}

void Sound::loadData(char *SoundFile)
// requires:  SoundFile name of file to load
// ensures:   as much of sound data into memory as possible
{

    SndStruc *b,*c;     // b=linked list cursor, c=potential next node
    long    ofs;

    int fhandle;
    if ((fhandle=open(SoundFile, O_RDONLY|O_BINARY))==-1) {
         ErrorStat=1;
         return;
    }

    ofs=1;

    // build the list
    b=&SndInfo;
    while((ofs+SndBlkSize<=filelength(fhandle))&&!ErrorStat) {
        c=new(SndStruc);
        c->next=0;
        c->s=ofs;
        c->SndLen=c->l=SndBlkSize;
        c->freq=SndInfo.freq;
        c->e=c->s+c->l-1;
        ofs+=SndBlkSize;
        if ((c->data=(unsigned char far *)farmalloc(c->l))==0) {
            ErrorStat=2;
            delete(c);
        } else {
            read(fhandle, c->data, c->l);
            b->next=c;
            b=b->next;
        }
    }
    if (!ErrorStat) {
        c=new(SndStruc);
        //b=b->next;
        c->next=0;
        c->s=ofs;
        c->SndLen=c->l=filelength(fhandle)-ofs+1;
        c->freq=SndInfo.freq;
        c->e=c->s+c->l-1;
        if ((c->data=(unsigned char far *)farmalloc(c->l))==0) {
            ErrorStat=2;
            delete(c);
        } else {
            read(fhandle, c->data, c->l);
            b->next=c;
        }
    }
    close(fhandle);
}

void Sound::massageData()
{
    SndStruc *b;
    struct REGPACK reg;

    b=SndInfo.next;
    while (b != 0) {
        reg.r_ax=0x68A;
        reg.r_ds=FP_SEG(b);
        reg.r_si=FP_OFF(b);
        intr(0x66, &reg);
        b=b->next;
    }
}

void Sound::play()
{
    SndStruc *b;
    struct REGPACK reg;

    b=SndInfo.next;
    while (b != 0) {
        reg.r_ax=0x68B;
        reg.r_ds=FP_SEG(b);
        reg.r_si=FP_OFF(b);
        intr(0x66, &reg);
        b=b->next;
    }
}

int Sound::isPlaying()
{
    struct REGPACK reg;
    reg.r_ax=0x689;
    intr(0x66, &reg);
    return reg.r_ax;
}

void Sound::stop()
{
    struct REGPACK reg;
    reg.r_ax=0x68f;
    intr(0x66, &reg);
}

Sound::~Sound()
{
    SndStruc *b, *c;

    b=SndInfo.next;
    while (b != 0) {
        c=b;
        b=b->next;
        farfree(c->data);
        delete(c);
    }
}