/*
// SNDDEMO.C
//
// Demo program for the Sefx sound effects library
//
// Copyright (c) 1992 Bri Productions
//
*/


#include "sefx.h"
#include <dos.h>
#include <bios.h>
#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <ctype.h>


/*
//-------------------------------------
//
// Function macros
//
//-------------------------------------
*/

#define  fgetw(a)    getw(a)
#define  fputw(a,b)  putw(a,b)

#define  INSERT      (((unsigned)*((unsigned far *)(0x400017l)) & 0x80) != 0)

#define  cursor(a)   _setcursortype(a)


/*
//-------------------------------------
//
// keystroke defines
//
//-------------------------------------
*/

#define  F1    0x3B00
#define  F2    0x3C00
#define  F3    0x3D00
#define  F4    0x3E00
#define  F5    0x3F00
#define  F6    0x4000
#define  F7    0x4100
#define  F8    0x4200
#define  F9    0x4300
#define  F10   0x4400
#define  ESC   0x011B
#define  UP    0x4800
#define  DOWN  0x5000
#define  LEFT  0x4B00
#define  RIGHT 0x4D00
#define  TAB   0x0F09
#define  STAB  0x0F00
#define  HOME  0x4700
#define  END   0x4F00
#define  PGUP  0x4900
#define  PGDN  0x5100
#define  ENTER 0x1C0D
#define  BS    0x0E08
#define  DEL   0x5300
#define  SPACE 0x3920


/*
//-------------------------------------
//
// screen colors
//
//-------------------------------------
*/

#define  VALUES   YELLOW|(LIGHTGRAY<<4)
#define  TEXT     BLUE|(LIGHTGRAY<<4)
#define  FIELD    YELLOW|(BLACK<<4)
#define  INFO     BLACK|(CYAN<<4)
#define  BORDER   DARKGRAY|(LIGHTGRAY<<4)
#define  TITLE    RED|(LIGHTGRAY<<4)


/*
//-------------------------------------
//
// screen coordinates
//
//-------------------------------------
*/

#define LMAR      21
#define TMAR      3
#define HSPC      5
#define VSPC      3
#define COL       5
#define SNDRES    72, TMAR + 9
#define SNDHOLD   71, TMAR + 12
#define SNDMS     LMAR + 9, 21


/*
//-------------------------------------
//
// Musical notes
//
//-------------------------------------
*/

#define  C     262
#define  D     294
#define  E     330
#define  F     349
#define  G     392
#define  A     440
#define  B     494


/*
//-------------------------------------
//
// misc constants
//
//-------------------------------------
*/

#define MAXFIELD  18
#define LARGE     (_SOLIDCURSOR )
#define SMALL     (_NORMALCURSOR)


/*
//-------------------------------------
//
// function prototype
//
//-------------------------------------
*/

void Init      (void);
void Uninit    (void);
void Highlight (struct field *_field, int on_off);
void UpdateMs  (void);
void delete    (char *del);
void insert    (char c, char *ins);
void blank     (char *blnk);
int  space     (char *c);


/*
//-------------------------------------
//
// global variables
//
//-------------------------------------
*/

TONE scale[] = { { C, 12 },
                 { D, 12 },
                 { E, 12 },
                 { F, 12 },
                 { G, 12 },
                 { A, 12 },
                 { B, 12 },
                 { C<<1, 12} };


static struct field{
   int value;           /* field value    */
   char *format;        /* printf format  */
   int row;             /* y coordinate   */
   int col;             /* x coordinate   */
   int next;            /* next field     */
   int prev;            /* previous field */
   int size;            /* field size     */
   }field[MAXFIELD] = {
      { 100, "%*u", TMAR + 0*VSPC, LMAR + 0 ,  1, 17, 4},  /* 0 SndTone    */
      {   3, "%*u", TMAR + 0*VSPC, LMAR + 6 ,  2,  0, 3},
      {  23, "%*u", TMAR + 1*VSPC, LMAR + 0 ,  3,  1, 3},  /* 2 SndPause   */
      {   1, "%*u", TMAR + 2*VSPC, LMAR + 11,  4,  2, 3},  /* 3 SndString  */
      { 100, "%*u", TMAR + 3*VSPC, LMAR + 0 ,  5,  3, 4},  /* 4 SndSlide   */
      {  -3, "%*d", TMAR + 3*VSPC, LMAR + 6 ,  6,  4, 3},
      {  18, "%*u", TMAR + 3*VSPC, LMAR + 11,  7,  5, 3},
      {   1, "%*u", TMAR + 3*VSPC, LMAR + 16,  8,  6, 3},
      {1400, "%*u", TMAR + 4*VSPC, LMAR + 0 ,  9,  7, 4},  /* 8 SndTrill   */
      { 600, "%*u", TMAR + 4*VSPC, LMAR + 6 , 10,  8, 4},
      {   1, "%*u", TMAR + 4*VSPC, LMAR + 12, 11,  9, 3},
      {   1, "%*u", TMAR + 4*VSPC, LMAR + 17, 12, 10, 3},
      {  18, "%*u", TMAR + 4*VSPC, LMAR + 22, 13, 11, 3},
      {1100, "%*u", TMAR + 5*VSPC, LMAR + 0 , 14, 12, 4},  /* 13 SndPulse  */
      {   2, "%*u", TMAR + 5*VSPC, LMAR + 6 , 15, 13, 3},
      {   9, "%*u", TMAR + 5*VSPC, LMAR + 11, 16, 14, 3},
      {   8, "%*u", TMAR + 5*VSPC, LMAR + 16, 17, 15, 3},
      {1000, "%*u", TMAR + 6*VSPC, LMAR + 0 ,  0, 16, 5}   /* 17 SndMs     */
   };


int x, y;                     /* cursor location   */
int tbase = 3;
char *tbase_txt[] = { "ultra ",
                      " high ",
                      "medium",
                      " low  "
                    };
int hold = OFF;
char *hold_txt[2] = { "OFF",
                      "ON "
                      };


/*
//-------------------------------------
//
// main()
//
//-------------------------------------
*/

int main(void)
{
register struct field *_field = field;    /* current field pointer   */
int pos = 0;                              /* cursor field location   */
int key;                                  /* keystroke               */
char buf[17];                             /* field buffer            */
long len, old_len = 1;                    /* sound queue length      */


      /* Call the initialization function and   */
      /* initialize the field buffer            */

   Init();
   sprintf(buf, _field->format, _field->size, _field->value);


      /* Monitor and update, if necessary, the sound queue status */
      /* and length while waiting for a keystroke.                */

   while(1)
   {

      textattr(VALUES);
      UpdateMs();
      while(!bioskey(1))
      {

            /* If the sound queue length has changed  */
            /* update the screen.                     */

         len = SndLen();
         if(len != old_len)
         {
            cursor(OFF);
            gotoxy(72, TMAR);
            cprintf("%-7lu", len);
            old_len = len;
         }


            /* Restore the cursor before  */
            /* processing the keystroke  */

         gotoxy(x, y);
         if(INSERT)
            cursor(LARGE);
         else
            cursor(SMALL);

      }
      textattr(FIELD);


         /* Save a copy of the keystroke and process it */

      cursor(OFF);
      switch(key = bioskey(0))
      {

         case ENTER:
         case TAB:
         case DOWN:
            _field->value = atoi(buf);
            Highlight(_field, OFF);
            _field = field + _field->next;
            sprintf(buf, _field->format, _field->size, _field->value);
            Highlight(_field, ON);
            pos = 0;
            break;


         case STAB:
         case UP:
            _field->value = atoi(buf);
            Highlight(_field, OFF);
            _field = field + _field->prev;
            sprintf(buf, _field->format, _field->size, _field->value);
            Highlight(_field, ON);
            pos = 0;
            break;



            /* home: go to the first character in the field */

         case HOME:
            x -= pos;
            pos = 0;
            gotoxy(x, y);
            break;


            /* end: go to the last character in the field */

         case END:
            x += _field->size - pos - 1;
            pos = _field->size - 1;
            gotoxy(x, y);
            break;


            /* page up: go to the first field */

         case PGUP:
            Highlight(_field, OFF);
            _field = field;
            sprintf(buf, _field->format, _field->size, _field->value);
            x = _field->row;
            y = _field->col;
            pos = 0;
            Highlight(_field, ON);
            break;


            /* page down: go to the last field */

         case PGDN:
            Highlight(_field, OFF);
            _field = field + MAXFIELD - 1;
            sprintf(buf, _field->format, _field->size, _field->value);
            x = _field->row;
            y = _field->col;
            pos = 0;
            Highlight(_field, ON);
            break;


            /* delete: delete the character at the cursor */

         case BS:
         case LEFT:
            if(pos)
            {
               pos--;
               x--;
               gotoxy(x, y);
            }
            else
               break;

            if(!INSERT || key == LEFT)
               break;

         case DEL:
            delete(buf + pos);
            cputs(buf + pos);
            _field->value = atoi(buf);
            break;


            /* right arrow: goto the next space */

         case SPACE:
         case RIGHT:
            if(pos < _field->size - 1)
            {
               pos++;
               x++;
               gotoxy(x, y);
            }
            break;



            /* process numeric characters, spaces and minus signs */
            /* (falls through)                                     */

         default:
            if(isdigit((char)key)
               || ((char)key == '-' && *(_field->format + 2) == 'd'))
            {
               x -= pos;
               pos -= space(buf + pos);
               x += pos;
               gotoxy(x, y);

               if(INSERT)
                  insert(key, buf + pos);
               else
                  *(buf + pos) = key;

               if(*(buf + pos + 1) == ' ')
                  blank(buf + pos + 2);

               _field->value = atoi(buf);
               cputs(buf + pos);

               if(pos < _field->size - 1)
               {
                  pos++;
                  x++;
               }
               gotoxy(x, y);
            }
            break;




            /* f1: load a tone into the sound queue */

         case F1:
            SndTone((field + 0)->value, (field + 1)->value);
            break;


            /* f2: load a pause into the sound queue */

         case F2:
            SndPause((field + 2)->value);
            break;


            /* f3: load a string of notes into the sound queue */

         case F3:
            SndString(scale, 8, (field + 3)->value);
            break;


            /* f4: load a sliding tone into the sound queue */

         case F4:
            SndSlide((field + 4)->value, (field + 5)->value,
                     (field + 6)->value, (field + 7)->value);
            break;


            /* f5: load a trill tone into the sound queue */

         case F5:
            SndTrill((field + 8)->value, (field + 9)->value,
                (field + 10)->value, (field + 11)->value, (field + 12)->value);
            break;


            /* f6: load a interval tone into the sound queue */

         case F6:
            SndPulse((field + 13)->value, (field + 14)->value,
                                    (field + 15)->value, (field + 16)->value);
            break;


            /* f7: flush the sound queue */

         case F7:
            SndFlush();
            break;


         case F8:
            tbase += 3;
            tbase &= 3;
            gotoxy(SNDRES);
            textattr(VALUES);
            cputs(*(tbase_txt + tbase));
            textattr(FIELD);
            SndTbase(tbase);
            break;

         case F9:
            SndHold(hold ^= 1);
            gotoxy(SNDHOLD);
            textattr(VALUES);
            cputs(*(hold_txt + hold));
            textattr(FIELD);
            break;

         case F10:
            SndTerm();
            break;

            /* escape: quit snddemo */

         case ESC:
            Uninit();
            return(0);

      }
   }
}



/*
//-------------------------------------
//
// Initialize
//
//-------------------------------------
*/

void Init(void)
{
int i = MAXFIELD;
int row = TMAR;
struct field *_field;
FILE *ini;
struct{
 int x;
 int y;
 char *text;
 }title[3] = {  {  5,  1, "Sound Generation"  },
       { 56,  1, "Status"     },
      { 56,  7, "Control"     }
     };

char *function[] = {

   "F1 - SndTone  (     ,    )                              SndLen() =  ",
   "F2 - SndPause (    )",
   "F3 - SndString(scale,   8,    )                     F7  - SndFlush()",
   "F4 - SndSlide (     ,    ,    ,    )                F8  - SndTbase(      )",
   "F5 - SndTrill (     ,     ,    ,    ,    )          F9  - SndHold(   )",
   "F6 - SndPulse (     ,    ,    ,    )                F10 - SndTerm() ",
   "         SndMs(      ) =                            ESC- Quit"
                  };

      /* Initialize sefx with a 1023 byte sound queue (341 tones)  */

   SndOpen(1023, 3);


      /* Read the initialization file if one exists   */

   ini = fopen("snddemo.ini", "rb");
   if(ini)
   {
      i = MAXFIELD;
      while(i--)
         (field + i)->value = fgetw(ini);
      fclose(ini);
   }


      /* Clear the screen */

   textattr(TEXT);
   clrscr();


      /* Put the function names on the screen */

   i = 0;
   while(i < 7)
   {
      gotoxy(COL, row);
      row += VSPC;
      cputs(*(function + i++));
   }


      /* Put the values on the screen and */
      /* and highlight the first field    */

   textattr(VALUES);
   gotoxy(SNDRES);
   cputs(*(tbase_txt + tbase));
   _field = field + 1;
   for(i=1; i<MAXFIELD; i++)
   {
      gotoxy(_field->col, _field->row);
      cprintf(_field->format, _field->size, _field->value);
      _field++;
   }
   gotoxy(SNDHOLD);
   textattr(VALUES);
   cputs(*hold_txt);
   textattr(FIELD);
   Highlight(field, ON);



  /* Draw the borders */

 gotoxy(1, 1);
 textattr(BORDER);
   cputs(" [                  ]"
         "[        ]͸" );
   for(i=2; i<24; i++)
   {
  if(i == 7)
  {
   cputs("                                                  "
       "  [         ]͵");
   continue;
  }
      gotoxy(2, i);
      putch('');
      gotoxy(53, i);
      putch('');
      gotoxy(79, i);
      putch('');
   }
   cputs("  "
         ";" );


 textattr(TITLE);
 for(i=0; i<3; i++)
 {
  gotoxy((title + i)->x, (title + i)->y);
  cputs((title + i)->text);
 }



      /* Put the footer on the screen */

   gotoxy(2,25);
   textattr(INFO);
   cputs("           Sefx version 1.0  -  Copyright (c) 1992 Bri Productions           ");


      /* locate the cursor */

   gotoxy(x, y);
}



/*
//-------------------------------------
//
// Uninitialize
//
//-------------------------------------
*/

void Uninit(void)
{
int i = MAXFIELD;
FILE *ini;


      /* clear the screen */

   textattr(LIGHTGRAY);
   clrscr();
   SndClose();


      /* write the values to the initialization file */

   ini = fopen("snddemo.ini", "wb");
   if(ini)
   {
      while(i--)
         fputw((field + i)->value, ini);
      fclose(ini);
   }


      /* put the trailing message on the screen */

   puts("Sefx version 1.0  -  Copyright (c) 1992 Bri Productions");
   cursor(SMALL);
}


/*
//-------------------------------------
//
// highlight or un-highlight a field
//
//-------------------------------------
*/

void Highlight(struct field *_field, int on_off)
{
int attr[2] = { VALUES, FIELD };


      /* Locate the cursor and set the text color */

   x = _field->col;
   y = _field->row;
   gotoxy(x, y);
   textattr(*(attr + on_off));


      /* reput the value in the new attribute */

   cprintf(_field->format, _field->size, _field->value);


      /* restore the text attribute and cursor location  */

   textattr(FIELD);
   gotoxy(x, y);
}


/*
//-------------------------------------
//
// update milliseconds to tics
//
//-------------------------------------
*/

void UpdateMs(void)
{
int tics;

   tics = SndMs((field + 17)->value);
   gotoxy(SNDMS);
   cprintf("%-4u", tics);

}


/*
//-----------------------------------------------------------------------------
//
// insert
//
// inserts a character into a string
//
//-----------------------------------------------------------------------------
*/

void insert(char c, char *ins)
{
char temp;

   do
   {
      temp = *ins;
      *ins++ = c;
      c = temp;
   }
   while(*ins);
}

/*
//-----------------------------------------------------------------------------
//
// delete
//
// deletes a character from a string
//
//-----------------------------------------------------------------------------
*/

void delete(char *del)
{
   while(*(del + 1))
      *del++ = *(del + 1);
   *del = ' ';
}


/*
//-----------------------------------------------------------------------------
//
// blank
//
// blanks a string
//
//-----------------------------------------------------------------------------
*/

void blank(char *blnk)
{
   while(*blnk)
      *blnk++ = ' ';
}


/*
//-----------------------------------------------------------------------------
//
// space
//
// counts the number of leading spaces
//
//-----------------------------------------------------------------------------
*/

int space(char *c)
{
int rv = 0;

   while(*--c == ' ')
      rv++;
   return(rv);
}
