/*
    Module: cfg_io.c - (cfg_io.h)
        Configuration File IO routines
                                                                  
    Copyright (c) 1991-92 by Jorgen Sven Abrahamsen (2:230/100.9@fidonet)

    Functions:
        ropencfg      - Open cfg file for reading
        closecfg      - Close cfg file
        cfg_findfirst - Find first occurrence of a tag in a section
        cfg_findnext  - Find next occurrence of a tag in a section
        free_cfgtoken - Free linked list of tokens

    Caveats:
        Sections must have a SECTION_CB as first character, and SECTION_CE
            as last character.
        Subtags must have a OPTION_C as first character.
        Comments are denoted by a COMMENT_C and continue to end of current 
            line. COMMENT_C's inside quotes or escaped (with \) are ignored.
        SECTION_CB, SECTION_CE, OPTION_C, COMMENT_C are defined in cfg_io.h.

    Programmer : JSA / Cirrus  27-01-1991  Ver. 1.0
*/
/*---------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cfg_io.h"

/*---------------------------------------------------------------------------*/
/*
    Internal prototypes
*/
static int read_token(FILE *file, char *buf);
static int read_option(FILE *file, char *buf);

/*---------------------------------------------------------------------------*/
/*
    Open a cfg file for reading. Returns a cfg file struct if successful.
*/
typeCfgFile * ropencfg(const char *name)
{
    typeCfgFile *cf = (typeCfgFile *) malloc(sizeof(typeCfgFile));

    if (cf != NULL)
        {
        cf->file = fopen(name, "rt");
        if (cf->file == NULL)
            {
            free (cf);
            cf = NULL;
            }
        else
            cf->entry = NULL;
        }
    return (cf);
}

/*---------------------------------------------------------------------------*/
/*
    Close a cfg file and deallocate cfg file struct.
*/
void closecfg(typeCfgFile *cf)
{
    fclose(cf->file);
    free(cf);
}

/*---------------------------------------------------------------------------*/
/*
    Find first occurrence of entry in section. Return allocated linked list
    of tokens if successful, NULL otherwise.
    Comments, leading/trailing spaces are removed.
    A pointer to cfg entry list is saved in cfg struct - if entry list is
    changed before a call to cfg_findnext, the behaviour of cfg_findnext
    is undefined.
*/
typeCfgToken * cfg_findfirst(typeCfgFile *cf, const char *section, const typeCfgEntry *entry)
{
    int found = 0;
    size_t len;
    char buf[MAX_CFG_TOKEN_LEN];

    rewind(cf->file);
    cf->entry = entry;
    while (!found && read_token(cf->file, buf))     /* search for section */
        found = (*buf == SECTION_CB) &&
            (buf[len = strlen(buf) - 1] == SECTION_CE) &&
            !strnicmp(section, buf + 1, len - 1);

    return (found ? cfg_findnext(cf) : NULL);
}

/*---------------------------------------------------------------------------*/
/*
    Find next occurrence of entry in section. Return allocated linked list
    of tokens if successful, NULL otherwise.
    Comments, leading/trailing spaces, quotes, escapes are removed.
    A pointer to cfg entry list is saved in cfg struct.
    If cfg_findnext is called without a preceding call to cfg_findfirst,
    or after a previous call to cfg_findnext has returned NULL, the
    behaviour of cfg_findnext is undefined.
*/
typeCfgToken * cfg_findnext(typeCfgFile *cf)
{
    char buf[MAX_CFG_TOKEN_LEN];
    typeCfgToken *tail, *head = NULL;
    int error = 0, found = 0;
    size_t len = 0;

    while (!found && read_token(cf->file, buf))
        {
        len = strlen(cf->entry->tag);
        found = (*buf != OPTION_C) && !strnicmp(cf->entry->tag, buf, len);
        }
    if (found)
        {
        head = tail = (typeCfgToken *) malloc(sizeof (typeCfgToken));
        error = (head == NULL);
        if (!error)
            {
            head->next = NULL;
            head->id = cf->entry->id;
            head->token = strdup(buf + len);
            error = (head->token == NULL);
            }
        while (!error && read_option(cf->file, buf))
            {
            const typeCfgEntry *centry = cf->entry;
            found = 0;
            while (!found && (len = strlen((++centry)->tag)) != 0)
                found = !strnicmp(centry->tag, buf, len);
            if (found)
                {
                tail->next = (typeCfgToken *) malloc(sizeof (typeCfgToken));
                error = (tail == NULL);
                if (!error)
                    {
                    tail = tail->next;
                    tail->next = NULL;
                    tail->id = centry->id;
                    tail->token = strdup(buf + len);
                    error = (tail->token == NULL);
                    }
                }
            }
        }
    if (error)
        free_cfgtoken(head);
    return (head);
}

/*---------------------------------------------------------------------------*/
/*
*/
void free_cfgtoken(typeCfgToken *head)
{
    while (head != NULL)
        {
        typeCfgToken *dummy = head->next;
        if (head->token != NULL)
            free(head->token);
        free(head);
        head = dummy;
        }
}

/*---------------------------------------------------------------------------*/
/*
    Read a token, stop at end of token, skip leading whitespace.

    Whitespace = spaces (except inside quotes or escaped), newlines,
        comments, EOF.
    Comments = led off by COMMENT_C (except inside quotes or escaped),
        terminated by newline.

    Return length of token or 0 if none found.
*/
static int read_token(FILE *file, char *buf)
{
    int c, comment, escape, found, quote;
    char *sp = buf;

    comment = escape = found = quote = 0;
    do
        {
        c = fgetc(file);
        comment = (c != '\n') &&    /* toggle comment flag */
            (comment || (!escape && !quote && c == COMMENT_C));

        found = (c == EOF) || ((sp != buf) &&       /* set found flag */
            ((c == '\n') || comment ||
            ((c == ' ')  && !escape && !quote) ||
            ((c == '\"') && !escape && quote)));

        if (!found && !comment && (c != '\n') &&    /* add a char to buffer */
            (c != '\\' || escape) &&
            (c != '\"' || escape) &&
            (c != ' '  || quote))
            *sp++ = (char) c;
                            
        if (!escape && (c == '\"'))     /* Toggle quote if valid \" */
            quote = !quote;
        escape = (c == '\\');           /* Set escape for next pass */
        }
    while (!found);
    *sp = '\0';     /* null terminate token */

    return (sp - buf);
}

/*---------------------------------------------------------------------------*/
/*
    Read an option = a token led off by an OPTION_C.
    Return length of token or 0 if none found.
*/
static int read_option(FILE *file, char *buf)
{
    fpos_t fp;
    int len;
    fgetpos(file, &fp);
    len = read_token(file, buf);
    if (*buf != OPTION_C)
        {
        fsetpos(file, &fp);
        len = 0;
        }
    else if (len != 0)
        {
        strcpy(buf, buf + 1);
        --len;
        }
    return (len);
}
