/*
    Module: cnv.c
        Main Module for Configurable Recursive Archive Converter

    Copyright (c) 1991-92 by Jorgen Sven Abrahamsen (2:230/100.9@fidonet)

    Please report bugs comments etc. to me

    Licensing:
        This program is freeware - you don't have to pay anything for
        private use. I would appreciate receiving a postcard though.
        The address is:
                J.S.Abrahamsen
                Lynge Bygade 33b
                DK-3540 Lynge
                DENMARK

        I have decided to include full source code (naive, huh?).
        You may use the source freely (in any way you want), BUT you
        must acknowledge the source of the code used (me), either in
        your documentation, or in a credits screen. I would like a
        copy of any improvements.

    Programmer : JSA / Cirrus  25-01-1991  Ver. 1.00
*/
/*---------------------------------------------------------------------------*/
/*
    Includes
*/
#include <conio.h>
#include <ctype.h>
#include <direct.h>
#include <dos.h>
#include <io.h>
#include <process.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef SWAP
#include <spawno.h>     /* Ralf Brown's Spawno library */
/* #define SWAPCFG    *//* enable swap configuration for spawno < v4.0 */
#endif

#include "cfg_io.h"     /* My configuration stuff */
#include "ms_dir.h"     /* My directory stuff - requires dos.h */

/*---------------------------------------------------------------------------*/
/*
    Defines
*/
#define PRGM_NAME       "CNV"
#define PRGM_VERS       "1.00"
#define ENV_NAME        "CNVCFG"

#define FALSE           0
#define TRUE            1
#define MAXLINELENGTH   255
#define SIGN_LEN        16
#define DONT_CARE       -2
#define EOSIGN          -1
#define LOG_LOW         0
#define LOG_DEBUG       255
#define LOG_STDERR      256

/*---------------------------------------------------------------------------*/
/*
    Typedefs
*/
typedef struct Packer typePacker;

struct Packer {     /* Packer program definition */
    typePacker *next;
    char *name;
    char *cmdUnpack;
    char *cmdPack;
    char *suffix;
    int  signature[SIGN_LEN];
    int  signOffset;
    int  success;
    };

typedef struct Tester typeTester;

struct Tester {     /* File tester program definition */
    typeTester *next;
    char *name;
    char *cmdLine;
    int success;
    };

typedef struct {                /* Configuration                    */
    char        *file;          /* cfg file                         */
    char        *destname;      /* destination packer suffix        */
    typePacker  *dest;          /* destination packer               */
    typePacker  *packers;       /* list of packers                  */
    typeTester  *testers;       /* list of testers                  */
#ifdef SWAPCFG
    char        *swappath;      /* swap path for spawn              */
#endif
    int         noRecursion;    /* Disables recursive conv. (def 0) */
    int         loglevel;       /* log / debug output level         */
    FILE        *logfile;       /* log / debug info file!           */
    } typeCfg;

/*---------------------------------------------------------------------------*/
/*
    Globals
*/
typeCfg cfg;            /* Global configuration information         */
int stop_program;       /* Error semaphore - stop execution         */
char *mem_alloc_err =   /* Error text for memory allocation error   */
    "Memory allocation error. That's right, we ran out of memory!\n";

/*---------------------------------------------------------------------------*/
/*
    Prototypes
*/
static int configure(char **cargv);
static int convert(const typeDir *wilds);
static const typePacker *test_for_packer(const typePacker *pl, const char *fn);
static int unpack(const typePacker *p, const char *file);
static int test(const typeTester *tl, const char *path);
static int pack(const typePacker *p, const char *path);
static int os_shell(char *cmd);
static void _cdecl ostatus(char *logstr, ...);
static void _cdecl oerror(char *logstr, ...);
static void _cdecl olog(int loglevel, char *logstr, ...);
static void _cdecl ctrl_c_handler(void);
static void test_for_escape(void);
static int xctoi(char c);

/*---------------------------------------------------------------------------*/
/*
    You guessed it! The main() function
*/
int _cdecl main( int argc, char **argv )
{
    extern typeCfg cfg;

    int i, error = FALSE;
    typeDir *wilds = open_dir();

    oerror(PRGM_NAME " - A Configurable, Recursive Archive Conversion Utility.\n" \
        "Copyright (c) 1991 J.S.Abrahamsen (2:230/100.9@fidonet.org).\n" \
        PRGM_NAME " -help for usage information.\n");

    /* parse the command line */

    for (i = 1; i < argc && !error; i++ )
        {   /* parse switch */
        if (argv[i][0] == '-') switch( toupper(argv[i][1]) )
            {
            case 'C':   /* Set configuration file name */
                error = (cfg.file = strdup( argv[i] + 2 )) == NULL;
                if (error)
                    oerror(mem_alloc_err);
                break;
            case 'D':   /* Set Destination archive name */
                error = (cfg.destname = strdup( argv[i] + 2 )) == NULL;
                if (error)
                    oerror(mem_alloc_err);
                break;
            case 'N':   /* Disable recursion */
                cfg.noRecursion = TRUE;
                break;
            case 'H':   /* Display Help */
                oerror(
                    "\n" PRGM_NAME " version " PRGM_VERS " / compiled " __DATE__ " " __TIME__ "\n\n"
                    "Usage: " PRGM_NAME " [-c<cfg file>] [-d<packer>] [-norecurse] [-help] [files ...]\n" \
                    "    files ...   : Specify which files to convert using DOS wildcards.\n" \
                    "                  Default is to convert all files in the current directory.\n" \
                    "                  Multiple file specifications are allowed, for example:\n" \
                    "                  \"cnv test.* dum??.* *.lzh\".\n" \
                    "    -c<cfg file>: Specifies a configuration file other than " PRGM_NAME ".cfg.\n" \
                    "    -d<packer>  : Selects a destination archive format other than that\n" \
                    "                  specified in the configuration file, for example:\n" \
                    "                  \"cnv -dLZH\" or \"cnv -dZIP\".\n" \
                    "    -n[orecurse]: Disables recursive conversion of archives in archives.\n" \
                    "    -h[elp]     : This screen!\n");
                return (0);
                /* break; */
            case 'Z':   /* Debug */
                cfg.loglevel = atoi(argv[i] + 2);
                break;
            default:
                oerror("Illegal command line switch: '%s' Naughty!\n", argv[i]);
                error = TRUE;
            }
        else
            {   /* add wildcard to list */
            error = strchr(argv[i], '\\') != NULL;
            if (error)
                oerror("Sorry - only files in the current directory " \
                    "can be converted!\n");
            else
                add_dir_entry(wilds, argv[i], 0xffff, 0L);
            }
        }

    if (!error)
        error = configure(argv);

    /* disable [CTRL-C] */

    error = signal(SIGINT, ctrl_c_handler) == SIG_ERR;
    if (error)
        oerror("Could not set signal!\n");

    if (!error)
        {
        int count = convert((wilds->head == NULL) ? ALLFILES : wilds);
        if (stop_program)
            {
            oerror("\n" PRGM_NAME " interrupted by user!\n");
            error = 255;
            }
        else if (count == 0)
            oerror("\nNo Archives found!\n");
        }

    if (error)
        oerror("Error=%d\n", error);
    if (cfg.logfile != NULL)
        fclose(cfg.logfile);

    return (error);
}

/*---------------------------------------------------------------------------*/
/*
    Configuration file tags
*/
/*lint +fie */  /* Use loose enum checking (int) */
enum cfgtag {
    PACKER,     UNPACK,     PACK,       SUFFIX,     SIGN,       OFFSET,
    SUCCESS,    TESTER,     CMDLINE,    DEST,       LOGFILE,    LOGLEVEL,
#ifdef SWAPCFG
    SWAPPATH,   NOXMS,      NOEMS,
#endif
    NULLENTRY
    };
typeCfgEntry null_entry = {NULLENTRY, "" };
typeCfgEntry cpackers[] = {
    {PACKER,    "Packer="       },
    {UNPACK,    "Unpack="       },
    {PACK,      "Pack="         },
    {SUFFIX,    "Suffix="       },
    {SIGN,      "Signature="    },
    {OFFSET,    "Offset="       },
    {SUCCESS,   "Success="      },
    {NULLENTRY, ""              }
    };
typeCfgEntry ctesters[] = {
    {TESTER,    "Tester="       },
    {CMDLINE,   "CmdLine="      },
    {SUCCESS,   "Success="      },
    {NULLENTRY, ""              }
    };
typeCfgEntry cdest[] = {
    {DEST,      "Destination="  },
    {NULLENTRY, ""              }
    };
typeCfgEntry clogfile[] = {
    {LOGFILE,   "LogFile="      },
    {LOGLEVEL,  "LogLevel="     },
    {NULLENTRY, ""              }
    };
#ifdef SWAPCFG
typeCfgEntry cswappath[] = {
    {SWAPPATH,  "SwapPath="     },
    {NOXMS,     "NoXMS"         },
    {NOEMS,     "NoEMS"         },
    {NULLENTRY, ""              }
    };
#endif

/*---------------------------------------------------------------------------*/
/*
    Read the configuration file
*/
static int configure(char **cargv)
{
    extern typeCfg cfg;
    int error = FALSE;
    char sp[MAXLINELENGTH];
    char *spp;
    unsigned ui;
    typeCfgFile *cfgfile;
    typeCfgToken *head, *work;

    /* Find configuration file */

    if (cfg.file == NULL)
        {
        /* Find home directory */

        if ((spp = getenv(ENV_NAME)) != NULL)
            {   /* From environment */
            strcpy(sp, spp);
            ui = strlen(sp);
            if (*(sp + ui - 1) == '\\')
                *(sp + ui - 1) = '\0';
            }
        else
            {   /* From Command line */
            strcpy(sp, cargv[0]);
            if ((spp = strrchr(sp, '\\')) != NULL)
                *spp = '\0';
            }
        strcat(sp, "\\" PRGM_NAME ".cfg");
        error = (cfg.file = strdup(sp)) == NULL;
        }

    if (!error && ((cfgfile = ropencfg(cfg.file)) == NULL))
        {
        oerror("No configuration file found!\n");
        return (TRUE);
        }

    /* Read Archiver cfg info */

    if ((head = work = cfg_findfirst(cfgfile, PRGM_NAME, cpackers)) != NULL) do
        {
        typePacker *pp;
        if (cfg.packers == NULL)
            pp = cfg.packers = (typePacker *) calloc(1, sizeof(typePacker));
        else
            {
            pp->next = (typePacker *) calloc(1, sizeof(typePacker));
            pp = pp->next;
            }
        pp->next = NULL;
        pp->signature[0] = EOSIGN;  /* default to no test */
        pp->name = work->token;     /* Name */
        work->token = NULL;
        olog(LOG_DEBUG, "\nPacker = %s\n", pp->name);
        while ((work = work->next) != NULL) switch (work->id)
            {
            case UNPACK:
                pp->cmdUnpack = work->token;
                work->token = NULL;
                olog(LOG_DEBUG, "  Unpack: \"%s\"\n", pp->cmdUnpack);
                break;
            case PACK:
                pp->cmdPack = work->token;
                work->token = NULL;
                olog(LOG_DEBUG, "  Pack  : \"%s\"\n", pp->cmdPack);
                break;
            case SUFFIX:
                pp->suffix = work->token;
                work->token = NULL;
                olog(LOG_DEBUG, "  Suffix: \"%s\"\n", pp->suffix);
                error = strlen(pp->suffix) > 3;
                if (error)
                    oerror("Illegal suffix: '%s'. Too long!\n", pp->suffix);
                break;
            case SIGN:
                ui = 0;
                spp = work->token;
                if (*spp == '0' && toupper(*(spp + 1)) == 'X')
                    {                                   /* hex string */
                    spp += 2;
                    while (*spp && ui < SIGN_LEN - 1)
                        {
                        if (*spp == '?')    /* a '?' = don't care */
                            pp->signature[ui++] = DONT_CARE;
                        else                /* ordinary hex */
                            {   /* this keeps giving me problems when written
                                    as a single line. Don't ask why! */
                            /* pp->signature[ui++] =
                                xctoi(*spp) << 4 + xctoi(*(spp + 1)); */
                            int temp = xctoi(*spp) << 4;
                            pp->signature[ui] = temp;
                            temp = xctoi(*(spp + 1));
                            pp->signature[ui++] += temp;
                            }
                        if (*++spp) ++spp;
                        }
                    } 
                else while (*spp && ui < SIGN_LEN - 1)  /* ASCII */
                    if (*spp != '?')        /* ordinary char */
                        pp->signature[ui++] = *spp++ & 0x00ff;
                    else if (*++spp == '?') /* literal ? = '??' */
                        pp->signature[ui++] = *spp++ & 0x00ff;   
                    else                    /* ? = don't care */
                        pp->signature[ui++] = DONT_CARE;
                pp->signature[ui] = EOSIGN;
                olog(LOG_DEBUG, "  Signature:");
                ui = 0;
                do  olog(LOG_DEBUG, " %x", pp->signature[ui]);
                while (pp->signature[++ui] != EOSIGN);
                break;
            case OFFSET:
                pp->signOffset = atoi(work->token);
                olog(LOG_DEBUG, "\n  Offset  = %d", pp->signOffset);
                break;
            case SUCCESS:
                pp->success = atoi(work->token);
                olog(LOG_DEBUG, "\n  Success = %d\n", pp->success);
                break;
            default:    break;
            }
        error = (pp->cmdPack != NULL) && (pp->suffix == NULL);
        if (error)
            oerror("Suffix missing for '%s'.\n", pp->name);
        free_cfgtoken(head);
        }
    while (!error && ((head = work = cfg_findnext(cfgfile)) != NULL));

    /* Read Tester cfg info */

    if ((head = work = cfg_findfirst(cfgfile, PRGM_NAME, ctesters)) != NULL) do
        {
        typeTester *tp;
        if (cfg.testers == NULL)
            tp = cfg.testers = (typeTester *) calloc(1, sizeof(typeTester));
        else
            {
            tp->next = (typeTester *) calloc(1, sizeof(typeTester));
            tp = tp->next;
            }
        tp->next = NULL;
        tp->name = work->token;     /* Name */
        work->token = NULL;
        olog(LOG_DEBUG, "\nTester = %s\n", tp->name);
        while ((work = work->next) != NULL) switch (work->id)
            {
            case CMDLINE:
                tp->cmdLine = work->token;
                work->token = NULL;
                olog(LOG_DEBUG, "  Cmdline: \"%s\"\n", tp->cmdLine);
                break;
            case SUCCESS:
                tp->success = atoi(work->token);
                olog(LOG_DEBUG, "  Success = %d\n", tp->success);
                break;
            default:    break;
            }
        free_cfgtoken(head);
        }
    while (!error && ((head = work = cfg_findnext(cfgfile)) != NULL));

    /* Read Destination cfg info */

    error = error ||
        (head = work = cfg_findfirst(cfgfile, PRGM_NAME, cdest)) == NULL;
    if (error)
        oerror("No default destination in configuration file!\n");
    else
        {       /* Find destination */
        typePacker *p = cfg.packers;
        char *destname = work->token;
        work->token = NULL;
        cfg.dest = NULL;    /* set dest to impossible value */
        if (p != NULL) do
            {
            if ((cfg.destname != NULL) &&
                (stricmp(cfg.destname, p->name) == 0))
                {   /* Command line destination found - override */
                cfg.dest = p;
                break;
                }
            if (stricmp(destname, p->name) == 0)
                cfg.dest = p;
            }
        while ((p = p->next) != NULL);
        error = (cfg.dest == NULL) || (cfg.dest->cmdPack == NULL);
        if (error)
            oerror("No valid destination found!\n");
        olog(LOG_DEBUG, "\nDestination = %s\n", destname);
        free_cfgtoken(head);
        }

    /* Read Logfile cfg info */

    if (!error &&
        (head = work = cfg_findfirst(cfgfile, PRGM_NAME, clogfile)) != NULL)
        {
        error = (cfg.logfile = fopen(work->token, "at+")) == NULL;
        if (error)
            oerror("Could not open log file!\n");
        olog(LOG_DEBUG, "\nLogFile = %s", work->token);
        if (((work = work->next) != NULL) && (work->id == LOGLEVEL))
            cfg.loglevel = max(atoi(work->token), cfg.loglevel);
        olog(LOG_DEBUG, " level = %d\n", cfg.loglevel);
        free_cfgtoken(head);
        }

#ifdef SWAPCFG
    /* Read SwapPath cfg info */

    if (!error &&
        (head = work = cfg_findfirst(cfgfile, PRGM_NAME, cswappath)) == NULL)
        {
        cfg.swappath = work->token;
        work->token = NULL;
        while ((work = work->next) != NULL)
            if (work->id == NOXMS)
                __spawn_xms = 0;
            else if (work->id == NOEMS)
                __spawn_ems = 0;
        olog(LOG_DEBUG, "\nSwapPath = %s%s%s\n", cfg.swappath,
            __spawn_xms ? "" : ", NoXMS", __spawn_ems ? "" : ", NoEMS");
        free_cfgtoken(head);
        }
#endif

    if (cfgfile != NULL)
        closecfg(cfgfile);

    return (error);
}

/*---------------------------------------------------------------------------*/
/*
    Convert all files in current directory matching list of wildcards
    Return total number of conversions, including recursed ones.
*/
static int convert(const typeDir *wilds)
{
    extern typeCfg cfg;
    extern int stop_program;
    static int recursion_level = 0;
    int error = FALSE;
    int count = 0;
    fpos_t tosz = 0;
    fpos_t tnsz = 0;
    typeDir       *dir = open_dir();    /* Build list of matching files */
    typeDirEntry *file = read_dir(dir, wilds);

    test_for_escape();

    if (!stop_program && !error && file != NULL) do
        {                           /* step thru files and convert 'em  */
        const typePacker *packer = NULL;

        if (!(file->attr & _A_SUBDIR) &&    /* skip testing dirs        */
            ((packer = test_for_packer(cfg.packers, file->name)) != NULL))
            {                               /* It's packed - convert !  */
            fpos_t osz, nsz;
            time_t tt;
            struct tm *ts;
            char newname[16];
            char *sp, *tempdir = "CPXXXXXX";

            osz = file->size;
            tosz += osz;
            tt = time(NULL);
            ts = localtime(&tt);
            if (recursion_level == 0)
                olog(LOG_LOW, PRGM_NAME \
                    " %04d-%02d-%02d %02d:%02d:%02d : %-12s/%8ldb -> ",
                    ts->tm_year + 1900, ts->tm_mon + 1, ts->tm_mday,
                    ts->tm_hour, ts->tm_min, ts->tm_sec, file->name, osz);

            sprintf(newname, "..\\%s", file->name); /* Add prefix '..\' */
            mktemp(tempdir);                /* Make temporary subdir    */
            mkdir(tempdir);
            chdir(tempdir);                 /* Move self to subdir      */
            rename(newname, file->name);    /* Move file to subdir      */

            error = unpack(packer, file->name); /* Unpack file          */
            if (error)
                {
                olog(LOG_LOW, "Unpacking error!\n");
                stop_program = TRUE;
                break;
                }

            if (!error)
                remove(file->name);         /* Delete file              */

            error = test(cfg.testers, "."); /* Call file testers        */
            if (error)
                {
                olog(LOG_LOW, "File test error!\n");
                stop_program = TRUE;
                break;
                }

            if (!cfg.noRecursion)
                {
                ++recursion_level;
                count += convert(ALLFILES); /* Convert files - recurse  */
                --recursion_level;
                }

            sp = strchr(newname + 3, '.');  /* Set suffix               */
            if (sp != NULL)
                *sp = '\0';
            strcat(newname, ".");
            strcat(newname, cfg.dest->suffix);

            error = pack(cfg.dest, newname + 3);    /* Pack files       */
            if (error)
                {
                olog(LOG_LOW, "Packing error!\n");
                stop_program = TRUE;
                break;
                }

            rename(newname + 3, newname);   /* Move file to parent dir  */
            chdir("..");                    /* Move self to parent dir  */
            kill_dir_tree(tempdir);         /* Remove subdir            */

            nsz = file_sz(newname + 3);
            tnsz += nsz;
            if (recursion_level == 0)
                {
                ostatus(PRGM_NAME " : %-12s/%8ldb -> %-12s/%8ldb <> %+ldb\n",
                    file->name, osz, newname + 3, nsz, nsz - osz);
                olog(LOG_LOW, "%-12s/%8ldb <> %+ldb\n",
                    newname + 3, nsz, nsz - osz);
                }
            ++count;
            }
        test_for_escape();
        }                                   /* Step to next file        */
    while (!stop_program && !error && ((file = file->next) != NULL));
    close_dir(dir);
    if (recursion_level == 0)
        ostatus("\n" PRGM_NAME \
            " : Total old sz/%8ldb -> Total new sz/%8ldb <> %+ldb\n",
            tosz, tnsz, tnsz - tosz);
    return (count);
}

/*---------------------------------------------------------------------------*/
/*
    Test file given by filename (fn) for packer type.
    Returns pointer to typePacker struct if found, else NULL.
*/
static const typePacker *test_for_packer(const typePacker *pl, const char *fn)
{
    extern typeCfg cfg;
    FILE *file;
    int found = FALSE;
    if (pl != NULL) do
        {
        const int *is = pl->signature;
        if ((*is != EOSIGN) && (pl->cmdUnpack != NULL) &&
            ((file = fopen(fn, "rb")) != NULL))
            {
            ostatus("\rTesting: %-12s = %8s", fn, pl->name);
            olog(LOG_DEBUG, "\nTesting: %-12s = %8s", fn, pl->name);
            fseek(file, pl->signOffset,
                (pl->signOffset < 0) ? SEEK_END : SEEK_SET);
            do
                {
                found = fgetc(file);
                found = (*is == found) || (*is == DONT_CARE);
                }
            while (found && *++is != EOSIGN);
            fclose(file);
            }
        }
    while (!found && (pl = pl->next) != NULL);
    if (found)
        ostatus(" -> %-8s\n", cfg.dest->name);
    return (pl);
}

/*---------------------------------------------------------------------------*/
/*
    Run the "unpacker" to extract archive contents
*/
static int unpack(const typePacker *p, const char *file)
{
    char cmd[MAXLINELENGTH];
    sprintf(cmd, p->cmdUnpack, file);
    ostatus("Unpack: \"%s\"\n", cmd);
    olog(LOG_DEBUG, "Unpack: \"%s\"\n", cmd);
    return (p->success != os_shell(cmd));
}

/*---------------------------------------------------------------------------*/
/*
    Run the "test programs" to perform any needed processing on the
    extracted files
*/
static int test(const typeTester *tl, const char *path)
{
    char cmd[MAXLINELENGTH];
    int icc = FALSE;
    if (tl != NULL) do
        {
        sprintf(cmd, tl->cmdLine, path);
        ostatus("\nTest: \"%s\"\n", cmd);
        olog(LOG_DEBUG, "Test: \"%s\"\n", cmd);
        icc = (tl->success != os_shell(cmd));
        }
    while (!icc  && ((tl = tl->next) != NULL));
    return (icc);
}

/*---------------------------------------------------------------------------*/
/*
    Run the "packer" to rearchive the extracted files
*/
static int pack(const typePacker *p, const char *file)
{
    char cmd[MAXLINELENGTH];
    sprintf(cmd, p->cmdPack, file);
    ostatus("\nPack: \"%s\"\n", cmd);
    olog(LOG_DEBUG, "Pack: \"%s\"\n", cmd);
    return (p->success != os_shell(cmd));
}

/*---------------------------------------------------------------------------*/
/*
    this function shells to the OS and runs the (max 15) parameter list.
*/
int os_shell(char *cmd)
{
#ifdef SWAPCFG
    extern typeCfg cfg;
#endif
    int icc, i;
    char **pp;
    char *pshell[16];

    /* Build Command Array */ 

    if (cmd == NULL)    /* Shell to command interpreter */
        {
        if ((pshell[0] = getenv("COMSPEC")) == NULL)
            pshell[0] = "COMMAND.COM";
        pshell[1] = pshell[2] = NULL;
        }
    else                /* Execute command */
        {
        pshell[0] = strtok(cmd, " ");
        i = 1;
        while ((i < 15) && ((pshell[i] = strtok(NULL," ")) != NULL))
            ++i;
        pshell[i] = NULL;
        }
    pp = pshell;

    /* shell */

#ifdef SWAP

#ifndef SWAPCFG
    icc = spawnvpo(NULL, pp[0], pp);
#else
    if (cfg.swappath != NULL)
        icc = spawnvpo(cfg.swappath, pp[0], pp);
    else
        icc = spawnvp(P_WAIT, pp[0], pp);
#endif

#else
    icc = spawnvp(P_WAIT, pp[0], pp);
#endif

    return (icc);
}

/*---------------------------------------------------------------------------*/
/*
    Output function for status messages

    Why all the different output functions? To make it easier to implement
    fancy screen output if so desired...
*/
static void _cdecl ostatus(char *logstr, ...)
{
	va_list args;
    va_start(args, logstr);
    vfprintf(stdout, logstr, args);
    va_end(args);
}

/*---------------------------------------------------------------------------*/
/*
    Output function for error messages
*/
static void _cdecl oerror(char *logstr, ...)
{
	va_list args;
    va_start(args, logstr);
    vfprintf(stderr, logstr, args);
    va_end(args);
}

/*---------------------------------------------------------------------------*/
/*
    Output function for log messages
*/
static void _cdecl olog(int loglevel, char *logstr, ...)
{
    extern typeCfg cfg;
	va_list args;
    if (loglevel <= cfg.loglevel &&
        (cfg.loglevel == LOG_STDERR || cfg.logfile != NULL))
        {
        va_start(args, logstr);
        vfprintf((cfg.loglevel == LOG_STDERR) ? stderr : cfg.logfile,
            logstr, args);
        va_end(args);
        }
}

/*---------------------------------------------------------------------------*/
/*
    Reset [CTRL-C] handler to ctrl_c_handler function
*/
static void _cdecl ctrl_c_handler(void)
{
    extern int stop_program;
    /* signal(SIGINT, SIG_IGN); */  /* eat ctrl-c ? */
    if (signal(SIGINT, ctrl_c_handler) == SIG_ERR)
        exit (-1);
    stop_program = TRUE;
}

/*---------------------------------------------------------------------------*/
/*
*/
static void test_for_escape(void)
{
    extern int stop_program;
    if (kbhit())
        stop_program = (0x1b == getch());
}

/*---------------------------------------------------------------------------*/
/*
*/
static int xctoi(char c)
{
    int x = 0;
    if (isdigit(c))
        x = c -'0';
    else if (isxdigit(c))
        x = tolower(c) - 'a' + 10;
    return (x);
}
