/*
    Module: ms_dir.c
        MS-DOS directory functions

    Copyright (c) 1991-92 by Jorgen Sven Abrahamsen (2:230/100.9@fidonet)

    Programmer : JSA / Cirrus  25-01-1991  Ver. 1.00
*/
/*---------------------------------------------------------------------------*/
/*
    Includes
*/
#include <direct.h>
#include <dos.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "ms_dir.h"     /* My directory stuff - requires dos.h */

static typeDirEntry    de = {NULL, "*.*", 0xffff, 0L};
static typeDir       star = {&de, &de};
const  typeDir * ALLFILES = &star;

/*---------------------------------------------------------------------------*/
/*
    Create a directory structure.
*/
typeDir * open_dir(void)
{
    typeDir *dir = (typeDir *) malloc(sizeof(typeDir));
    if (dir != NULL)
        dir->head = dir->tail = NULL;
    return (dir);
}

/*---------------------------------------------------------------------------*/
/*
    Build a linked list of files matching the linked list of wildcards.
    Return pointer to first match or NULL if no matches found.
*/
typeDirEntry * read_dir(typeDir *dir, const typeDir *wilddir)
{
    struct find_t ffblk;
    typeDirEntry *first = NULL;
    typeDirEntry *wild = wilddir->head;

    while ((wild != NULL) &&
        (_dos_findfirst(wild->name, wild->attr, &ffblk) == 0))
        {
        do  /* Add file entry to ll - skip current dir and parent dir */
            if (!((ffblk.attrib & _A_SUBDIR) && ('.' == *ffblk.name)))
                {
                add_dir_entry(dir, ffblk.name, (unsigned) ffblk.attrib,
                    ffblk.size);
                if (first == NULL)
                    first = dir->tail;
                }
        while (_dos_findnext(&ffblk) == 0);
        wild = wild->next;
        }
    return (first);
}

/*---------------------------------------------------------------------------*/
/*
    Add a file to a directory linked list.
    Return pointer to first file or NULL allocation error.
*/
typeDirEntry * add_dir_entry(typeDir *dir, const char *name, unsigned attr, fpos_t size)
{
    typeDirEntry *file = (typeDirEntry *) malloc(sizeof(typeDirEntry));
    if (file != NULL)
        if ((file->name = strdup(name)) == NULL)
            {
            free(file);
            file = NULL;
            }
        else
            {
            file->attr = attr;
            file->size = size;
            file->next = NULL;
            if (dir->head == NULL)
                dir->head = dir->tail = file;
            else
                {
                dir->tail->next = file;
                dir->tail = file;
                }
            }
    return (file);
}

/*---------------------------------------------------------------------------*/
/*
    Free a directory linked list.
*/
void close_dir(typeDir *dir)
{
    while (dir->head != NULL)
        {
        dir->tail = dir->head->next;
        free(dir->head->name);
        free(dir->head);
        dir->head = dir->tail;
        }
    free(dir);
}

/*---------------------------------------------------------------------------*/
/*
    Kill (delete) a directory tree.
    Does not restore current working directory.
*/
void kill_dir_tree(const char *path)
{
    typeDir *dir = open_dir();

    if (!chdir(path))                           /* Move to 'path' */
        {
        read_dir(dir, ALLFILES);                /* Read current dir */
        while (dir->head != NULL)               /* Step thru dir */
            {
            if (dir->head->attr & _A_SUBDIR)    /* Recurse thru subdirs */
                kill_dir_tree(dir->head->name);
            else                                /* Delete file */
                remove(dir->head->name);
            dir->head = dir->head->next;        /* next file */
            }
        chdir("..");                            /* Move to parent of 'path' */
        rmdir(path);                            /* Remove path */
        }
    close_dir(dir);
}

/*---------------------------------------------------------------------------*/
/*
    Return size of file.
*/
fpos_t file_sz(char *filename)
{
    struct find_t ffblk;
    return (_dos_findfirst(filename, 0xffff, &ffblk) == 0) ?
        ffblk.size : 0L;
}

/*---------------------------------------------------------------------------*/
/*
    Return allocated string to name of current directory, NULL for error.
*/
char * get_dir(void)
{
    return _getdcwd(0, NULL, _MAX_PATH);
}

/*---------------------------------------------------------------------------*/
/*
    Set current working directory to dir. Also set drive.
    Return 0 for success.
*/
int set_dir(const char *dir)
{
    unsigned ui, uj;
    ui = (unsigned)(toupper(*dir) - 'A') + 1;
    _dos_setdrive(ui, &uj);
    _dos_getdrive(&uj);
    if (ui == uj)
        return chdir(dir);
    else
        return (-1);
}

#ifdef MAIN
/*---------------------------------------------------------------------------*/
/*
    What's this? Yes, it's a freebie - a directory remover.
*/
void main(int argc, char **argv)
{
    puts(get_dir());    /* just for test */
    if (argc != 2)
        puts("Kill a directory tree!\nUsage: <progname> <path>");
    else
        kill_dir_tree(argv[1]);
}
#endif
