/*

Binary to OBJ File Conversion Utility.
 (c) N.P.S. Software 1993.
  by Mark Thomas.

Files : 2OBJ.C
	SLINK.H SLINK.C
	TYPES.H TYPES.C
	ERRORS.H ERRORS.C
	DECODER.C ERRS.H STD.H

Compiler : QuickC 2.0, options: /AL /W3 /Ze /O /Ol /Gs

Last Modified : 16-12-93

Description :

 2OBJ is a program that can create an object (.OBJ) file from binary files
 so  that  it  can be linked directly into an program along with compiled
 source code and other data.  It provides  a  smaller,  faster  and  more
 secure alternative to the traditional methods of including data files in
 programs.  A number of enhancements are also present: multiple segments
 can be created to handle input files larger than the segment limit of 64k.
 These when linked together in the correct order will recreate the binary
 data in a continuous region of memory.  2OBJ can also process and decode
 CEL/GIF/LBM/PCX files and their VGA palettes.


 The OBJ list is created in memory using a series of singly linked
 lists.  Each record in the OBJ file forms one node in the main list,
 and the data stored inside the record is part of a sub-ordinate list.
 When the OBJ file is completed in memory then the lists are traversed
 to create the output file.  The record list and the data field lists
 are also indexed to support the indices in OBJ files (on SEGDEFS,
 GRPDEFS, LNAMES entries etc).


 OBJList
 ͻ
  C  H 
 ͼ
          THEHDR          LNAMES                       C Count
       Ŀ    Ŀ                    N Next pointer
       ȵ0x80LDNĴ0x96LDN etc             D Data pointer
	                        L Length of data
		                                      H Head pointer
		ͻ      ͻ               ListHandle
		 C  H        C  H                ListNode
		 ͼ       ͼ
		                      
		       Ŀ   Ŀ    Ŀ
		       ȵ0x00LDN   ȵ0x00LDNĴ0x80LDN etc
			        
				                              
				                              
				"\<n>Name"     0x00           "\<n>_TEXT"


 A segment is defined by a SEGSpecs structure, and the procedures will
 construct the OBJ file according to the information in it.

 Strings in OBJ files are not C null terminated string instead they
 are the Pascal type: a byte at the begining of the string specifies its
 length.  In the program they are call nstr.

 All memory in the OBJ lists must be dynamically allocated using malloc,
 although some procedures will automatically allocate and copy input data.
 This is to ensure that a list can be destroyed by returning the memory
 from all nodes.


Future Enhancements :

 Add better decoding options for files:
		/DP Decode Planar
		/DB Decode Bitmap (colour bytes)
		/DX Decode Bitmap, VGA mode-x
		/DR Decode to Raw, VGA palette then bitmap
		/D? Decode planar, where each plane is separate (also modex)

*/


#pragma pack(1)         /* This may cause problems with SLINK.C */

#include "2obj.h"
#include "types.h"
#include "slink.h"
#include "errors.h"

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <io.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys\types.h>
#include <sys\stat.h>


	/* The FileType structure is used to handle the differences     */
	/* between input file types by storing pointers to the functions*/
	/* that process each file type.  The default extension is also  */
	/* stored here as well as some general flags.                   */

typedef struct {
	char    TypeChar;
	char    TypeExt[5];
	int     (*IsType)(int);
	int     (*DecodeType)(int, int);
	int     (*HeaderType)(void);
	unsigned long (*SizeType)(int, int);
	int     (*EndType)(void);
	int     TypeFlags;
	} FileType;

					/* FileType.TypeFlags           */
#define T_HEADTAIL      0x01            /* Supports binary head/tails   */
#define T_HEADER        0x02            /* Supports file format headers */


FileType TypeBIN = { 'B', ".BIN", IsBIN, NULL,      NULL,      SizeBIN, EndBIN, T_HEADTAIL };
FileType TypeGIF = { 'G', ".GIF", IsGIF, DecodeGIF, HeaderGIF, SizeGIF, EndGIF, T_HEADER };
FileType TypeLBM = { 'L', ".LBM", IsLBM, DecodeLBM, HeaderLBM, SizeLBM, EndLBM, T_HEADER };
FileType TypePCX = { 'P', ".PCX", IsPCX, DecodePCX, HeaderPCX, SizePCX, EndPCX, T_HEADER };
FileType TypePAL = { 'V', ".PCX", IsPAL, DecodePAL, NULL,      SizePAL, EndPAL, 0 };
FileType TypeCEL = { 'C', ".CEL", IsCEL, DecodeCEL, HeaderCEL, SizeCEL, EndCEL, T_HEADER };


FileType *Type;                 /* Pointer to input file Typexxx struc  */

FileType *Types[] = { &TypeBIN,
		      &TypeGIF,
		      &TypeLBM,
		      &TypePCX,
		      &TypePAL,
		      &TypeCEL,
		      NULL };


/* Structures are defined for several of the OBJ records, as these are  */
/* used to for memory allocation and disk writes they _MUST_NOT_ contain*/
/* any compiler padding, so set byte packing.                           */

typedef ListNode RecordHeader;

struct  SEGDEF
	{
	char    attrib;
	short   seg_length;
	char    iSegmentName;
	char    iClassName;
	char    iOverlayName;
	};

struct GRPDEF
	{
	char    iGroupName;
	char    type;
	char    iSegment;
	};

struct  PUBDEF
	{
	char    iGroupRecord;
	char    iSegmentRecord;
	};

struct  LEDATA
	{
	char    iSegmentRecord;
	short   EnumDataOffset;
	};


typedef struct
	{
	char    *GRP_Str;
	char    *SEG_Str;
	char    ALIGN_Code;
	char    *CLASS_Str;
	char    iSEGDEF;
	char    iGRPDEF;
	ListHandle PublicsList;
	} SEGSpecs;

struct  PublicNode
	{
	char    *pSymbolName;
	unsigned short DataOffset;
	};

char    *str2nstr(char *src);
int     nstrlen(char *StringN);
int     OpenFile(char *file, int flag, int mode);
void    *AllocateMemory(short Size);
char    FindAlignment(char *pAlignStr);
void    *StrDup(char *String);
SEGSpecs *GetOBJSpecification(char *InputString);
int     isFilename(char *InputString);
void    CreateOBJFile(void);
void    AddPublicSymbol(SEGSpecs *SegSpecs, char *pSymbol, short Offset);

RecordHeader *SeekRecord(RecordHeader *pRecord, char SeekToType, char *RecordCount);
RecordHeader *AddRecord(ListHandle *pOBJHandle, char RecordType, char AfterRecordType, char *iNewRecord);
char    AddDataField(RecordHeader *pRecord, void* pData, short DataLength);
void    CreateTHEHDR(ListHandle *pOBJHandle, char *HdrStr);
char    AddLNAMES(ListHandle *pOBJHandle, char *NameStr);
void    AddSEGDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs, unsigned short SegmentLength);
void    AddGRPDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs);
void    AddPUBDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs);
void    AddLEDATA(ListHandle *pOBJHandle, SEGSpecs *SegSpecs, unsigned short DataOffset, unsigned short DataLength, char *pBuffer);
void    CreateRecord(ListHandle *pDataList, char *OutputBuffer, short *RecordLength);
void    WriteRecord(short hOBJFile, char *OutputBuffer, short RecordLength);
void    WriteOBJFile(ListHandle *pOBJHandle, int hOutputFile);
void    DestroyOBJList(ListHandle *pOBJHandle);



/* Static data for user .MODEL segment attributes.                      */

static char _TEXT[] = { "_TEXT" };
static char _DATA[] = { "_DATA" };
static char WORD[] = { "WORD" };
static char PUBLIC[] = { "PUBLIC" };
static char CODE[] = { "CODE" };
static char DATA[] = { "DATA" };
static char DGROUP[] = { "DGROUP" };
static char FAR_DATA[] = { "FAR_DATA" };

static SEGSpecs SMALL_CODE = { DGROUP, _TEXT, 0x48, CODE };
static SEGSpecs SMALL_DATA = { DGROUP, _DATA, 0x48, DATA };

static SEGSpecs LARGE_CODE = { NULL  , NULL, 0x48, CODE };
static SEGSpecs LARGE_DATA = { NULL  , NULL, 0x60, FAR_DATA };


static char     UsageMsg[] = { "\\
Usage: 2OBJ filetype [switches] infile [outfile[.OBJ | .BIN]] segspecs\n\\
 filetype : [B]inary | [C]EL | [G]IF | [L]BM | [P]CX | [V]GA palette\n\\
 switches : /C      Adds an underscore (_) to the symbol name.\n\\
	    /Hsize  Removes a header of size bytes from a binary input file.\n\\
	    /H[+|-] Include or exclude(default) the graphic file header.\n\\
	    /V      Verbose mode, displays extra information.\n\\
	    /L      Convert all symbols to lowercase.\n\\
	    /S      Split large (>64k) input files.\n\\
	    /Tsize  Removes a tail of size bytes from a binary input file.\n\\
	    /U      Convert all symbols to uppercase.\n\\
	    /X      Disables output of .OBJ file when using /D.\n\\
	    /D      Decode file type.\n\\
 infile   : Input file, extension defaults to BIN/CEL/PCX/GIF/LBM.\n\\
 outfile  : Output file, defaults to infile.OBJ, or infile.BIN with /X.\n\\
 segspecs : Segment specifications (not required with /X), either\n\\
		.model:SYMBOL\n\\
			model = [TINY|SMALL|MEDIUM|COMPACT|LARGE|HUGE]\n\\
		[group:]SEGNAME:[align:]['class':]SYMBOL\n\\
			align = [BYTE|WORD|PARA|PAGE]" };


char    InputFile[ _MAX_PATH ];         /* Input filename.      */
char    InputDrive[_MAX_DRIVE], InputDir[_MAX_DIR];
char    InputFname[_MAX_FNAME], InputExt[_MAX_EXT];

char    OutputFile[_MAX_PATH ];         /* Output filename.     */
SEGSpecs *OutputSeg;                    /* Definition of output segment */

unsigned long DataLength;               /* Flags and control variables. */
unsigned long HeaderLength = 0;
unsigned long TailLength = 0;
int     DecodeFile = 0;
int     InformationFlag = 0;
int     DisableOBJ = 0;
int     UnderscoreFlag = 0;
int     ToUpperFlag = 0;
int     ToLowerFlag = 0;
int     SplitFlag = 0;
int     HeaderFlag = 0;
int     hInputFile;
char    OutputType;
int     TMPInput = 0;
char    TMPFile[] = { "__2OBJ__.TMP" };

char    OBJExt[] = { ".OBJ" };
char    BINExt[] = { ".BIN" };

char    *DefaultOutputExt = OBJExt;


void    ParseCommandLine(int argc, char **argv)
{
	char    drive[_MAX_DRIVE], dir[_MAX_DIR];
	char    fname[_MAX_FNAME], ext[_MAX_EXT];
	char    *pCurrentArg;
	short   iargv;


	if( argc < 2 ) {
		puts( UsageMsg );
		Error(E_DONE, ES_NONE, NULL, 1);
	}

		/* The first thing on the command line should be a      */
		/* single character file type.                          */
	iargv = 1;

	if( (strlen(argv[iargv]) != 1) )
		Error(E_FATAL, ES_BAD_FILE_TYPE, argv[iargv], 1);
	else {
		FileType **CheckTypes;

		OutputType = toupper( *(argv[iargv]) );

		CheckTypes = Types;

		while(*CheckTypes != NULL) {
			if( (*CheckTypes)->TypeChar == OutputType) {
				Type = *CheckTypes;
				break;
				}
			CheckTypes++;
			}

		if(*CheckTypes == NULL)
			Error(E_FATAL, ES_BAD_FILE_TYPE, NULL, 1);
		}


		/* Next are the switches, process each argument that    */
		/* has a switch character '-' or '/'.                   */

	if(++iargv >= argc)
		Error(E_FATAL, ES_BAD_COMMAND_LINE, NULL, 1);

	pCurrentArg = argv[iargv];

	while( *pCurrentArg == '-' || *pCurrentArg == '/' )
		{
		char    SwitchChar;
		char    *pSwitchString;

		pSwitchString = pCurrentArg;    /* Read the character after */
		SwitchChar = *(pCurrentArg+1);  /* the switch.              */
		pCurrentArg+=2;

		switch(toupper(SwitchChar))     /* Then process it.     */
		{
		case 'C':
			UnderscoreFlag = 1;
			break;

		case 'H':
			if( (Type->TypeFlags & T_HEADER) ) {
				if(*pCurrentArg == '+') {
					HeaderFlag = 1;
					pCurrentArg++;
					}
				else
					if (*pCurrentArg == '-') {
						HeaderFlag = 0;
						pCurrentArg++;
						}
					else
						Error(E_FATAL, ES_BAD_HEADER, NULL, 1);

				break;
			}
				/* /H is followed by a decimal count    */
				/* convert this count to a long and     */
				/* skip any digit characters to ensure  */
				/* there is nothing after the switch.   */

			if( (Type->TypeFlags & T_HEADTAIL) ) {

				if((HeaderLength = atol(pCurrentArg)) == 0)
					Error(E_FATAL, ES_BAD_HEADER_LENGTH, NULL, 1);

				while(isdigit(*(++pCurrentArg)));

				break;
			}

		case 'U':
			ToUpperFlag = 1;
			break;

		case 'L':
			ToLowerFlag = 1;
			break;

		case 'S':
			SplitFlag = 1;
			break;

		case 'T':
			if(!(Type->TypeFlags & T_HEADTAIL))
				Error(E_FATAL, ES_BAD_TAIL, NULL, 1);

			if((TailLength = atol(pCurrentArg)) == 0)
				Error(E_FATAL, ES_BAD_TAIL_LENGTH, NULL, 1);

			while(isdigit(*(++pCurrentArg)));
			break;

		case 'V':
			InformationFlag = 1;
			break;

		case 'D':
			if( Type->DecodeType != NULL )
				DecodeFile = 1;
			else
				puts("Warning: File type does not support decoding, /D ignored");

			break;

		case 'X':
			DisableOBJ = 1;
			break;

		default:
			Error(E_FATAL, ES_BAD_SWITCH, pSwitchString, 1);
		}
		if(*pCurrentArg != '\0')
			break;

		if(++iargv >= argc)
			Error(E_FATAL, ES_BAD_COMMAND_LINE, NULL, 1);

		pCurrentArg = argv[iargv];
		}


	if(DisableOBJ) {
		if(!DecodeFile) {
			puts("Warning: /X can be used only with decoded files, /X ignored\n");
			DisableOBJ = 0;
			}
		else
			DefaultOutputExt = BINExt;
		}

	if(HeaderFlag) {
		if(DecodeFile) {
			puts("Warning: Header is not included with decoded files. /H+ ignored\n");
			HeaderFlag = 0;
			}
		}

		/* After the switches is the input file, split it and   */
		/* then append an appropriate extension if none is given*/

	_splitpath(argv[iargv], InputDrive, InputDir, InputFname, InputExt);
	if(*InputExt == '\0') {
		strcpy(InputExt, Type->TypeExt);
	}
	_makepath(InputFile, InputDrive, InputDir, InputFname, InputExt);


	if(++iargv >= argc) {
		if (DisableOBJ)
			_makepath(OutputFile, NULL, NULL, InputFname, DefaultOutputExt);
		else
			Error(E_FATAL, ES_BAD_COMMAND_LINE, NULL, 1);
		}
	else {


		/* The next argument can be either the output file or   */
		/* the segment specifications.  The major difference    */
		/* between the two is the number of semi-colons, which  */
		/* is what isFilename checks.  The default output file  */
		/* is simply the input filename with a .OBJ extension.  */

		if( isFilename(argv[iargv]) ) {
			_splitpath(argv[iargv], drive, dir, fname, ext);
			if(*ext == '\0')
				strcpy(ext, DefaultOutputExt);
			_makepath(OutputFile, drive, dir, fname, ext);

			iargv++;
		}
		else
			_makepath(OutputFile, NULL, NULL, InputFname, DefaultOutputExt);
	}

	if(!DisableOBJ) {

		if(iargv >= argc)
			Error(E_FATAL, ES_BAD_COMMAND_LINE, NULL, 1);


		OutputSeg = GetOBJSpecification(argv[iargv++]);

		/* Split mode requires some overrides to whatever was   */
		/* on the command line segment.  Firstly groups are not */
		/* not allowed to exceed 64K+ of memory to line so they */
		/* cannot be used.  And also the segment is made        */
		/* PRIVATE, so no other data can be combined in.        */

		if(SplitFlag) {
			free(OutputSeg->GRP_Str);
			OutputSeg->GRP_Str = NULL;
			OutputSeg->iGRPDEF = 0;
			OutputSeg->ALIGN_Code = 0x60;
		}

		/* That should be everything, check that the local index*/
		/* is equal to the count of command line arguments.     */
	}

	if(iargv != argc)
		Error(E_FATAL, ES_EXTRA_CHARS, NULL, 1);


	hInputFile = OpenFile(InputFile, O_BINARY | O_RDONLY, 0);


	if( !(Type->IsType(hInputFile)) )
		Error(E_FATAL, ES_NOT_TYPE, NULL ,1);

	if(HeaderFlag)
		lseek(hInputFile, 0L, SEEK_SET);

	if( (Type->HeaderType != NULL) && (InformationFlag) )
		Type->HeaderType();

	if(DecodeFile) {
		int     hTMPFile;


		hTMPFile = OpenFile(TMPFile, O_CREAT | O_BINARY | O_RDWR | O_TRUNC, S_IREAD | S_IWRITE);

		if( Type->DecodeType(hInputFile, hTMPFile) )
			exit(-1);
		DataLength = filelength(hTMPFile);

		close(hInputFile);
		close(hTMPFile);

		hInputFile = OpenFile(TMPFile, O_BINARY | O_RDONLY, 0);

		TMPInput = 1;
	} else
		DataLength = Type->SizeType(hInputFile, HeaderFlag);

	Type->EndType();

	if( DataLength == 0L )
		Error(E_FATAL, ES_ZERO_LENGTH_INFILE, NULL, 1);

	printf("Input File  : %s  %lu bytes%s\n", InputFile, DataLength,
			 DecodeFile ? ", decoded." : "." );


		/* Now check that reasonable Header and tail lengths    */
		/* were given.                                          */

	if(!DisableOBJ) {
		if( (HeaderLength >= DataLength) ||
		(TailLength >= DataLength) ||
		((HeaderLength + TailLength) >= DataLength) )
			Error(E_FATAL, ES_BAD_HEAD_TAIL, NULL, 1);

		/* Check that the final length of the data is not       */
		/* larger than 64K, unless splitting is requested.      */

		if( (DataLength - HeaderLength - TailLength) > 65536 && !SplitFlag)
			Error(E_FATAL, ES_BIG_INFILE, NULL, 1);

		/* If splitting is required check that the 9 output     */
		/* file limit will not be exceeded.                     */

		if( (DataLength - HeaderLength - TailLength) > (65536L*9L) )
			Error(E_FATAL, ES_BIG_SPLIT, NULL, 1);
	}
}


char    *str2nstr(char *src)
{
	/* This procedure will convert a '\0' terminated string into    */
	/* a Pascal type string.  It does not destroy the original src  */
	/* string and will allocate a destination buffer itself.        */
	/* No checking is made on src.                                  */

	int     StrLength;
	char    *StringN;

	StrLength = strlen(src);

	StringN = AllocateMemory(StrLength+1);

	*StringN = (char)StrLength;
	strncpy(StringN + 1, src, StrLength);

	return StringN;
}


int     nstrlen(char *StringN)
{
	/* This procedure will return the length of an nstr, which can  */
	/* be got from the first byte of the string.  This length       */
	/* includes the length byte.                                    */

	return (int)((*StringN) + 1);
}


int     OpenFile(char *file, int flag, int mode)
{
	/* Opens a file using open, and will abort the program with an  */
	/* error message if it fails.                                   */

	int     handle;

	if( (handle = open(file, flag, mode)) == -1 )
		Error(E_FATAL, ES_BAD_OPEN_FILE, file, 1);
	else
		return handle;
}


int     WriteFile(int handle, char *buffer, unsigned int count)
{

	if( count != (write(handle, buffer, count)) )
		Error(E_FATAL, ES_BAD_FILE_WRITE, NULL, 1);
	else
		return count;
}


void    *AllocateMemory(short Size)
{
	/* Allocates memory using malloc, and will abort the program    */
	/* with an error message if it fails.                           */

	void    *pNewBlock;

	if( (pNewBlock = malloc(Size)) == NULL)
		Error(E_FATAL, ES_NO_MEMORY, NULL, 1);

	return pNewBlock;
}


int     isFilename(char *InputString)
{
	/* This procedure will check if the input string might possibly */
	/* be a filename or a segment specification on the command line.*/
	/* The only real check that can be made is the number and       */
	/* position of semicolons in the string.  A valid filename can  */
	/* have only one semi-colon, and that must be a drive letter,so */
	/* it must be the second character in the string.               */

	int     iSemiColon;

	if( (iSemiColon = strcspn(InputString, ":")) == 1)
		return 1;
	else
		return( iSemiColon == strlen(InputString));
}


char    FindAlignment(char *pAlignStr)
{
	/* This procedure will attempt to match the input string with   */
	/* one of the valid alignment strings.  It will return the type */
	/* of alignment as defined in the OBJ file (1=BYTE, 2=WORD...)  */
	/* If the strings do not match then 0xFF will be returned.      */

	static char AlignStrings[4][5] = { "BYTE", "WORD", "PARA", "PAGE" };
	char   iTemp;

	for(iTemp=0; iTemp < 4; iTemp++)
		if( stricmp(pAlignStr, AlignStrings[iTemp]) == 0 )
			break;

	if( iTemp > 3 )
		iTemp = 0xFF;
	else
		iTemp++;        /* Add one to convert from index to the */
				/* alignment types in the SEGDEF record.*/
	return iTemp;
}


void    *StrDup(char *String)
{
	/* This procedure will duplicate a '\0' terminated string by    */
	/* copying it into memory it has allocated.  If the input       */
	/* pointer is NULL then no action is taken and NULL is returned */
	/* otherwise the pointer to the new memory block is returned.   */

	if( String != NULL )
		return( strcpy(AllocateMemory(strlen(String)+1), String));
	else
		return NULL;
}


SEGSpecs *GetOBJSpecification(char *InputString )
{
	/* This procedure will attempt to convert the segment specifier */
	/* from the command line into a SEGSpecs structure.  Any errors */
	/* will abort the program.                                      */

	SEGSpecs *SegmentSpecs;
	SEGSpecs *ReturnSpecs;
	SEGSpecs TempSpecs;
	char    SegName[ _MAX_FNAME + 5 + 1];
	char    *pSegmentAttr[4];
	char    **ppTemp, *pSymbol, *pTemp;
	short   CountSemiColons;
	short   iTemp;
	char    *pSymbolStr;
	static  char ModelStrings[6][8] = { ".TINY", ".SMALL", ".MEDIUM",
					    ".COMPACT", ".LARGE", ".HUGE" };
	static  SEGSpecs *Models[6] = { &SMALL_DATA,
					&SMALL_DATA,
					&SMALL_DATA,
					&LARGE_DATA,
					&LARGE_DATA,
					&LARGE_DATA };

	CountSemiColons = 0;
	pSymbol = InputString;

		/* Now process each part of the segment specified on the*/
		/* command line by copying each ':' delimited string    */
		/* into the pSegmentAttr array.  The final string (the  */
		/* symbol) will be pointed to by pSymbol.               */

	while( (pTemp = strchr(pSymbol, ':' )) != NULL ) {
		if( CountSemiColons >= 4)
			Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		/* Get the length of the substring by subtracting the   */
		/* start of the string and the position of its          */
		/* terminating :.  Then copy the string into allocated  */
		/* memory and store the pointer.                        */

		iTemp = pTemp - pSymbol;

		if(iTemp == 0)
			Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		pSegmentAttr[CountSemiColons] = AllocateMemory(iTemp+1);
		strncpy(pSegmentAttr[CountSemiColons], pSymbol, iTemp);
		*(pSegmentAttr[CountSemiColons] + iTemp) = '\0';
		pSymbol = pTemp+1;      /* +1, skips the : at the end.  */
		CountSemiColons++;
	}

		/* Check that at least one semi-colon was found, if not */
		/* then the input string could not have been valid.     */

	if( CountSemiColons == 0 )
		Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		/* Check that the symbol isn't a possible 'CLASS'       */
		/* string, which would be invalid.                      */

	if( *pSymbol == '\'' )
		Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		/* Now try to match the first string with one of the    */
		/* six .MODEL directives.  If a match is found check    */
		/* that a segment name is set in the model structure    */
		/* if not then create one from the input file name.     */

	SegmentSpecs = NULL;

	for(iTemp=0; iTemp != 6; iTemp++) {
		if( stricmp(pSegmentAttr[0], ModelStrings[iTemp]) == 0 ) {
			SegmentSpecs = Models[iTemp];
			if(SegmentSpecs->SEG_Str == NULL) {
				strcpy(SegName, InputFname);
				strcat(SegName, _DATA);
				strupr(SegName);
				SegmentSpecs->SEG_Str = SegName;
			}
		break;
		}
	}

	if( SegmentSpecs == NULL ) {

		/* Use ppTemp to step through each string in the array, */
		/* starting from the last string, because this can be   */
		/* easily identified as a class string becuase of the   */
		/* ' characters delimiting it.  The complete SEGSpecs   */
		/* are built up in TempSpecs.                           */

		ppTemp = &pSegmentAttr[CountSemiColons-1];

		if( **ppTemp == '\'') {
			TempSpecs.CLASS_Str = (*ppTemp)+1;

			if( *(*ppTemp + strlen(*ppTemp) - 1) != '\'')
				Error(E_FATAL, ES_BAD_CLASS, NULL, 1);
			(*(*ppTemp + strlen(*ppTemp) - 1) = 0);

			ppTemp--;
		}
		else                    /* There is no default CLASS_Str*/
			TempSpecs.CLASS_Str = NULL;

		/* Now try to match an alignment string, using the      */
		/* FindAlignment procedure.  If no match is found then  */
		/* the default alignment is set to PARA.                */

		if((TempSpecs.ALIGN_Code = FindAlignment(*ppTemp)) == (char)0xFF)
			TempSpecs.ALIGN_Code = 3;
		else
			ppTemp--;

		/* Convert the alignment code returned by FindAligment  */
		/* into the proper value used in OBJ files, and set     */
		/* the combine type to public.                          */

		TempSpecs.ALIGN_Code = (TempSpecs.ALIGN_Code << 5) | 0x08;

		/* If there is less than one segment string left then   */
		/* the input string was bad, otherwise the next string  */
		/* is the name of the segment.                          */

		if( ppTemp >= pSegmentAttr) {
			TempSpecs.SEG_Str = *ppTemp;
			ppTemp --;
		}
		else
			Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		/* Check the count of strings again, if there is        */
		/* another then it is the group name, otherwise there   */
		/* is no default group name.                            */

		if( ppTemp >= pSegmentAttr ) {
			TempSpecs.GRP_Str = *ppTemp;
			ppTemp --;
		}
		else
			TempSpecs.GRP_Str = NULL;

		/* That should be all the segment specifier strings     */
		/* processed, if not then there is an error.            */

		if( ppTemp >= pSegmentAttr )
			Error(E_FATAL, ES_BAD_SEGMENT, NULL, 1);

		SegmentSpecs = &TempSpecs;
	}

		/* Now allocate and copy from the completed user        */
		/* SEGSpecs into the final structure.                   */

	ReturnSpecs = AllocateMemory(sizeof( SEGSpecs ));

	if(SegmentSpecs->GRP_Str != NULL)
		strupr(SegmentSpecs->GRP_Str);
	if(SegmentSpecs->SEG_Str != NULL)
		strupr(SegmentSpecs->SEG_Str);
	if(SegmentSpecs->CLASS_Str != NULL)
		strupr(SegmentSpecs->CLASS_Str);

	ReturnSpecs->GRP_Str = StrDup(SegmentSpecs->GRP_Str);
	ReturnSpecs->SEG_Str = StrDup(SegmentSpecs->SEG_Str);
	ReturnSpecs->ALIGN_Code = SegmentSpecs->ALIGN_Code;
	ReturnSpecs->CLASS_Str = StrDup(SegmentSpecs->CLASS_Str);
	ReturnSpecs->iSEGDEF = 0;       /* 0 out the segment and group  */
	ReturnSpecs->iGRPDEF = 0;       /* indices, this is the default.*/

		/* Now create the public symbols list, and add a symbol */
		/* string to it.  Firstly check all the flags which     */
		/* effect the symbol string.                            */

	InitialiseList(&(ReturnSpecs->PublicsList));

	pSymbolStr = AllocateMemory(strlen(pSymbol)+2);

	*pSymbolStr = '\0';
	if(UnderscoreFlag == 1)
		strcat(pSymbolStr,"_");
	strcat(pSymbolStr, pSymbol);

	if(ToUpperFlag)
		strupr(pSymbolStr);
	if(ToLowerFlag)
		strlwr(pSymbolStr);

	AddPublicSymbol(ReturnSpecs, pSymbolStr, 0);

		/* Make sure all temporary memory allocated by this     */
		/* procedure is freed.                                  */

	free(pSymbolStr);

	for(iTemp=0; iTemp<CountSemiColons; iTemp++)
		free(pSegmentAttr[iTemp]);

		/* And return the completed SEGSpecs pointer.           */

	return ReturnSpecs;
}


void    AddPublicSymbol(SEGSpecs *SegSpecs, char *pSymbol, short Offset)
{
	/* This procedure will add a public symbol into the public      */
	/* symbol list of a segment.  A public symbol requires a string */
	/* name and the offset in the segment of the symbol's data. The */
	/* symbol string will be converted into a nstr, so it is not    */
	/* required to be allocated memory.                             */

	struct PublicNode *pPublicInfo;

	pPublicInfo = AllocateMemory( sizeof(struct PublicNode) );
	pPublicInfo->pSymbolName = str2nstr(pSymbol);
	pPublicInfo->DataOffset = Offset;

	InsertListNode(&(SegSpecs->PublicsList), 0x00, sizeof( struct PublicNode), pPublicInfo, NULL);
}


RecordHeader *SeekRecord(RecordHeader *pRecord, char SeekToType, char *RecordCount)
{
	/* This procedure will attempt to find a specified record type  */
	/* in the list of OBJ records.  Because OBJ files often use     */
	/* indices to records this procedure can also return a count of */
	/* the record in the list which can be used as an index.  This  */
	/* value is returned using RecordCount pointer in the arguments */
	/* if no count is required then this should be NULL.  If there  */
	/* are a number of the required record type then only the final */
	/* one in the list will be returned.  If there are no matching  */
	/* records in the list then NULL will be returned.              */

	RecordHeader *pReturnRecord;
	unsigned char TempCount;

		/* Find the first occurence of the record type in the   */
		/* list.                                                */

	pReturnRecord = NULL;
	pRecord = SeekRecordType(pRecord, SeekToType);

		/* Now repeat the search until the record cannot be     */
		/* found, the previous find will be the last occurence  */
		/* of the record type in the list.  If the original     */
		/* search has failed then this loop not be entered and  */
		/* the NULL value will be returned.                     */

	TempCount = 1;
	while(pRecord != NULL) {
		TempCount++;
		pReturnRecord = pRecord;
		pRecord = SeekRecordType(pRecord->pNextNode, SeekToType);
	}
		/* If a pointer to a record count variable was given    */
		/* then store the count value there.                    */

	if( RecordCount != NULL )
	       *RecordCount = TempCount;

	return pReturnRecord;
}


RecordHeader *AddRecord(ListHandle *pOBJHandle, char RecordType, char AfterRecordType, char *iNewRecord)
{
	/* This procedure will add a record entry into the record list. */
	/* The position of the record is specified in AfterRecordType,  */
	/* the new record will be placed after the last occurence of a  */
	/* record of AfterRecordType type in the list.  Because there   */
	/* can be numerous records with the same type as the record to  */
	/* be added, this procedure will also ensure that the new record*/
	/* is placed after all the other records with the same type.    */
	/* If there is no record of type AfterRecordType in the list    */
	/* the program will be forced to terminate with an error message*/
	/* However to insert a record at the head of the list a record  */
	/* type of 0x00 can be given. Also a count of the number of     */
	/* records with the same type can be returned via iNewRecord    */
	/* for use as an index.  iNewRecord should be NULL if no count  */
	/* is required.                                                 */

	RecordHeader *pPreviousRecord;
	RecordHeader *pTemp;

		/* If an insertion at the head was requested then the   */
		/* previous record pointer should be NULL.              */

	if( AfterRecordType == 0x00 )
		pPreviousRecord = NULL;
	else {
		/* Otherwise search for the requested AfterRecordType,  */
		/* if it is not found then terminate.  If it is found   */
		/* there may be a several other records following it    */
		/* with the same type as the new record, so search for  */
		/* records with the same type.  This will also give the */
		/* value for iNewRecord.  If there are no records with  */
		/* the same type then use the AfterRecordType record    */
		/* pointer, otherwise use the pointer to the last       */
		/* record with the same type.                           */

		if( (pPreviousRecord = SeekRecord(pOBJHandle->pHead, AfterRecordType, NULL)) == NULL)
			Error(E_INTERNAL, ES_BAD_SEEK, NULL, 1);
		if((pTemp = SeekRecord(pPreviousRecord, RecordType, iNewRecord)) != NULL)
			pPreviousRecord = pTemp;
	}

	return (RecordHeader *)(InsertListNode(pOBJHandle, RecordType, 0, NULL, pPreviousRecord));
}


char    AddDataField(RecordHeader *pRecord, void* pData, short DataLength)
{
	/* This procedure will add a data field node to a OBJ record.   */
	/* The new data will be inserted at the end of the data list for*/
	/* the record.  If the record does not have a data list then a  */
	/* ListHandle will be allocated an initialised for it.  The data*/
	/* will not be copied into a new memory block, so pData should  */
	/* be allocated memory.  The count of nodes in the data list is */
	/* returned and so it can be used as index to a data field.     */

	ListNode *pPreviousField;

		/* First check the record does have a data list, of not */
		/* then create and initialise one.                      */

	if(pRecord->pDataRecord == NULL) {
		pRecord->pDataRecord = AllocateMemory( sizeof(ListHandle) );
		InitialiseList(pRecord->pDataRecord);
	}

		/* Now search to the end of the data list to get the    */
		/* insert after pointer.                                */

	pPreviousField = IndexedSeekNode(pRecord->pDataRecord, 0xFFFF);

		/* Insert the new data node, all data nodes have a      */
		/* record type of 0x00.                                 */

	InsertListNode(pRecord->pDataRecord, 0x00, DataLength, pData, pPreviousField);

		/* And return the new node count.                       */
	return (char)((ListHandle *)pRecord->pDataRecord)->NodeCount;
}


void    CreateTHEHDR(ListHandle *pOBJHandle, char *HdrStr)
{
	/* This procedure will create THEHDR, as well as the LNAMES     */
	/* COMENT, and MODEND records.  This will create the basis for  */
	/* adding all other OBJ records, and therefore _MUST_ be the    */
	/* first procedure called when constructing an OBJ list.  The   */
	/* HdrStr is used as the module name string in THEHDR record.   */
	/* It will be copied into an nstr so it is not required to be   */
	/* dynamic memory.                                              */

	static char MODENDRec[] = { 0x00 };
	static char COMENTRec[] = { 0x00, 0xA2, 0x01 };
	static char NPSCOMENT[] = { "\x00\x00 2OBJ by NPS Software Division.  " };
	static char LNAMESRec[] = { 0x00 };
	RecordHeader *pRecord;
	char    *pHdrStrN;

		/* THEHDR is the very first record in the list, insert  */
		/* it at the head.  Then add the HdrStr as a data field */
		/* and the record is complete.                          */
	pRecord = AddRecord(pOBJHandle, 0x80, 0x00, NULL);
	pHdrStrN = str2nstr(HdrStr);
	AddDataField(pRecord, pHdrStrN, nstrlen(pHdrStrN));

		/* Now add a LNAMES record after THEHDR and also add    */
		/* the default zero-length name field.                  */
	AddRecord(pOBJHandle, 0x96, 0x80, NULL);
	AddLNAMES(pOBJHandle, LNAMESRec);

		/* Now add the COMENT record after the LNAMES, it data  */
		/* field has to be copied into allocated memory, so do  */
		/* so.                                                  */
	pRecord = AddRecord(pOBJHandle, 0x88, 0x96, NULL);
	AddDataField(pRecord, memcpy(AllocateMemory(3), COMENTRec, 3), 3);

		/* Add a little NPS comment.                            */
	pRecord = AddRecord(pOBJHandle, 0x88, 0x96, NULL);
	AddDataField(pRecord, memcpy(AllocateMemory(35), NPSCOMENT, 35), 35);

		/* Lastly the MODEND record is added after the COMENT   */
		/* record, its data field must also be copied into      */
		/* allocated memory.                                    */
	pRecord = AddRecord(pOBJHandle, 0x8A, 0x88, NULL);
	AddDataField(pRecord, memcpy(AllocateMemory(1), MODENDRec, 1), 1);
}


char    AddLNAMES(ListHandle *pOBJHandle, char *NameStr)
{
	/* This procedure will insert a string into the LNAMES.  The    */
	/* NameStr must be '\0' terminated and will be converted into an*/
	/* nstr, so is not required to be allocated memory.  The index  */
	/* to the name field will be returned.                          */

	RecordHeader *pRecord;
	char    *pData;

		/* Firstly create a nstr of the name string then find   */
		/* the address of the LNAMES record in the list and add */
		/* the data to that record.                             */

	pData = str2nstr(NameStr);
	pRecord = SeekRecord(pOBJHandle->pHead, 0x96, NULL);
	return (AddDataField(pRecord, pData, nstrlen(pData)));
}


void    AddSEGDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs, unsigned short SegmentLength)
{
	/* This procedure will add a SEGDEF record to the OBJ list.  The*/
	/* segment is defined in a SEGSpecs record, except for the      */
	/* length which is a separate argument.  The iSEGDEF record will*/
	/* be set in the SEGSpecs for future references.                */

	struct SEGDEF *pNewSEGDEF;
	RecordHeader *pRecord;

		/* Add a new SEGDEF record after the LNAMES.  Store the */
		/* index to the record in SegSpecs.                     */

	pRecord = AddRecord(pOBJHandle, 0x98, 0x96, &(SegSpecs->iSEGDEF));

		/* Now create the SEGDEF structure and initialise its   */
		/* components.                                          */
	pNewSEGDEF = AllocateMemory( sizeof(struct SEGDEF) );
	pNewSEGDEF->attrib = SegSpecs->ALIGN_Code;
	pNewSEGDEF->seg_length = SegmentLength;
	pNewSEGDEF->iSegmentName = AddLNAMES(pOBJHandle, SegSpecs->SEG_Str);
	if(SegSpecs->CLASS_Str != NULL)
		pNewSEGDEF->iClassName = AddLNAMES(pOBJHandle, SegSpecs->CLASS_Str);
	else
		pNewSEGDEF->iClassName = 1;
	pNewSEGDEF->iOverlayName = 1;

		/* And finally add the SEGDEF data field to the record. */
	AddDataField(pRecord, pNewSEGDEF, sizeof(struct SEGDEF));
}


void    AddGRPDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs)
{
	/* This procedure will add a GRPDEF record to the OBJ list. It  */
	/* will not check whether a group is actually defined (in the   */
	/* GRP_Str element of SegSpecs) and this should be done before  */
	/* calling this procedure.  This should also be called only     */
	/* after the SEGDEF record has been created.                    */

	struct GRPDEF *pNewGRPDEF;
	RecordHeader *pRecord;

		/* Add the GRPDEF record after the SEGDEFs.  Get the    */
		/* index into the SegSpecs.                             */
	pRecord = AddRecord(pOBJHandle, 0x9A, 0x98, &(SegSpecs->iGRPDEF));

		/* Now create the GRPDEF data field.                    */

	pNewGRPDEF = AllocateMemory( sizeof(struct GRPDEF) );
	pNewGRPDEF->iGroupName = AddLNAMES(pOBJHandle, SegSpecs->GRP_Str);
	pNewGRPDEF->type = 0xFF;
	pNewGRPDEF->iSegment = SegSpecs->iSEGDEF;

		/* And add it to the record.                            */
	AddDataField(pRecord, pNewGRPDEF, sizeof(struct GRPDEF));
}


void    AddPUBDEF(ListHandle *pOBJHandle, SEGSpecs *SegSpecs)
{
	/* This procedure will add the PUBDEF records for the public    */
	/* symbols defined in the SegSpecs of a segment.                */

	struct PUBDEF *pNewPUBDEF;
	RecordHeader *pRecord;
	char *pPublicNameRec;
	short PubNameLength;
	ListNode *pSymbolNodes;
	struct PublicNode *pSymbolInfo;

		/* Add the PUBDEF record after the GRPDEF if it exists  */
		/* otherwise after the SEGDEF record.                   */

	if(SegSpecs->GRP_Str != NULL)
		pRecord = AddRecord(pOBJHandle, 0x90, 0x9A, NULL);
	else
		pRecord = AddRecord(pOBJHandle, 0x90, 0x98, NULL);

		/* Create the new PUBDEF structure and add it to the    */
		/* records data list.                                   */

	pNewPUBDEF = AllocateMemory( sizeof(struct PUBDEF) );
	pNewPUBDEF->iGroupRecord = SegSpecs->iGRPDEF;
	pNewPUBDEF->iSegmentRecord = SegSpecs->iSEGDEF;
	AddDataField(pRecord, pNewPUBDEF, sizeof(struct PUBDEF));

		/* Now traverse the PublicsList for the SegSpecs and    */
		/* add a data field for each public symbol that is      */
		/* defined.                                             */

	pSymbolNodes = (SegSpecs->PublicsList).pHead;

	while(pSymbolNodes != NULL) {
		pSymbolInfo = (struct PublicNode *)(pSymbolNodes->pDataRecord);
		PubNameLength = nstrlen(pSymbolInfo->pSymbolName);
		pPublicNameRec = AllocateMemory(PubNameLength + 3);
		strncpy(pPublicNameRec, pSymbolInfo->pSymbolName, PubNameLength);
		*(short *)(pPublicNameRec+PubNameLength) = pSymbolInfo->DataOffset;
		*(pPublicNameRec+PubNameLength+2) = 0x00;
		AddDataField(pRecord, pPublicNameRec, PubNameLength + 3);

		pSymbolNodes = pSymbolNodes->pNextNode;
	}
}


void    AddLEDATA(ListHandle *pOBJHandle, SEGSpecs *SegSpecs, unsigned short DataOffset, unsigned short DataLength, char *pBuffer)
{
	/* This procedure will add LEDATA records for a segment.  The   */
	/* data buffer can be any length, and will be broken up into    */
	/* 1024 byte LEDATA records by the procedure.  The offset in the*/
	/* segment that the data will be placed is defined in the       */
	/* DataOffset argument and the caller must ensure that the      */
	/* offset is consistent between calls to this procedure.  All   */
	/* memory in the input buffer is copied into new allocated      */
	/* memory so when this procedure returns the input buffer can   */
	/* be reused or returned.                                       */

	struct LEDATA *pNewLEDATA;
	RecordHeader *pRecord;
	unsigned short TempCount;
	char *MemBuffer;

		/* First find how many 1024 byte records there are in   */
		/* the data buffer.                                     */
	TempCount = DataLength / 1024;

		/* Then add each 1024 byte record and data field.       */
	while(TempCount != 0) {
		pRecord = AddRecord(pOBJHandle, 0xA0, 0x88, NULL);
		pNewLEDATA = AllocateMemory( sizeof(struct LEDATA) );
		pNewLEDATA->iSegmentRecord = SegSpecs->iSEGDEF;
		pNewLEDATA->EnumDataOffset = DataOffset;

		AddDataField(pRecord, pNewLEDATA, sizeof(struct LEDATA));
		MemBuffer = AllocateMemory(1024);
		memcpy(MemBuffer, pBuffer, 1024);
		AddDataField(pRecord, MemBuffer, 1024);
		DataOffset += 1024;
		pBuffer += 1024;
		TempCount--;
	}

		/* Then add the remaining data.                         */

	if(DataLength %1024 != 0) {
		pRecord = AddRecord(pOBJHandle, 0xA0, 0x88, NULL);
		pNewLEDATA = AllocateMemory(sizeof(struct LEDATA));
		pNewLEDATA->iSegmentRecord = SegSpecs->iSEGDEF;
		pNewLEDATA->EnumDataOffset = DataOffset;
		AddDataField(pRecord, pNewLEDATA, sizeof(struct LEDATA));
		MemBuffer = AllocateMemory(DataLength % 1024);
		memcpy(MemBuffer, pBuffer, DataLength % 1024);
		AddDataField(pRecord, MemBuffer, DataLength % 1024);
	}
}


void    CreateRecord(ListHandle *pDataList, char *OutputBuffer, short *RecordLength)
{
	/* This procedure creates the complete OBJ record by combining  */
	/* all of its data field nodes into the OutputBuffer.  It also  */
	/* tallies the length of each nodes data into the RecordLength  */
	/* variable, which should be initialised before calling this    */
	/* procedure.                                                   */

	ListNode *pCurrentNode;

		/* Copy each data field nodes data into the output      */
		/* buffer and add the length of the data up.            */

	pCurrentNode = pDataList->pHead;
	while(pCurrentNode != NULL) {
		memcpy(OutputBuffer, pCurrentNode->pDataRecord, pCurrentNode->FieldLength);
		OutputBuffer += pCurrentNode->FieldLength;
		*RecordLength += pCurrentNode->FieldLength;

		pCurrentNode = pCurrentNode->pNextNode;
	}
}


void    WriteRecord(short hOBJFile, char *OutputBuffer, short RecordLength)
{
	/* This procedure will write a complete OBJ record to the       */
	/* output file, and also create the checksum.                   */

	unsigned char ChkSum;
	short   iTemp;
	char    *pTemp;

	RecordLength += 2;
	iTemp = RecordLength;
	pTemp = OutputBuffer;
	ChkSum = 0;

	while (iTemp--)
		ChkSum -= *pTemp++;

	*pTemp = ChkSum;
	WriteFile(hOBJFile, OutputBuffer, RecordLength+1);
}


void    WriteOBJFile(ListHandle *pOBJHandle, int hOutputFile)
{
	/* This procedure will traverse the OBJ list and build each     */
	/* record into a complete record with header, data and checksum */
	/* so it can be written to the OutputFile.                      */

	RecordHeader *pRecord;
	char    *OutputBuffer;
	short   *RecordLength;


		/* Allocate the output buffer where the OBJ records     */
		/* will be assembled.  4096 byte should be large enough */
		/* ,none of the records used are larger than this.      */

	OutputBuffer = AllocateMemory(4096);
		/* The record length is stored in the output buffer, at */
		/* the length field in an OBJ record.                   */
	RecordLength = (short *)(OutputBuffer + 1);

		/* Now traverse each record, and create the complete    */
		/* record and write it to the output file.              */
	pRecord = pOBJHandle->pHead;
	while (pRecord != NULL) {
		if( pRecord->pDataRecord != NULL ) {
			*RecordLength = 1;
			*OutputBuffer = pRecord->RecordType;
			CreateRecord(pRecord->pDataRecord, OutputBuffer+3, RecordLength);
			WriteRecord(hOutputFile, OutputBuffer, *RecordLength);
		}
		pRecord = pRecord->pNextNode;
	}

		/* After all records have been written, display the     */
		/* size of the file created, then close the file and    */
		/* free the output buffer.                              */

	printf("%lu bytes.\n", tell(hOutputFile));
	free(OutputBuffer);
}




void    DestroyOBJList(ListHandle *pOBJHandle)
{
	/* This procedure will free the entire OBJ list, all records    */
	/* all data nodes and data buffers and all ListHandles.  The    */
	/* OBJ list will be returned to its original initialised state. */

	RecordHeader *pRecord;
	ListNode *pCurrentNode;

		/* Traverse all records, freeing of its data list nodes */
		/* and then the data list ListHandle, then deleting the */
		/* actual record node itself.                           */

	pRecord = pOBJHandle->pHead;

	while (pRecord != NULL) {
		if( pRecord->pDataRecord != NULL ) {
			pCurrentNode = ((ListHandle *)pRecord->pDataRecord)->pHead;
			while(pCurrentNode != NULL) {
				free( pCurrentNode->pDataRecord);

				pCurrentNode = pCurrentNode->pNextNode;
				DeleteListNode((ListHandle *)pRecord->pDataRecord,NULL);
			}
		free(pRecord->pDataRecord);
		}
		pRecord = pRecord->pNextNode;
		DeleteListNode(pOBJHandle, NULL);
	}
}


void    CreateOBJFile()
{
	/* This procedure is used to create the OBJ file(s) defined by  */
	/* the ParseCommandLine procedure.  It will copy the data from  */
	/* the input file into one or more OBJ files that it will create*/
	/* It is capable of splitting large input files, and also       */
	/* removing the head and tail information.                      */

	unsigned short LengthRemainder;
	unsigned short LengthModulus;
	char    *pInputBuffer;
	ListHandle OBJList;
	int     hOutputFile;

		/* Firstly open the OutputFile and print a message that */
		/* the file is being created.                           */
	hOutputFile = OpenFile(OutputFile, O_CREAT | O_BINARY | O_WRONLY | O_TRUNC, S_IREAD | S_IWRITE);

	printf("Output File : %s  ", OutputFile);


		/* Initialise the OBJList.                              */
	InitialiseList(&OBJList);

		/* Now remove the header by positioning the file        */
		/* pointer and adjusting the DataLength.                */

	if(HeaderLength != 0) {
		if(InformationFlag)
			printf("Removing %u bytes of header data.\n", HeaderLength);

		lseek(hInputFile, HeaderLength, SEEK_SET);
		DataLength -= HeaderLength;
	}

		/* Removing the tail data is simply a matter of         */
		/* subtracting the tail length from the DataLength.     */

	if(TailLength != 0) {
		if(InformationFlag)
			printf("Removing %u bytes of tail data.\n", TailLength);
		DataLength -= TailLength;
	}

		/* Now find how many complete big segments there are in */
		/* the input file.                                      */

	LengthModulus = (unsigned short)(DataLength / 65536);
	LengthRemainder = (unsigned short)(DataLength % 65536);

		/* If there is a remainder then write it out first      */
		/* this also covers files that are smaller than 64k     */

	if(LengthRemainder != 0) {
		pInputBuffer = AllocateMemory(LengthRemainder);

		read(hInputFile, pInputBuffer, LengthRemainder);


		/* Now create the OBJ list, and write the OBJ file.     */

		CreateTHEHDR(&OBJList, InputFile);
		AddSEGDEF(&OBJList, OutputSeg, LengthRemainder);
		if(OutputSeg->GRP_Str != NULL)
			AddGRPDEF(&OBJList, OutputSeg);
		AddPUBDEF(&OBJList, OutputSeg);
		AddLEDATA(&OBJList, OutputSeg, 0, LengthRemainder, pInputBuffer);

		free(pInputBuffer);
	}

		/* Split segments after the first PARA aligned segment  */
		/* should all be BYTE aligned, so modify the align code.*/

	if(SplitFlag)
		OutputSeg->ALIGN_Code = 0x20;

		/* Also set the BIG bit in the align code for all 65536 */
		/* byte segments.                                       */

	if(LengthModulus != 0)
		OutputSeg->ALIGN_Code |= 0x02;

		/* Allocate an input buffer for the file.  And create   */
		/* an OBJ file for each complete segment.               */

	pInputBuffer = AllocateMemory(8192);

	while(LengthModulus != 0) {
		int i;
		unsigned short SegOffset;

		/* SegOffset is used to keep track of the data offset   */
		/* in the segment.                                      */

		SegOffset = 0;

		/* Create a the OBJ file and add the data from the      */
		/* input file.  Then write the completed OBJ list, and  */
		/* loop around again.                                   */

		AddSEGDEF(&OBJList, OutputSeg, 0);

		for(i=0;i<8;i++) {
			read(hInputFile, pInputBuffer, 8192);
			AddLEDATA(&OBJList, OutputSeg, SegOffset, 8192, pInputBuffer);
			SegOffset += 8192;
		}

		LengthModulus--;
	}

	WriteOBJFile(&OBJList, hOutputFile);
	DestroyOBJList(&OBJList);

	close(hOutputFile);
	close(hInputFile);

	free(pInputBuffer);
}


void    CreateBINFile()
{
	printf("Output File : %s  ", OutputFile);

	remove(OutputFile);
	rename(TMPFile, OutputFile);
	TMPInput = 0;

	printf("%lu bytes.\n", filelength(hInputFile));

	close(hInputFile);
}


void main(int argc, char **argv)
{
	printf("\\
Binary to OBJ File Conversion Utility.  Version 1.10\n\\
  (c) N.P.S. Software 1993\n\n");

	InitialiseSLINK(AllocateMemory);

	ParseCommandLine(argc, argv);

	if(DisableOBJ)
		CreateBINFile();
	else
		CreateOBJFile();


	/* The only allocated memory remaining at this point is the     */
	/* OutputSeg structure and its PublicsList.  All OBJ lists have */
	/* been freed.                                                  */

			/* Make MAKE happy.                             */
	Error(E_DONE, ES_NONE, NULL, 0);
}
