/*

Title:          COM2DOS.CPP             Version 1.3
Author:         Shawn A. Clifford,      sac@eng.ufl.edu
Date:           14-DEC-1993
Purpose:        This program will Terminate and Stay Resident.  Its purpose
		is to copy characters from the comm. port to the DOS keyboard
		buffer, either for performing remote commands, or for passing
		information to programs that read from the keyboard.

Usage:          COM2DOS [h] [u] [d] [v] [COM1..COM4]
			H       =   Help
			U       =   Unload (remove from memory)
			D       =   Display some debug information
			V       =   Version/author information
			COMx    =   Com. port to use (1..4); default is COM2

Notes:          This program is written in Borland C/C++ v3.1.  It should be
		capable of being compiled on a C compiler if the inline
		comments are removed.  The peek and poke functions can be/are
		duplicated in most C compilers for the PC.  This was compiled
		using the SMALL memory model, which means that the Code
		Segment and the PSP are the same.

Disclaimer:     The software is provided "as is" and the author accepts no
		responsibility for any damage caused by its use.  This
		program is ShareWare, and therefore may be distributed
		freely, as long as it is not used for commercial purposes and
		is distributed intact in its original form.  Donations are
		welcome.  Licensing for commercial use can be obtained by
		sending $10.00 US to:

			Shawn Clifford
			2021-3 SW 36th Way
			Gainesville, FL  32607

		Or contact me via e-mail at:   sac@eng.ufl.edu
*/


#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <string.h>
#include <dos.h>


#include "serial.h"                     // Serial comm. defs and routines
#include "get_scan.cpp"                 // Converts ASCII to scan code


#define KEYBD_INTR      0x09            // The keyboard interrupt
#define CLOCK_INTR      0x1C            // The clock tick interrupt

#ifdef __cplusplus                      // See the example on 'keep'
   #define __CPPARGS ...
#else
   #define __CPPARGS
#endif



/* Reduce heap and stack to make a smaller program in memory. */
extern unsigned _heaplen = 16;          // *Must* have _some_ heap space
extern unsigned _stklen = 0;            // Be careful with this


/* Global for the offset to the end of resident code. */
static unsigned keep_offset;


/* Global for the segment of the environment space we want to free up. */
static unsigned env_segment;


/* Global definitions for the comport and comm address to use. */
volatile unsigned ComPort = COM2_MASK;  // See serial.h for defs
volatile unsigned ComAddr = COM2_ADDR;


/* Make the pointer to the old clock tick interrupt global so that we */
/* may call it from our replacement handler when we are finished.     */
void interrupt (* old_clock_tick)(__CPPARGS);




/* This routine gets called first by the clock tick interrupt.        */
/* It will copy characters from the serial port to the key buffer.    */

void interrupt com2_to_dos_key_buffer(__CPPARGS)
{
static unsigned char c;
static struct REGPACK regs;

   _disable();                  // Disable interupts

   /* Check if our signature is still in memory.  If not, then uninstall. */

   if ( (peekb(0x0000, 0x04FA) & 0x77) != 0x77) {

      /* Reset the interrupts. */
      setvect(CLOCK_INTR, old_clock_tick);

      /* Free the memory. */
      regs.r_ax = 0x4900;               // Free Allocated Memory function
      regs.r_es = _psp;                 // Point ES at our PSP
      intr(0x21, &regs);                // Call DOS Interrupt
   }
   else

      /* If there is anything to receive, read it and stuff it. */

      if ( serial_status(ComPort) & DATA_READY ) {

	    /* Get a character - stuff a character */
//
// This doesn't work on my machine without TSRCOMM serial driver loaded, and
// that program kills windows (it should work, since it is using int 16).
//
//          receive_char(ComPort, &c);
//
// Aha!  Windows sets the COM speed to unkown when this is loaded.
// Let's fix that here.  This way the program will continue before and
// after windows.  Too bad it doesn't work _with_ windows.  I guess I should
// really only do this if the comm settings change.  I'm assuming that this
// whole clock tick interrupt of mine is being stolen by Windows, so I could
// just check the settings, but hey, it takes as many cycles to check as it
// does to just force it.  Comment this out if you are not going to use
// windows on your machine.
//
	    set_port(ComPort, B9600, NONE, STOP_1, CHAR_8);

// Workaround:  read directly from the port.
	    c = (unsigned char) inportb(ComAddr);

//
// Instead of using:
//
//          keybuf(&c);                 // Stuff the char in the buffer
//
// Use the following.  If your BIOS doesn't support function 5 in intr 16h,
// then go back to using keybuf().
//

	    regs.r_ax = 0x0500;         // Choose service to stuff char
	    regs.r_cx = (get_scan(c)<<8) + c;
					// The char to stuff goes here
					// Scan code in CH, ASCII in CL
	    intr(0x16, &regs);          // Call Keyboard services interrupt
      };

   _enable();                   // Set (enable) interrupts

   /* Call the previous clock tick interrupt routine. */
   old_clock_tick();

} /* com2_to_dos_key_buffer() */



/* This routine checks the command line for options and processes them. */
void getopt(void)
{
static int i;
static char quit = 0;

   /* Check for command line switches. */
   for (i = 0; i < _argc; i++) {

      /* Need help?  (The 'h' option)  */
      if ((*_argv[i] == 'h') || (*_argv[i] == 'H') || (*_argv[i] == '?')) {
	 printf("\nUsage:  %s [h] [u] [d] [COM1..COM4]\n", _argv[0]);
	 printf("\tH = Help (this listing)\n\tU = Unload the TSR\n");
	 printf("\tD = Debug information\n\tV = Version/author info\n");
	 printf("\tCOMx = Comm. port to use (1..4); default is COM2\n");
	 quit = 1;
      }

      /* Check for the COM port */
      if (strncmp(_argv[i], "COM1", 4) == 0 ||
	  strncmp(_argv[i], "com1", 4) == 0) {
		ComPort = COM1_MASK;
		ComAddr = COM1_ADDR;
	  }
      if (strncmp(_argv[i], "COM2", 4) == 0 ||
	  strncmp(_argv[i], "com2", 4) == 0) {
		ComPort = COM2_MASK;
		ComAddr = COM2_ADDR;
	  }
      if (strncmp(_argv[i], "COM3", 4) == 0 ||
	  strncmp(_argv[i], "com3", 4) == 0) {
		ComPort = COM3_MASK;
		ComAddr = COM3_ADDR;
	  }
      if (strncmp(_argv[i], "COM4", 4) == 0 ||
	  strncmp(_argv[i], "com4", 4) == 0) {
		ComPort = COM4_MASK;
		ComAddr = COM4_ADDR;
	  }

      /* Need debug? (The 'd' option)  */
      if ((*_argv[i] == 'd') || (*_argv[i] == 'D')) {
	 printf("\n** Debug Information **\n");
	 printf("Program Segment Prefix:    0%04x0\n", _psp);
	 printf("Segment of end of TSR:     0%04x0\n", keep_offset);
	 printf("Amount of memory used:     %d  bytes\n",
		(keep_offset*16) + 48);
	 printf("Segment of environment:    0%04x0", env_segment);
	 printf("\t(This space is FREEd before TSRing)\n");
      }

      /* Need to unload the TSR? (The 'u' option)  */
      if ((*_argv[i] == 'v') || (*_argv[i] == 'V')) {
	 printf("\nCOM2DOS v1.3  [14-DEC-1993]  by Shawn A. Clifford");
	 printf("  --  sac@eng.ufl.edu\n");
	 quit = 1;                      // Get outta here
      }

      /* Need to unload the TSR? (The 'u' option)  */
      if ((*_argv[i] == 'u') || (*_argv[i] == 'U')) {

	 if ( (peekb(0x0000, 0x04FA) & 0x77) == 0x77) {
	    printf("\nUninstalling COM2DOS ...\n");

	    /* Now signal the TSR to unload itself by erasing our signature. */
	    pokeb(0x0000, 0x04FA, (peekb(0x0000, 0x04FA) & ~0x77));
	 }
	 else
	    printf("\nCOM2DOS is not installed.\n");

	 /* And exit */
	 quit = 1;
      }

   } /* for loop */

   if (quit) exit(0);

   return;

} /* getopt() */



int main (void)
{
static struct REGPACK regs;


   /* From the Borland online help on keep:

   _psp is the starting address of the
   program in memory.  The top of the stack
   is the end of the program.  Using _SS and
   _SP together we can get the end of the
   stack.  You may want to allow a bit of
   saftey space to insure that enough room
   is being allocated ie:
   (_SS + ((_SP + safety space)/16) - _psp)

   */

   /* Calculate the offset from the top of our Code Segment to the */
   /* first byte beyond the resident portion of my code.           */
   keep_offset = (_SS + ((_SP + 0)/16) - _psp);


   /* Get the segment address of environment copied to this program.      */
   /* We want to FREE this memory before TSRing because we don't need it. */
   env_segment = peek(_psp, 44);                // Offset 44 into the PSP


   /* Check and process the command line options. */
   getopt();


   /* Set the COM port the way we want.  See serial.h for settings. */
   set_port(ComPort, B9600, NONE, STOP_1, CHAR_8);


   /* Check for my signature in the Intra-Application Comm. Area */
   if ( (peekb(0x0000, 0x04FA) & 0x77) != 0x77) {

      /* Print a message. */
      printf("\nInstalling COM2DOS ...\n");

      /* Get the address of the current clock tick interrupt. */
      old_clock_tick = getvect(CLOCK_INTR);

      /* Install the new interrupt handler. */
      setvect(CLOCK_INTR, com2_to_dos_key_buffer);

      /* Put our signature in place. */
      pokeb(0x0000, 0x04FA, (peekb(0x0000, 0x04FA) | 0x77));

      /* Get rid of the wasted environment. */
      regs.r_ax = 0x4900;               // Free Allocated Memory function
      regs.r_es = env_segment;          // Point ES at our environment
      intr(0x21, &regs);                // Call DOS Interrupt

      /* Terminate but Stay Resident, baby */
      keep(0, keep_offset);

   }
   else

      /* We are already installed. */
      printf ("\n%cCOM2DOS is already resident!\n", '\007');


   return 0;

}  /* main */
