//////////////////////////////////////////////////////////////////////////////
// WindActs.h
// Definitions of data objects

/************************* object var structures ****************************/
// Data is stored in the following structures
// Notice that a short begins every object type - this short (called vtype)
// identifies what type of object follows

#define	NULLTYPE 0
#define	INTTYPE	1
#define	FLOATTYPE	2
#define	DOUBLETYPE	3
#define	STRINGTYPE	4
#define	BOOLTYPE	5
#define	ERRORTYPE	6
#define	DATETYPE	7
#define	HMSTYPE		8
#define	COMPLEXTYPE	9
#define	EQUATIONTYPE	10
#define	ARRAYTYPE	11

/* Storage for an integer 32 bit */
// a NULL object has vtype=0, and value=0
typedef struct t_int {
        short   vtype;	// vtype=1 for an integer, vtype=0 for a NULL
        long    value;
        } IntObj;

/* Storage for a floating point number */
typedef struct t_float {
        short   vtype;	// vtype = 2 for a float
        float   value;
        } FloatObj;

/* Storage for a double floating point number */
typedef struct t_double {
        short   vtype;	// vtype = 3 for a double
        double  value;
        } DoubleObj;

/* Storage for a string object */
typedef struct t_string {
        short   vtype;	// vtype = 4 for a string
        short   length;
        char    value[1];
        } StringObj;

/* storage for boolean object (same as IntObj)*/
typedef struct  t_boolean {
        short   vtype;	// vtype = 5 for boolean
        long   value;
        } BoolObj;

/* Storage for a error object (Errors are string Objects)*/
typedef struct t_error {
        short   vtype;	// vtype = 6 for an error
        short   length;
        char    value[1];
        } ErrorObj;

/* storage for a date object */
typedef struct  t_date {
        short   vtype;	// vtype = 7 for a date
        short   day;	//1-31
        short   month;	// 1-12
        short   year;	// e.g. 1995
        } DateObj;

/* storage for a Time Object type */
typedef struct t_hms {
        short   vtype;	// vtype = 8 for a time
        short   hours;	//0-23
        short   minutes;	//0-59
        short   seconds;	//0-59
        } HmsObj;

/* storage for a complex number type */
typedef struct t_complex {
        short   vtype;	// vtype = 9 for complex number
        double   realpart;
        double   imagepart;
        } ComplexObj;

/* storage for an equation (same as stringobj)*/
typedef struct t_equation {
        short   vtype;	// vtype = 10 for equation
        short   length;
        char    value[1];   
        } EquationObj;

/* storage for an array (data objects follow at 'Value' position)*/
typedef struct t_array {
        short   vtype;	// vtype = 11 for array
        short   elements;
        char    value[1];
        } ArrayObj;

/* structure for a complete variable object */
// Given a ptr to a DataObj, by examining the tint.vtype field you can
// find what type of data object the ptr points to.
typedef union  varObj {
        struct  t_int  tint;		// integer and null
        struct  t_float tfloat;		// floating point number
        struct  t_double tdouble;	// double precision number
        struct  t_string tstring;	// string object
        struct  t_boolean tboolean;	// boolean object
        struct  t_error terror;		// error object
        struct  t_date tdate;		// date object
        struct  t_hms thms;			// hours minutes seconds object (time)
        struct  t_complex tcomplex;	// complex number object
        struct  t_equation tequation;	// equation object
        struct  t_array tarray;		// array of DataObjs 
        } DataObj;


//////////////////////////////////////////////////////////////////////////////////////////////////
// macros used to define the locking of objects
// These macros lock objects and cast the ptr to the correct types

// global lock a data obj and cast to type DataObj *
#define DataGlobalLock(a)	(DataObj FAR *)GlobalLock(a)

// step to the next data object in an array
#define NextDataObj(a)		(DataObj FAR *)(((char FAR *)(a))+GetVarSize(a))

// return a ptr to the contents of an array as a DataObj *
#define PtrContent(optr)	((DataObj FAR *)(optr->tarray.value))

// move ptr 'a' by 'b' bytes
#define	StepBytes(a,b)	(DataObj *)(((char *)(a))+b)


/////////////////////////////////////////////////////////////////////////////////////////////////
// Functions prototypes for functions in WindActs.c

long GetVarSize(DataObj  FAR *op1);
UINT	RegisterResourceString(HINSTANCE hInst, UINT msgid);
HGLOBAL ProduceActList(HINSTANCE hInst, UINT baseID);
void	RegisterActList(HINSTANCE hInst, UINT baseID,UINT FAR * messID);
void FindActName(HINSTANCE hInst, UINT ActId, UINT baseID, LPSTR pBuff, int len);
void SetWindowFluteName(HWND hWnd, LPCSTR lpBuff);
void SetDlgItemFluteName(HWND hDlg, UINT id, LPCSTR lpBuff);
void ClearWindowFluteName(HWND hWnd);
void ClearDlgItemFluteNames(HWND hDlg);
HGLOBAL	MakeString(LPCSTR lpBuff);
HGLOBAL	GeneralError(HINSTANCE hInst, UINT id);
HGLOBAL MakeInt(long lval);
HGLOBAL MakeDouble(double dval);
HGLOBAL NullObj();
long GetALong(DataObj FAR *optr, BOOL FAR *pTrans);
double GetADouble(DataObj FAR *optr, BOOL FAR *pTrans);
void GetAString(DataObj FAR *optr, BOOL FAR *pTrans, char FAR *pBuff, UINT len);
HGLOBAL MakeArrayN(HGLOBAL FAR *pHands, UINT N);
HGLOBAL	MakeArray2(HGLOBAL	Obj1, HGLOBAL Obj2);
HGLOBAL	MakeArray3(HGLOBAL	Obj1, HGLOBAL Obj2, HGLOBAL Obj3);
HGLOBAL	MakeArray4(HGLOBAL	Obj1, HGLOBAL Obj2, HGLOBAL Obj3, HGLOBAL Obj4);


