//////////////////////////////////////
//  Program Name: KEYMOUSE.CPP
//  Programmer: Charlie Calvert
//  Description: Show how to trap keyboard and mouse input.
//////////////////////////////////////

#define STRICT
#include <windows.h>
#include <windowsx.h>
#include <string.h>
#include <stdio.h>
#pragma warning (disable: 4068)
#pragma warning (disable: 4100)

// ---------------------------------------------------------
// Interface
// ---------------------------------------------------------

// Some variables
static int XVal = 10;
static int YVal = 30;
static char szAppName[] = "KeyMouse";
static HWND MainWindow;

// Some procs
LRESULT CALLBACK _export WndProc(HWND hWindow, UINT Message,
WPARAM wParam, LPARAM lParam);
BOOL Register(HINSTANCE hInst);
HWND Create(HINSTANCE hInst, int nCmdShow);

#define KeyMouse_DefProc    DefWindowProc
void KeyMouse_OnDestroy(HWND hwnd);
void KeyMouse_OnChar(HWND hwnd, UINT ch, int cRepeat);
void KeyMouse_OnKey(HWND hwnd, UINT vk, BOOL fDown,
                    int cRepeat, UINT flags);
void KeyMouse_OnLButtonDown(HWND hwnd, BOOL fDoubleClick,
                            int x, int y, UINT keyFlags);
void KeyMouse_OnLButtonUp(HWND hwnd, int x, int y,
                          UINT keyFlags);
void KeyMouse_OnMouseMove(HWND hwnd, int x,
                          int y, UINT keyFlags);
void KeyMouse_OnPaint(HWND hwnd);
void KeyMouse_OnSysKey(HWND hwnd, UINT vk,
                       BOOL fDown, int cRepeat, UINT flags);


// ---------------------------------------------------------
// Initialization
// ---------------------------------------------------------

//////////////////////////////////////
// Program entry point
//////////////////////////////////////

#pragma argsused
int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInstance,
LPSTR  lpszCmdParam, int nCmdShow)
{
  MSG  Msg;

  if (!hPrevInstance)
     if (!Register(hInst))
      return FALSE;

  MainWindow = Create(hInst, nCmdShow);
  if (!MainWindow)
    return FALSE;

  while (GetMessage(&Msg, NULL, 0, 0))
  {
     TranslateMessage(&Msg);
     DispatchMessage(&Msg);
  }

  return Msg.wParam;
}

//////////////////////////////////////
// Register the window
//////////////////////////////////////
BOOL Register(HINSTANCE hInst)
{
  WNDCLASS WndClass;

  WndClass.style          = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
  WndClass.lpfnWndProc     = WndProc;
  WndClass.cbClsExtra     = 0;
  WndClass.cbWndExtra      = 0;
  WndClass.hInstance       = hInst;
  WndClass.hIcon          = LoadIcon(NULL, IDI_APPLICATION);
  WndClass.hCursor        = LoadCursor(NULL, IDC_ARROW);
  WndClass.hbrBackground  = GetStockBrush(WHITE_BRUSH);
  WndClass.lpszMenuName    = NULL;
  WndClass.lpszClassName  = szAppName;

  return RegisterClass (&WndClass);
}

//////////////////////////////////////
// Create the window
//////////////////////////////////////
HWND Create(HINSTANCE hInstance, int nCmdShow)
{

  HWND hwnd = CreateWindow(szAppName, szAppName,
          WS_OVERLAPPEDWINDOW,
          CW_USEDEFAULT, CW_USEDEFAULT,
          CW_USEDEFAULT, CW_USEDEFAULT,
            NULL, NULL, hInstance, NULL);

  if (hwnd == NULL)
    return hwnd;

  ShowWindow(hwnd, nCmdShow);
  UpdateWindow(hwnd);

  return hwnd;
}

// ---------------------------------------------------------
// WndProc and Implementation
// ---------------------------------------------------------

//////////////////////////////////////
// The Window Procedure
//////////////////////////////////////
LRESULT CALLBACK _export WndProc(HWND hwnd, UINT Message, WPARAM
wParam, LPARAM lParam)
{
  switch(Message)
  {
    HANDLE_MSG(hwnd, WM_DESTROY, KeyMouse_OnDestroy);
    HANDLE_MSG(hwnd, WM_CHAR, KeyMouse_OnChar);
    HANDLE_MSG(hwnd, WM_KEYDOWN, KeyMouse_OnKey);
    HANDLE_MSG(hwnd, WM_KEYUP, KeyMouse_OnKey);
    HANDLE_MSG(hwnd, WM_MOUSEMOVE, KeyMouse_OnMouseMove);
    HANDLE_MSG(hwnd, WM_LBUTTONDBLCLK,  KeyMouse_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_LBUTTONDOWN, KeyMouse_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_LBUTTONUP, KeyMouse_OnLButtonUp);
    HANDLE_MSG(hwnd, WM_PAINT, KeyMouse_OnPaint);
    HANDLE_MSG(hwnd, WM_SYSKEYUP, KeyMouse_OnSysKey);
    HANDLE_MSG(hwnd, WM_SYSKEYDOWN, KeyMouse_OnSysKey);
  default:
    return KeyMouse_DefProc(hwnd, Message, wParam, lParam);
  }
}

//////////////////////////////////////
// Handle WM_DESTROY
//////////////////////////////////////
#pragma argsused
void KeyMouse_OnDestroy(HWND hwnd)
{
  PostQuitMessage(0);
}

//////////////////////////////////////
// Handle regular keyboard hits
// Use if you want to trap the letter keys or number keys.
//////////////////////////////////////
#pragma argsused
void KeyMouse_OnChar(HWND hwnd, UINT ch, int cRepeat)
{
  char S[100];

  HDC DC = GetDC(hwnd);

  sprintf(S,
     "WM_CHAR ==> Ch = %c  cRepeat = %d    ", ch, cRepeat);
  TextOut(DC, XVal, YVal + 20, S, strlen(S));

  ReleaseDC(hwnd, DC);
}

//////////////////////////////////////
// Handle a key press
// Don't try to process letter or number keys here. Instead,
// use WM_CHAR messages.
//////////////////////////////////////
#pragma argsused
void KeyMouse_OnKey(HWND hwnd, UINT vk,
                    BOOL fDown, int cRepeat, UINT flags)
{
  char S[100];

  HDC DC = GetDC(hwnd);

  if (fDown)
    sprintf(S, "WM_KEYDOWN == > vk = %d  fDown = %d cRepeat = %d"
            " flags = %d         ", vk, fDown, cRepeat, flags);
  else
    sprintf(S, "WM_KEYUP == > vk = %d  fDown = %d cRepeat = %d "
            "flags = %d         ", vk, fDown, cRepeat, flags);
  TextOut(DC, XVal, YVal + 40, S, strlen(S));

  ReleaseDC(hwnd, DC);
}

//////////////////////////////////////
// This function is called when the left mouse button is
// click or when the user double-clicks the mouse
//////////////////////////////////////
void KeyMouse_OnLButtonDown(HWND hwnd, BOOL fDoubleClick, int x,
int y, UINT keyFlags)
{
  char S[100];
  HDC PaintDC = GetDC(hwnd);

  if (fDoubleClick)
    sprintf(S,
      "WM_LBUTTONDBLCLK ==> Db = %d x = %d y = %d Flags = %d  ",
      fDoubleClick, x, y, keyFlags);
  else
    sprintf(S,
      "WM_LBUTTONDOWN ==> Db = %d x = %d y = %d Flags = %d  ",
      fDoubleClick, x, y, keyFlags);
  TextOut(PaintDC, XVal, YVal + 100, S, strlen(S));

  ReleaseDC(hwnd, PaintDC);
}

//////////////////////////////////////
// This function is called when the mouse button is released
//////////////////////////////////////
void KeyMouse_OnLButtonUp(HWND hwnd, int x, int y, UINT keyFlags)
{
  char S[100];
  HDC PaintDC = GetDC(hwnd);

  sprintf(S, "WM_LBUTTONUP ==> x = %d y = %d F = %d   ",
          x, y, keyFlags);
  TextOut(PaintDC, XVal, YVal + 120, S, strlen(S));

  ReleaseDC(hwnd, PaintDC);
}

//////////////////////////////////////
// This function is called whenever the mouse moves
//////////////////////////////////////
void KeyMouse_OnMouseMove(HWND hwnd, int x, int y, UINT keyFlags)
{
  char S[100];
  HDC PaintDC = GetDC(hwnd);

  sprintf(S, "WM_MOUSEMOVE ==> x = %d y = %d keyFlags = %d    ",
             x, y, keyFlags);
  TextOut(PaintDC, XVal, YVal + 80, S, strlen(S));

  ReleaseDC(hwnd, PaintDC);
}

//////////////////////////////////////
// Handle WM_PAINT messages.
//////////////////////////////////////
void KeyMouse_OnPaint(HWND hwnd)
{
  PAINTSTRUCT PaintStruct;
  RECT Rect;
  static char *Message[] =
  {
  "WM_CHAR",
  "WM_KEY",
  "WM_SYSKEY",
  "WM_MOUSEMOVE",
  "WM_MOUSEDOWN",
  "WM_MOUSEUP"
  };

  HDC PaintDC = BeginPaint(hwnd, &PaintStruct);
  HFONT OldFont = SelectFont(PaintDC,
                  GetStockObject(OEM_FIXED_FONT));

  GetClientRect(hwnd, &Rect);
  DrawText(PaintDC, "MOUSE AND KEYBOARD DEMONSTRATION", -1,
           &Rect, DT_CENTER);
  Rect.top    = 20;
  Rect.bottom = 40;
  DrawText(PaintDC,
           "(Try experimenting with the mouse and keyboard)",
           -1, &Rect, DT_CENTER);

  SelectFont(PaintDC, OldFont);

  for (int i = 0; i < 6; i++)
    TextOut(PaintDC, XVal, YVal + (20 * (i + 1)),
            Message[i], strlen(Message[i]));

  EndPaint(hwnd, &PaintStruct);
}

//////////////////////////////////////
// This function is called whenever the ALT key is pressed.
//////////////////////////////////////
void KeyMouse_OnSysKey(HWND hwnd, UINT vk, BOOL fDown,
                       int cRepeat, UINT flags)
{
  char S[100];

  HDC DC = GetDC(hwnd);

  if (fDown)
  {
    sprintf(S,"WM_SYSKEYDOWN == > "
            "vk = %d  fDown = %d cRepeat = %d flags = %d       ",
              vk, fDown, cRepeat, flags);
    TextOut(DC, XVal, YVal + 60, S, strlen(S));
    FORWARD_WM_SYSKEYDOWN(hwnd, vk, cRepeat, flags,
                          KeyMouse_DefProc);
  }
  else
  {
    sprintf(S, "WM_SYSKEYUP == > "
            " vk = %d  fDown = %d cRepeat = %d flags = %d      ",
               vk, fDown, cRepeat, flags);
    TextOut(DC, XVal, YVal + 60, S, strlen(S));
    FORWARD_WM_SYSKEYUP(hwnd, vk, cRepeat, flags,
                        KeyMouse_DefProc);
  } // end if

  ReleaseDC(hwnd, DC);
}
