//////////////////////////////////////
//  Program Name: MAPPER.CPP
//  Programmer: Charlie Calvert
//  Description: Mapper windows program
//  Date: 7/18/93
//////////////////////////////////////

#define STRICT
#include <windows.h>
#include <windowsx.h>
#pragma hdrstop
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "mapper.h"
#pragma warning (disable: 4068)
#pragma warning (disable: 4100)

// -------------------------------------
// Interface
// -------------------------------------

// Variables
static char szAppName[] = "Mapper";
static HWND MainWindow;
static HINSTANCE hInstance;
static FARPROC lpfnNewBitProc, lpfnArrowBox;
char Map[MaxY][MaxX];
WNDPROC StatProc1, StatProc2;
HBITMAP Grass, Road;
TDAT Dat;

// -------------------------------------
// Initialization
// -------------------------------------

//////////////////////////////////////
// Program entry point
//////////////////////////////////////
#pragma argsused
int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInstance,
                   LPSTR  lpszCmdParam, int nCmdShow)
{
  MSG  Msg;

  if (!hPrevInstance)
    if (!Register(hInst))
      return FALSE;

  MainWindow = Create(hInst, nCmdShow);
  if (!MainWindow)
    return FALSE;  

  while (GetMessage(&Msg, NULL, 0, 0))
  {
     TranslateMessage(&Msg);
     DispatchMessage(&Msg);
  } 

  return Msg.wParam;
}

//////////////////////////////////////
// Register the window
//////////////////////////////////////
BOOL Register(HINSTANCE hInst)
{
  WNDCLASS WndClass;

  WndClass.style         = CS_HREDRAW | CS_VREDRAW;
  WndClass.lpfnWndProc   = WndProc;
  WndClass.cbClsExtra    = 0;
  WndClass.cbWndExtra    = 0;
  WndClass.hInstance     = hInst;
  WndClass.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
  WndClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
  WndClass.hbrBackground = GetStockBrush(WHITE_BRUSH);
  WndClass.lpszMenuName  = NULL;
  WndClass.lpszClassName = szAppName;

  return RegisterClass (&WndClass);
}


//////////////////////////////////////
// Create the window
//////////////////////////////////////
HWND Create(HINSTANCE hInst, int nCmdShow)
{

  nCmdShow = SW_MAXIMIZE;
  hInstance = hInst;

  HWND hWindow = CreateWindow(szAppName, szAppName,
                              WS_POPUP,
                              CW_USEDEFAULT, CW_USEDEFAULT,
                              CW_USEDEFAULT, CW_USEDEFAULT,
                              NULL, NULL, hInstance, NULL);

  if (hWindow == NULL)
    return hWindow;

  ShowWindow(hWindow, nCmdShow);
  UpdateWindow(hWindow);

  return hWindow;
}

// ---------------------------------------
// WndProc and Implementation
// ---------------------------------------

//////////////////////////////////////
// The Window Procedure
//////////////////////////////////////
LRESULT CALLBACK _export WndProc(HWND hwnd, UINT Message, 
                                 WPARAM wParam, LPARAM lParam)
{
  switch(Message)
  {
    case WM_DIALOG: {
      FARPROC OptionsBox = 
        MakeProcInstance((FARPROC)OptionsDlgProc, hInstance);
        DialogBox(hInstance, "Options", hwnd, (DLGPROC)OptionsBox);
        FreeProcInstance(OptionsBox);
      return FALSE;
    }
    
    case WM_HELP: 
    {
      char *S = "MAPPER\n\n"
          "Use this program to design screens for SNAKO\n\n"
          "RSCREEN takes you one screen to the right\n"
          "LSCREEN takes you one screen to the left\n"
          "LEFT takes you one column to the left\n"
          "RIGHT takes you one column to the right\n"
          "To save a screen open the options DIALOG";
      MessageBox(hwnd, S, "Info", 
                 MB_OK | MB_ICONINFORMATION);
      return FALSE;
    }
    HANDLE_MSG(hwnd, WM_CREATE, Mapper_OnCreate);
    HANDLE_MSG(hwnd, WM_DESTROY, Mapper_OnDestroy);
    HANDLE_MSG(hwnd, WM_LBUTTONDOWN, Mapper_OnLButtonDown);
    HANDLE_MSG(hwnd, WM_RBUTTONDOWN, Mapper_OnRButtonDown);
    HANDLE_MSG(hwnd, WM_PAINT, Mapper_OnPaint);
    default: return Mapper_DefProc(hwnd, Message, wParam, lParam);
  }
}

int ReadArray(HWND hwnd)
{
  FILE * fp;

  if ((fp = fopen("Screen.Dta", "r")) == NULL)   
  {
    MessageBox(hwnd, "Can't find SCREEN.DTA" , NULL, MB_OK);
    return 0;
  }

  fread(&Map, sizeof(Map), 1, fp);
  fclose(fp);

  for (int i = 0; i < MaxY; i++)
    for (int j = 0; j < MaxX; j++)  
      if (Map[i][j] != 1)
        Map[i][j] = 2;

  return 1;
}

//////////////////////////////////////
// Handle WM_CREATE
//////////////////////////////////////
#pragma argsused
BOOL Mapper_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct)
{
  Grass = LoadBitmap(hInstance, "Grass");
  Road = LoadBitmap(hInstance, "Road");

  if ((!Grass) || (!Road))
  {
    MessageBox(hwnd, "Bitmaps missing!", 
               "Fatal", MB_OK | MB_ICONSTOP);
    return FALSE;
  }

  int X = GetSystemMetrics(SM_CXSCREEN);

  if (X == 1024)
  {
    Dat.GrassX = Dat.GrassY = 32;
    Dat.SizeX = Dat.SizeY = 32;
  }

  if (X == 800)
  {
    Dat.GrassX = Dat.GrassY = 25;
    Dat.SizeX = Dat.SizeY = 25;
  }

  if (X == 640)
  {
    Dat.GrassX = Dat.GrassY = 20;
    Dat.SizeX = Dat.SizeY = 20;
  }

  Dat.SPos = 0;
  Dat.MenuSpace = 3;

  lpfnArrowBox = 
     MakeProcInstance((FARPROC)ArrowDlgProc, hInstance);
     CreateDialog(hInstance, "Arrows", hwnd, (DLGPROC)lpfnArrowBox);
     ReadArray(hwnd);

  return TRUE;
}

//////////////////////////////////////
// Handle WM_DESTROY
//////////////////////////////////////
#pragma argsused
void Mapper_OnDestroy(HWND hwnd)
{
  DeleteObject(Road);
  DeleteObject(Grass);
  FreeProcInstance(lpfnArrowBox);
  PostQuitMessage(0);
}

//////////////////////////////////////
// OnRButtonDown
//////////////////////////////////////
#pragma argsused
void Mapper_OnRButtonDown(HWND hwnd, BOOL fDoubleClick, 
                          int x, int y, UINT keyFlags)
{
  char BoxX = (x / Dat.GrassX) + Dat.SPos;
  char BoxY = (y / Dat.GrassY) - Dat.MenuSpace;
  Map[BoxY][BoxX] = 2;
  InvalidateRect(hwnd, NULL, FALSE);
}

//////////////////////////////////////
// OnLButtonDown
//////////////////////////////////////
#pragma argsused
void Mapper_OnLButtonDown(HWND hwnd, BOOL fDoubleClick, 
                          int x, int y, UINT keyFlags)
{
  char BoxX = (x / Dat.GrassX) + Dat.SPos;
  char BoxY = (y / Dat.GrassY) - Dat.MenuSpace;
  Map[BoxY][BoxX] = 1;
  InvalidateRect(hwnd, NULL, FALSE);
}

//////////////////////////////////////
// PAINT Bitmaps
// The GrassX * MenuSpace leaves room for menu
//////////////////////////////////////
void PaintBitmaps(HDC PaintDC, HDC GrassDC, HDC RoadDC)
{
  for (int i = 0; i < MaxY; i++)
    for (int j = Dat.SPos; j < MaxX; j++)
    {
      if (Map[i][j] == 1)
        BitBlt(PaintDC, (j - Dat.SPos) * Dat.GrassX, 
               (i * Dat.GrassY) + (Dat.GrassX * Dat.MenuSpace),
               Dat.GrassX, Dat.GrassY, GrassDC, 0, 0, SRCCOPY);
      else
        BitBlt(PaintDC, (j - Dat.SPos) * Dat.GrassX, 
               (i * Dat.GrassY) + (Dat.GrassX * Dat.MenuSpace),
               Dat.GrassX, Dat.GrassY, RoadDC, 0, 0, SRCCOPY);
    }
}

//////////////////////////////////////
// WM_PAINT
//////////////////////////////////////
void Mapper_OnPaint(HWND hwnd)
{
  PAINTSTRUCT PaintStruct;
  HBITMAP OldGrass, OldRoad, OldGrass1, OldRoad1;

  HDC PaintDC = BeginPaint(hwnd, &PaintStruct);

  HDC TGrassDC = CreateCompatibleDC(PaintDC);
  HDC TRoadDC = CreateCompatibleDC(PaintDC);
  HDC GrassDC = CreateCompatibleDC(PaintDC);
  HDC RoadDC = CreateCompatibleDC(PaintDC);
  HBITMAP TGrass = 
    CreateCompatibleBitmap(PaintDC, Dat.GrassX, Dat.GrassY);
  HBITMAP TRoad = 
    CreateCompatibleBitmap(PaintDC, Dat.GrassX, Dat.GrassY);

  OldGrass = SelectBitmap(TGrassDC, Grass);
  OldRoad = SelectBitmap(TRoadDC, Road);
  OldGrass1 = SelectBitmap(GrassDC, TGrass);
  OldRoad1 = SelectBitmap(RoadDC, TRoad);

  StretchBlt(GrassDC, 0, 0, Dat.GrassX, Dat.GrassY,
             TGrassDC, 0, 0, BITWIDTH, BITHEIGHT, SRCCOPY);
  StretchBlt(RoadDC, 0, 0, Dat.GrassX, Dat.GrassY,
             TRoadDC, 0, 0, BITWIDTH, BITHEIGHT, SRCCOPY);

  PaintBitmaps(PaintDC, GrassDC, RoadDC);

  SelectBitmap(TGrassDC, OldGrass);
  SelectBitmap(TRoadDC, OldRoad);
  SelectBitmap(GrassDC, OldGrass1);
  SelectBitmap(RoadDC, OldRoad1);
  DeleteObject(TRoad);
  DeleteObject(TGrass);

  DeleteDC(GrassDC);
  DeleteDC(RoadDC);
  DeleteDC(TGrassDC);
  DeleteDC(TRoadDC);

  EndPaint(hwnd, &PaintStruct);
}

// -------------------------------------
// The Dialog procs
// -------------------------------------

//////////////////////////////////////
// Save Array
//////////////////////////////////////
int SaveArray(HWND hwnd)
{
  FILE * fp;

  if ((fp = fopen("screen.dta", "w+")) == NULL)
  {
    MessageBox(hwnd, "Can't save SCREEN.DTA", NULL, MB_OK);
    return 0;
  }

  fwrite(&Map, sizeof(Map), 1, fp);
  fclose(fp);

  MessageBox(hwnd, "SCREEN.DTA saved", "Success", MB_OK);

  return 1;
}

//////////////////////////////////////
//  The Options Dialog Procedure
//////////////////////////////////////
#pragma argsused
BOOL CALLBACK OptionsDlgProc(HWND hDlg, WORD Message, 
                             WPARAM wParam, LPARAM lParam)
{
  char S[100];
  HWND hwnd1, hwnd2;

  switch(Message)
  {
    case WM_DESTROY:
      hwnd1 = GetDlgItem(hDlg, ID_BITMAP1);
      hwnd2 = GetDlgItem(hDlg, ID_BITMAP2);
      SubclassWindow(hwnd1, StatProc1);
      SubclassWindow(hwnd2, StatProc2);
      FreeProcInstance(lpfnNewBitProc);
      return TRUE;
      
    case WM_INITDIALOG:
      itoa(Dat.GrassX, S, 10);
      SetWindowText(GetDlgItem(hDlg, ID_XSIZE), S);
      itoa(Dat.GrassY, S, 10);
      SetWindowText(GetDlgItem(hDlg, ID_YSIZE), S);

      itoa(MaxX, S, 10);
      SetWindowText(GetDlgItem(hDlg, ID_NUMX), S);
      itoa(MaxY, S, 10);
      SetWindowText(GetDlgItem(hDlg, ID_NUMY), S);

      lpfnNewBitProc = 
      MakeProcInstance(FARPROC(BitmapProc), hInstance);
      hwnd1 = GetDlgItem(hDlg, ID_BITMAP1);
      StatProc1 = SubclassWindow(hwnd1, lpfnNewBitProc);
      hwnd2 = GetDlgItem(hDlg, ID_BITMAP2);
      StatProc2 = SubclassWindow(hwnd2, lpfnNewBitProc);
      return TRUE; 

    case WM_COMMAND:
      switch(wParam)
      {
        case IDOK:
        case IDCANCEL:
          EndDialog(hDlg, TRUE);
          return TRUE;
     
        case ID_SAVE:
          SaveArray(hDlg);          
          return TRUE;

        case ID_HELP:
          PostMessage(GetParent(hDlg), WM_HELP, 0, 0);
          return TRUE;
      }
  }
  return FALSE;
}


BOOL ExitMapper(HWND hDlg)
{
  int Result = MessageBox(hDlg, "Are you sure you want to quit?",
                 "Are we thinking?", MB_YESNO | MB_ICONQUESTION);
  if (Result == IDYES)
  {
    SendMessage(GetParent(hDlg), WM_CLOSE, 0, 0);
    return TRUE;
  }
  else
    return FALSE;
}

//////////////////////////////////////
// HandleCommand
//////////////////////////////////////
BOOL HandleCommand(HWND hDlg, WPARAM wParam)
{
  switch(wParam)
  {
    case IDOK:
    case IDCANCEL:
      PostMessage(hDlg, WM_CLOSE, 0, 0);
      return TRUE;

    case ID_RIGHT:
      Dat.SPos++;
      InvalidateRect(GetParent(hDlg), NULL, TRUE);
      return TRUE;

    case ID_LEFT:
      Dat.SPos--;
      if (Dat.SPos < 0)
        Dat.SPos = 0;
      InvalidateRect(GetParent(hDlg), NULL, TRUE);
      return TRUE;

    case ID_RSCREEN:
      if (Dat.SPos < JUMPSPACE)
        Dat.SPos = JUMPSPACE;
      else
        if (Dat.SPos < JUMPSPACE * 2)
          Dat.SPos = JUMPSPACE * 2;
        else
          if (Dat.SPos < JUMPSPACE * 3)
            Dat.SPos = JUMPSPACE * 3;
          else
            Dat.SPos = JUMPSPACE * 3;
      InvalidateRect(GetParent(hDlg), NULL, TRUE);
      return TRUE;

    case ID_LSCREEN:
      if (Dat.SPos <= JUMPSPACE)
        Dat.SPos = 0;
      else
        if (Dat.SPos <= JUMPSPACE * 2)
          Dat.SPos = JUMPSPACE;
        else
          if (Dat.SPos <= JUMPSPACE * 3)
            Dat.SPos = JUMPSPACE * 2;
          else
            Dat.SPos = JUMPSPACE * 3;
      InvalidateRect(GetParent(hDlg), NULL, TRUE);
      return TRUE;

    case ID_DIALOG:
      PostMessage(GetParent(hDlg), WM_DIALOG, 0, 0);
      break;

    case ID_HELP:
      PostMessage(GetParent(hDlg), WM_HELP, 0, 0);
      break;
       
    case ID_QUIT:
      ExitMapper(hDlg);
      return TRUE;
  }

  return FALSE;    
}

//////////////////////////////////////
//  The Bitmap Dialog Procedure
//////////////////////////////////////
LRESULT CALLBACK _export BitmapProc(HWND hwnd, UINT Message,
                                    WPARAM wParam, LPARAM lParam)
{
  WORD id;
  LRESULT lResult = 0;
  BOOL CallOrig = TRUE;
  HDC PaintDC, MemDC;
  HBITMAP OldBitmap;
  PAINTSTRUCT PaintStruct;

  id = GetWindowWord(hwnd, GWW_ID);

  switch(Message)
  {
     BITMAP Bits;

     case WM_PAINT:
       PaintDC = BeginPaint(hwnd, &PaintStruct);
       
       MemDC = CreateCompatibleDC(PaintDC);
       switch(id)
       {
         case ID_BITMAP1:
           OldBitmap = SelectBitmap(MemDC, Road);
           break;
         case ID_BITMAP2:
           OldBitmap = SelectBitmap(MemDC, Grass);
           break;
         case ID_BITMAP3:
           break;
         case ID_BITMAP4:
           break;
       }
       GetObject(Grass, sizeof(BITMAP), &Bits);
       BitBlt(PaintDC, 1, 1, Bits.bmWidth, Bits.bmHeight,
              MemDC, 0, 0, SRCCOPY);
       SelectBitmap(MemDC, OldBitmap);
       DeleteDC(MemDC);  

       EndPaint(hwnd, &PaintStruct);
       CallOrig = FALSE;
       break;
   }

   if (CallOrig)
    if (id == ID_BITMAP1)
      lResult = CallWindowProc(StatProc1, hwnd, 
                               Message, wParam, lParam);
    else
      lResult = CallWindowProc(StatProc2, hwnd, 
                               Message, wParam, lParam);

  return lResult;
}

//////////////////////////////////////
//  The Arrow Dialog Procedure
//////////////////////////////////////
#pragma argsused
BOOL CALLBACK ArrowDlgProc(HWND hDlg, WORD Message, 
                           WPARAM wParam, LPARAM lParam)
{
  switch(Message)
  {                                      
    case WM_COMMAND:
      return HandleCommand(hDlg, wParam);

    case WM_CLOSE:
      ExitMapper(hDlg);
      return TRUE;
  }
  return FALSE;
}
