// =========================================================
// Module Name: GRUNT.CPP
// Programmer: Charlie Calvert
// Description: Do the grunt work of moving the snake
// =========================================================

#define STRICT
#include <windowsx.h>
#define hdrstop
#include <string.h>
#include <stdlib.h>
#include "grunt.h"
#pragma warning (disable : 4100)
#pragma warning (disable : 4068)

// local variables
BOOL NewSect;
static int VK_p = 112;
static unsigned int SnakeTimer = 1;
static long TotalClicks = 0;
static int Sections;
static int NumPrizes;
static TPrize Prizes[MAXPRIZES];
static TSectInfo SectInfo[MAXSECTIONS];
TTurn TurnList[25];
int NumTurns;

// Global variables
extern div_t NumXSects;
extern div_t NumYSects;
extern int Size;
extern int MaxCols;
extern int MaxRows;
extern HBITMAP Head, Body, OneHundred;

//////////////////////////////////////
// GameWndProc
//////////////////////////////////////
LRESULT CALLBACK __export GameWndProc(HWND hWindow, UINT Message,
                                    WPARAM wParam, LPARAM lParam)
{
  switch(Message)
  {
    HANDLE_MSG(hWindow, WM_CREATE, Game_OnCreate);
    HANDLE_MSG(hWindow, WM_CHAR, Game_OnChar);
    HANDLE_MSG(hWindow, WM_KEYDOWN, Game_OnKey);
    HANDLE_MSG(hWindow, WM_PAINT, Game_OnPaint);
    HANDLE_MSG(hWindow, WM_TIMER, Game_OnTimer);
    default:
      return Game_DefProc(hWindow, Message, wParam,lParam);
  }
}

//////////////////////////////////////
// WM_CREATE
//////////////////////////////////////
#pragma argsused
BOOL Game_OnCreate(HWND hwnd, CREATESTRUCT FAR* lpCreateStruct)
{
  if (!SetTimer(hwnd, SnakeTimer, 125, NULL))
  {
    MessageBox(hwnd, "No Timers Available", "Snake Info", MB_OK);
    return FALSE;
  }

  NumPrizes = 0;
  InitializeSections();

  return TRUE;
}

//////////////////////////////////////
// WM_DESTROY
//////////////////////////////////////
void Game_OnDestroy(HWND hWindow)
{
  KillTimer(hWindow, SnakeTimer);
}

//////////////////////////////////////
// WM_CHAR
//////////////////////////////////////
#pragma argsused
void Game_OnChar(HWND hWindow, UINT ch, int cRepeat)
{
  if (ch == UINT(VK_p))
    KillTimer(hWindow, SnakeTimer);
}

//////////////////////////////////////
// WM_KEYDOWN
//////////////////////////////////////
#pragma argsused
void Game_OnKey(HWND hWindow, UINT vk,
                BOOL fDown, int cRepeat, UINT flags)
{
  switch(vk)
  {
    case VK_DOWN: GetOldDir(Down); break;
    case VK_UP: GetOldDir(Up); break;
    case VK_LEFT: GetOldDir(Left); break;
    case VK_RIGHT: GetOldDir(Right); break;
  }
  MoveBitmap(hWindow);
}

//////////////////////////////////////
// WM_PAINT
//////////////////////////////////////
void Game_OnPaint(HWND hWindow)
{
  PAINTSTRUCT PaintStruct;

  HDC DC = BeginPaint(hWindow, &PaintStruct);

  HDC PicDC = CreateCompatibleDC(DC);

  // Draw Head
  HBITMAP OldBmp = SelectBitmap(PicDC, Head);
  BitBlt(DC, SectInfo[0].Col, SectInfo[0].Row,
         Size, Size, PicDC, 0, 0, SRCINVERT);
  SelectObject(PicDC, OldBmp);

  // Draw Body
  OldBmp = SelectBitmap(PicDC, Body);
  for (int i = 1; i <= Sections; i++)
    BitBlt(DC, SectInfo[i].Col, SectInfo[i].Row,
           Size, Size, PicDC, 0, 0, SRCINVERT);

  SelectObject(PicDC, OldBmp);
  DeleteDC(PicDC);

  EndPaint(hWindow, &PaintStruct);

  SectInfo[0].DirChange = FALSE;
}

//////////////////////////////////////
// WM_TIMER
//////////////////////////////////////
#pragma argsused
void Game_OnTimer(HWND hwnd, UINT id)
{
  HBITMAP SaveBmp;

  SetFocus(hwnd);

  if (id == SnakeTimer)
  {
    MoveBitmap(hwnd);
    TotalClicks++;

    if ((TotalClicks % 10) == 0)
    {
      SetSections();
      SetPrizes();

      HDC PaintDC = GetDC(hwnd);
      HDC PicDC = CreateCompatibleDC(PaintDC);

      NewSect = TRUE;

      // Paint new prize
      SaveBmp = SelectBitmap(PicDC, OneHundred);
      BitBlt(PaintDC, Prizes[NumPrizes].Col,
         Prizes[NumPrizes].Row, 32, 32, PicDC, 0, 0, SRCINVERT);
      SelectObject(PicDC, SaveBmp);

      DeleteDC(PicDC);
      ReleaseDC(hwnd, PaintDC);
    }
  }
}

//--------------------------------------
// The Implementation
//--------------------------------------

//////////////////////////////////////
// Called by MoveBitmap
// Keeps track of internal Col and Row
// for each section of snake
//////////////////////////////////////
void SetColRow(void)
{
  int i;

 for (i = 0; i <= Sections; i++)
  {
    SectInfo[i].OldCol = SectInfo[i].Col;
    SectInfo[i].OldRow = SectInfo[i].Row;

    switch(SectInfo[i].Dir)
    {
      case Up: SectInfo[i].Row -= Size; break;
      case Down: SectInfo[i].Row += Size; break;
      case Left: SectInfo[i].Col -= Size; break;
      case Right: SectInfo[i].Col += Size; break;
    }
  }

  for (i = Sections; i > 0; i--)
  {
    if (SectInfo[i - 1].DirChange)
    {
      SectInfo[i].DirChange = TRUE;
      SectInfo[i].Dir = SectInfo[i - 1].Dir;
      SectInfo[i - 1].DirChange = FALSE;
    }
  }
}

//////////////////////////////////////
// Called by MoveBitmap
// Keeps track of which section of
// snake will turn next
//////////////////////////////////////
void SetNewTurnSection(void)
{
  for (int i = 1; i <= Sections; i++)
  {
    if (TurnList[NumTurns].SectChangeDir[i] == TRUE)
    break;
  }
  TurnList[NumTurns].SectChangeDir[i] = FALSE;
  i++;
  TurnList[NumTurns].SectChangeDir[i] = TRUE;
}

//////////////////////////////////////
// Make a noise and set constants
// when a collision occurs
//////////////////////////////////////
void Boom(HWND hwnd)
{
  memset(SectInfo, 0, sizeof(SectInfo));
  Sections = 0;
  NumPrizes = 0;
  for (int i = 0; i < 10; i++)
    MessageBeep(-1);
  InitializeSections();
  InvalidateRect(hwnd, NULL, TRUE);
}

//////////////////////////////////////
// Erase prizes
//////////////////////////////////////
void WhiteOut(HWND hwnd, int i)
{
  HDC PaintDC = GetDC(hwnd);
  HDC MemDC = CreateCompatibleDC(PaintDC);

  HBITMAP SaveBmp = SelectBitmap(MemDC, OneHundred);
  BitBlt(PaintDC, Prizes[i].Col,
         Prizes[i].Row, 32, 32, MemDC, 0, 0, SRCINVERT);
  SelectObject(MemDC, SaveBmp);

  DeleteDC(MemDC);
  ReleaseDC(hwnd, PaintDC);
}

//////////////////////////////////////
//
//////////////////////////////////////
void CheckForCollision(HWND hwnd)
{

  if (SectInfo[0].Col < 0) Boom(hwnd);
  if (SectInfo[0].Row < 0) Boom(hwnd);
  if (SectInfo[0].Col + Size > MaxCols) Boom(hwnd);
  if (SectInfo[0].Row + Size > MaxRows) Boom(hwnd);

  // See if Snake hit his own tail
  for (int i = 1; i <= Sections; i++)
    if (SectInfo[i].Col == SectInfo[0].Col &&
        SectInfo[i].Row == SectInfo[0].Row)
      Boom(hwnd);

  // Erase a prize when he hits it
  for (i = 0; i <= NumPrizes; i++)
  {
    if (Prizes[i].Exists)
    {
      if ( ((SectInfo[0].Row >= Prizes[i].Row) &&
            (SectInfo[0].Row <= Prizes[i].Row + (Size - 1))) &&
           ((SectInfo[0].Col >= Prizes[i].Col) &&
            (SectInfo[0].Col <= Prizes[i].Col + (Size - 1))) )
      {
        Prizes[i].Exists = FALSE;
        WhiteOut(hwnd, i);
      }
    }
  }
}

//////////////////////////////////////
// Init procedure at start of game
//////////////////////////////////////
void InitializeSections()
{
  int i, StartCol, StartRow;

  Sections = 5;
  StartCol = Size * 3;
  StartRow = Size * 3;

  for (i = 0; i <= Sections; i++)
  {
    SectInfo[i].Dir = Right;
    SectInfo[i].DirChange = FALSE;
    SectInfo[i].Col = StartCol - (Size * i);
    SectInfo[i].Row = StartRow;
    SectInfo[i].OldCol = SectInfo[i].Col;
    SectInfo[i].OldRow = SectInfo[i].Row;
  }
  NewSect = FALSE;
}

//////////////////////////////////////
//
//////////////////////////////////////
void GetOldDir(TDir NewDir)
{
  TurnList[NumTurns].Dir = NewDir;
  TurnList[NumTurns].SectChangeDir[1] = TRUE;

  SectInfo[0].OldDir = SectInfo[0].Dir;
  SectInfo[0].Dir = NewDir;
  SectInfo[0].DirChange = TRUE;
  TurnList[0].TurnCol = SectInfo[0].Col;
  TurnList[0].TurnRow = SectInfo[0].Row;
}

//////////////////////////////////////
// Draw the Snake
//////////////////////////////////////
void MoveBitmap(HWND hwnd)
{
  SetColRow();

  HDC DC = GetDC(hwnd);
  HDC PicDC = CreateCompatibleDC(DC);

  CheckForCollision(hwnd);

  // Paint head
  HBITMAP OldBmp = SelectBitmap(PicDC, Head);
  BitBlt(DC, SectInfo[0].OldCol, SectInfo[0].OldRow,
         Size, Size, PicDC, 0, 0, SRCINVERT);
  BitBlt(DC, SectInfo[0].Col, SectInfo[0].Row,
         Size, Size, PicDC, 0, 0, SRCINVERT);
  SelectObject(PicDC, OldBmp);

  // Paint body
  OldBmp = SelectBitmap(PicDC, Body);
  BitBlt(DC, SectInfo[1].Col, SectInfo[1].Row,
         Size, Size, PicDC, 0, 0, SRCINVERT);
  BitBlt(DC, SectInfo[Sections].Col, SectInfo[Sections].Row,
         Size, Size, PicDC, 0, 0, SRCINVERT);

  // If he grew
  if (NewSect)
  {
    BitBlt(DC, SectInfo[Sections].Col, SectInfo[Sections].Row,
         Size, Size, PicDC, 0, 0, SRCINVERT);
    NewSect = FALSE;
  }

  SelectObject(PicDC, OldBmp);

  SectInfo[0].DirChange = FALSE;

  DeleteDC(PicDC);
  ReleaseDC(hwnd, DC);

  SetNewTurnSection();
}

//////////////////////////////////////
//
//////////////////////////////////////
void SetPrizes(void)
{
  NumPrizes++;

  Prizes[NumPrizes].Exists = TRUE;
  Prizes[NumPrizes].Row = (rand() % NumYSects.quot) * Size;
  Prizes[NumPrizes].Col = (rand() % NumXSects.quot) * Size;
}

//////////////////////////////////////
//
//////////////////////////////////////
void SetSections(void)
{
  Sections++;

  SectInfo[Sections].Dir = SectInfo[Sections - 1].Dir;
  SectInfo[Sections].DirChange = FALSE;

  switch (SectInfo[Sections].Dir)
  {
    case Left:
    {
      SectInfo[Sections].Col = SectInfo[Sections - 1].Col + Size;
      SectInfo[Sections].Row = SectInfo[Sections - 1].Row;
      break;
    }

    case Right:
    {
      SectInfo[Sections].Col = SectInfo[Sections - 1].Col - Size;
      SectInfo[Sections].Row = SectInfo[Sections - 1].Row;
      break;
    }

    case Up:
    {
      SectInfo[Sections].Col = SectInfo[Sections - 1].Col;
      SectInfo[Sections].Row = SectInfo[Sections - 1].Row + Size;
      break;
    }

    case Down:
    {
      SectInfo[Sections].Col = SectInfo[Sections - 1].Col;
      SectInfo[Sections].Row = SectInfo[Sections - 1].Row - Size;
      break;
    }
  } // end switch
  SectInfo[Sections].OldCol = SectInfo[Sections - 1].Col;
  SectInfo[Sections].OldRow = SectInfo[Sections - 1].Row;
}
