//////////////////////////////////////
// WINSIZE.CPP -- What Size is the Window?
// programmer: Charlie Calvert
// date: 3/26/93
//////////////////////////////////////
#define STRICT
#include <windows.h>
#include <windowsx.h>
#pragma hdrstop
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "winsize.h"
#pragma warning (disable: 4068)

// ----------------------------------------------------
// Setup
// ----------------------------------------------------

static char szAppName[] = "WinSize";
static HWND MainWindow;
static HINSTANCE hInstance;

//////////////////////////////////////
// Program entry point
//////////////////////////////////////
#pragma argsused
int PASCAL WinMain(HINSTANCE hInst, HINSTANCE hPrevInstance,
                   LPSTR  lpszCmdParam, int nCmdShow)
{
  MSG  Msg;

  hInstance = hInst;

  if (!hPrevInstance)
    if (!Register(hInst))
      return FALSE;

  MainWindow = Create(hInst, nCmdShow);
  if (!MainWindow)
    return FALSE;

  while (GetMessage(&Msg, NULL, 0, 0))
  {
     TranslateMessage(&Msg);
     DispatchMessage(&Msg);
  }

  return Msg.wParam;
}

//////////////////////////////////////
// Register the window
//////////////////////////////////////
BOOL Register(HINSTANCE hInst)
{
  WNDCLASS WndClass;

  WndClass.style          = CS_HREDRAW | CS_VREDRAW;
  WndClass.lpfnWndProc    = WndProc;
  WndClass.cbClsExtra    = 0;
  WndClass.cbWndExtra    = 0;
  WndClass.hInstance     = hInst;
  WndClass.hIcon         = LoadIcon(NULL, IDI_APPLICATION);
  WndClass.hCursor       = LoadCursor(NULL, IDC_ARROW);
  WndClass.hbrBackground = GetStockBrush(WHITE_BRUSH);
  WndClass.lpszMenuName   = "MENU";
  WndClass.lpszClassName = szAppName;

  return RegisterClass (&WndClass);
}

//////////////////////////////////////
// Create the window
//////////////////////////////////////
HWND Create(HINSTANCE hInst, int nCmdShow)
{
  HWND hWindow = CreateWindow(szAppName, szAppName,
                    WS_OVERLAPPEDWINDOW,
                    CW_USEDEFAULT, CW_USEDEFAULT,
                    CW_USEDEFAULT, CW_USEDEFAULT,
                      NULL, NULL, hInst, NULL);

  if (hWindow == NULL)
    return hWindow;

  ShowWindow(hWindow, nCmdShow);
  UpdateWindow(hWindow);

  return hWindow;
}

//----------------------------------------------------------
// The Implementation
//----------------------------------------------------------

//----------------------------------------------------------
// The WndProc
//----------------------------------------------------------

LRESULT CALLBACK _export WndProc(HWND hwnd, UINT Message,
                                 WPARAM wParam, LPARAM lParam)
{
  switch (Message)
  {
    HANDLE_MSG(hwnd, WM_COMMAND, WinSize_OnCommand);
    HANDLE_MSG(hwnd, WM_CREATE, WinSize_OnCreate);
    HANDLE_MSG(hwnd, WM_PAINT, WinSize_OnPaint);
    HANDLE_MSG(hwnd, WM_SIZE, WinSize_OnSize);
    HANDLE_MSG(hwnd, WM_DESTROY, WinSize_OnDestroy);
    default:
      return WinSize_DefProc(hwnd, Message, wParam, lParam);
  }
}

//////////////////////////////////////
// Handle WM_CREATE
//////////////////////////////////////
#pragma argsused
BOOL WinSize_OnCreate(HWND hwnd, CREATESTRUCT FAR*
lpCreateStruct)
{
  HINSTANCE hIn;

  hIn = lpCreateStruct->hInstance;

  lpfnSizeBox = MakeProcInstance((FARPROC)SizeBoxProc, hIn);

  BlueBrush = CreateSolidBrush(RGB(0, 255, 255));
  GreenBrush = CreateSolidBrush(RGB(255, 255, 127));
  return TRUE;
}

//////////////////////////////////////
// Handle WM_DESTROY
//////////////////////////////////////
#pragma argsused
void WinSize_OnDestroy(HWND hwnd)
{
  DeleteObject(BlueBrush);
  DeleteObject(GreenBrush);
  FreeProcInstance(lpfnSizeBox);
  PostQuitMessage(0);
}

//////////////////////////////////////
// WM_COMMAND
//////////////////////////////////////
#pragma argsused
void WinSize_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT
codeNotify)
{
  switch(id)
  {
    case CM_SIZEBOX:
      if (hSizeBox == 0)
      {
        hSizeBox = CreateDialog(hInstance, "SizeDialog",
                                hwnd,(DLGPROC)lpfnSizeBox);
        InvalidateRect(hwnd, NULL, FALSE);
      }
      break;

    case CM_MAKESHAPE:
      lpfnMakeShape = MakeProcInstance((FARPROC)MakeShapeProc,
                                       hInstance);
      DialogBox(hInstance, "MakeShape",
                hwnd,(DLGPROC)lpfnMakeShape);
      FreeProcInstance(lpfnMakeShape);
      InvalidateRect(hwnd, NULL, FALSE);
      break;

    case CM_EXIT:
      DestroyWindow(hwnd);
      break;
  }
}

//////////////////////////////////////
//  If the Size Dialog is open, this will fill it in with
//  the size of the window as expressed in each of the
//  different mapping modes
//////////////////////////////////////
void FillSizeBox(HWND hwnd, HDC PaintDC)
{
  TSIZEDATA    SizeData;
  for (int i = 0; i < 6; i++)
  {
    SetMapMode(PaintDC, Modes[i]);
    GetClientRect(hwnd, &SizeData[i]);
    DPtoLP(PaintDC, (LPPOINT)&SizeData[i], 2);
  }
  SendMessage(hSizeBox, WM_SETDATA, 0, (long)&SizeData);
}

//////////////////////////////////////
// WM_PAINT
//////////////////////////////////////
void WinSize_OnPaint(HWND hwnd)
{
  RECT         Rect;
  HDC          PaintDC ;
  PAINTSTRUCT  ps ;

  PaintDC = BeginPaint (hwnd, &ps) ;
  SetMapMode (PaintDC, Modes[ModeCase]);
  GetClientRect(hwnd, &Rect);

  switch (OriginMode)
  {
    case ORG_UPLEFT :
      SetViewportOrg(PaintDC, 0, 0);
      break;

    case ORG_LOWLEFT:
      SetViewportOrg(PaintDC, 0, Rect.bottom);
      break;

    case ORG_CENTER:
      SetViewportOrg(PaintDC,Rect.right/2,Rect.bottom / 2);
      break;
  }

  HBRUSH OldBrush = SelectBrush(PaintDC, BlueBrush);

  switch (Shape)
  {
    case RECTANGLE:
      Rectangle(PaintDC, RLeft, RTop, RRight, RBottom);
      break;

    case CIRCLE:
      Ellipse(PaintDC, RLeft, RTop, RRight, RBottom);
      break;
  }
  SelectObject(PaintDC, OldBrush);

  if (hSizeBox != 0)
    FillSizeBox(hwnd, PaintDC);

  EndPaint (hwnd, &ps) ;
}

//////////////////////////////////////
// WM_SIZE
//////////////////////////////////////
#pragma argsused
void WinSize_OnSize(HWND hwnd, UINT state, int cx, int cy)
{
  HDC PaintDC = GetDC(hwnd);
  if (hSizeBox != 0)
    FillSizeBox(hwnd, PaintDC);
  ReleaseDC(hwnd, PaintDC);
}

//////////////////////////////////////
// MakeShape Dialog
//////////////////////////////////////
void SetRectText(HWND hwnd)
{
  char S[45];
  HWND ParentWindow;

  ParentWindow = GetWindow(hwnd, GW_OWNER);
  InvalidateRect(ParentWindow, NULL, TRUE);
  sprintf(S, "%d", RLeft);
  SendDlgItemMessage(hwnd, ID_RLEFT, WM_SETTEXT, 0, (long)S);
  sprintf(S, "%d", RRight);
  SendDlgItemMessage(hwnd, ID_RRIGHT, WM_SETTEXT, 0, (long)S);
  sprintf(S, "%d", RTop);
  SendDlgItemMessage(hwnd, ID_RTOP, WM_SETTEXT, 0, (long)S);
  sprintf(S, "%d", RBottom);
  SendDlgItemMessage(hwnd, ID_RBOTTOM, WM_SETTEXT, 0, (long)S);
}

//////////////////////////////////////
// GetUserCords
//////////////////////////////////////
void GetUserCords(HWND hwnd)
{
  char S[100];

  GetWindowText(GetDlgItem(hwnd, ID_RLEFT), S, 100);
  RLeft = atoi(S);
  GetWindowText(GetDlgItem(hwnd, ID_RTOP), S, 100);
  RTop = atoi(S);
  GetWindowText(GetDlgItem(hwnd, ID_RRIGHT), S, 100);
  RRight = atoi(S);
  GetWindowText(GetDlgItem(hwnd, ID_RBOTTOM), S, 100);
  RBottom = atoi(S);
}

//////////////////////////////////////
// MakeShape
//////////////////////////////////////
void MakeShape(HWND hDlg)
{
  RECT R;

  GetClientRect(GetParent(hDlg), &R);
  HDC DC = GetDC(GetParent(hDlg));
  SetMapMode(DC, Modes[ModeCase]);
  DPtoLP(DC,(LPPOINT)&R, 2);
  ReleaseDC(GetParent(hDlg), DC);

  switch (OriginMode)
  {
    case ORG_CENTER:
      RLeft = -(R.right / 3);
      RTop = -(R.bottom / 3);
      RRight = (R.right / 3);
      RBottom = (R.bottom / 3);
      break;

    case ORG_LOWLEFT:
      RBottom = -(R.bottom / 3);
      RLeft = R.left / 3;
      RTop = R.top / 3;
      RRight = R.right / 3;
      break;

    case ORG_UPLEFT:
      RLeft = R.left / 3;
      RTop = R.top / 3;
      RRight = R.right / 3;
      RBottom = R.bottom / 3;
      break;
  }
}


//////////////////////////////////////
// DoDraw
//////////////////////////////////////
void DoDraw(HWND hwnd)
{
  if (UseCompCords)
    MakeShape(hwnd);
  SetRectText(hwnd);
}

//////////////////////////////////////
// DoRectCommand
//////////////////////////////////////
BOOL DoRectCommand(HWND hwnd, WORD wParam)
{
  switch(wParam)
  {
    case ID_DRAW:
      GetUserCords(hwnd);
      InvalidateRect(GetParent(hwnd), NULL, TRUE);
      return TRUE;

    case ID_RTEXT:
      ModeCase = TEXT;
      DoDraw(hwnd);
      return TRUE;

    case ID_RLOMETRIC:
      ModeCase = LOMETRIC;
      DoDraw(hwnd);
      return TRUE;

    case ID_RHIMETRIC:
      ModeCase = HIMETRIC;
      DoDraw(hwnd);
      return TRUE;

    case ID_RLOENGLISH:
      ModeCase = LOENGLISH;
      DoDraw(hwnd);
      return TRUE;

    case ID_RHIENGLISH:
      ModeCase = HIENGLISH;
      DoDraw(hwnd);
      return TRUE;

    case ID_RTWIPS:
      ModeCase = TWIPS;
      DoDraw(hwnd);
      return TRUE;

    case ID_ORGUPLEFT:
      GetUserCords(hwnd);
      OriginMode = ORG_UPLEFT;
      DoDraw(hwnd);
      return TRUE;

    case ID_ORGLOWLEFT:
      GetUserCords(hwnd);
      OriginMode = ORG_LOWLEFT;
      DoDraw(hwnd);
      return TRUE;

    case ID_ORGCENTER:
      GetUserCords(hwnd);
      OriginMode = ORG_CENTER;
      DoDraw(hwnd);
      return TRUE;

    case ID_USERCORDS:
      GetUserCords(hwnd);
      UseCompCords = FALSE;
      return TRUE;

    case ID_COMPCORDS:
      UseCompCords = TRUE;
      return FALSE;

    case ID_RECT:
      Shape = RECTANGLE;
      DoDraw(hwnd);
      return TRUE;

    case ID_CIRCLE:
      Shape = CIRCLE;
      DoDraw(hwnd);
      return TRUE;

    case IDOK:
    case IDCANCEL:
    {
      EndDialog(hwnd, TRUE);
      return TRUE;
    }
  }
  return FALSE;
}

//////////////////////////////////////
// MakeShapeProc
//////////////////////////////////////
#pragma argsused
BOOL CALLBACK _export MakeShapeProc(HWND hwnd, WORD Msg, WORD
wParam, LONG lParam)
{
  switch(Msg)
  {
    case WM_INITDIALOG:
      SendDlgItemMessage(hwnd, ID_RTEXT, BM_SETCHECK, 1, 0L);
      SendDlgItemMessage(hwnd, ID_RECT, BM_SETCHECK, 1, 0L);
      SendDlgItemMessage(hwnd, ID_ORGUPLEFT, BM_SETCHECK, 1, 0L);
      SendDlgItemMessage(hwnd, ID_COMPCORDS, BM_SETCHECK, 1, 0L);
      return TRUE;

    case WM_COMMAND:
      return (!DoRectCommand(hwnd, wParam));

    case WM_CTLCOLOR:
       switch(HIWORD(lParam))
       {
         case CTLCOLOR_STATIC:
         case CTLCOLOR_BTN:
         case CTLCOLOR_EDIT:
           /* Set text to white and background to green */
           SetTextColor((HDC)wParam, RGB(0, 127, 0));
           SetBkMode((HDC)wParam, TRANSPARENT);
           return (BOOL) GreenBrush;

         case CTLCOLOR_DLG:
           return (BOOL) GreenBrush;
       }
       return (LRESULT) NULL;
  }
  return FALSE;
}

//-------------------------------------------------
// SizeBoxProc Dialog
//-------------------------------------------------
#pragma argsused
BOOL CALLBACK _export SizeBoxProc(HWND hDlg, WORD Msg, WORD
wParam, LONG lParam)
{
  TSIZEDATA SizeData;
  void *lpSizeData;
  char S[20];
  int i;

  int Indexs[] = {2100, 2200, 2300, 2400, 2500, 2600};

  switch(Msg)
  {
    case WM_CLOSE:
      DestroyWindow(hDlg);
      hSizeBox = 0;
      return TRUE;

    case WM_CTLCOLOR:
      switch(HIWORD(lParam))
       {
         case CTLCOLOR_STATIC:
           SetTextColor((HDC)wParam, RGB(255, 0, 0));
           SetBkMode((HDC)wParam, TRANSPARENT);
           return (BOOL) GetStockObject(BLACK_BRUSH);

         case CTLCOLOR_DLG:
           return (BOOL) GetStockObject(BLACK_BRUSH);
       }
       return (BOOL) NULL;

    case WM_SETDATA:
      lpSizeData = (void *)lParam;
      memcpy(SizeData, lpSizeData, sizeof(TSIZEDATA));
      for (i = 0; i < 6; i++)
        {
        sprintf(S, "%d", SizeData[i].left);
        SendDlgItemMessage(hDlg, Indexs[i] + 1,
                           WM_SETTEXT, 0, long(S));
        sprintf(S, "%d", SizeData[i].right);
        SendDlgItemMessage(hDlg, Indexs[i] + 2,
                           WM_SETTEXT, 0, long(S));
        sprintf(S, "%d", SizeData[i].top);
        SendDlgItemMessage(hDlg, Indexs[i] + 3,
                           WM_SETTEXT, 0, long(S));
        sprintf(S, "%d", SizeData[i].bottom);
        SendDlgItemMessage(hDlg, Indexs[i] + 4,
                           WM_SETTEXT, 0, long(S));
        }
      return TRUE;
  }
  return FALSE;
}
