
/*---------------------------------------------------------------------------*/
/*                                                                           */
/*          WEDL - Windows Enhanced Dialog Library                           */
/*          Copyright (c) 1991, Mike Smedley                                 */
/*          All Rights Reserved                                              */
/*          Module:  DEMO2.C                                                 */
/*                                                                           */
/*---------------------------------------------------------------------------*/

#include <windows.h>
#include "wedl.h"
#include "demo.h"
#include "demohelp.h"

/*---------------------------------------------------------------------------*/

static HFORM define_the_form( HWND hDlg, int curr_rec );
static void fill_media_list_box( HWND hDlg );
static int get_next_record( HWND hDlg, int curr_rec );
static int get_previous_record( HWND hDlg, int curr_rec );
static BOOL okay_to_leave( HWND hDlg, HFORM hform, int new_rec );
static BOOL okay_to_save_changes( HWND hDlg );

/*---------------------------------------------------------------------------*/

struct albums_t {
    char artist[25+1];
    char title[30+1];
    char label[15+1];
    int copyright;
    char media[15+1];
};

static struct albums_t albums[] = {
    { "Aerosmith", "Pump", "Geffen", 1989, "Compact Disc" },
    { "Bell Biv Devoe", "Poison", "MCA", 1990, "Cassette" },
    { "The Cure", "Disintegration", "Elektra", 1989, "Compact Disc" },
    { "Front 242", "Front by Front", "Wax Trax", 1988, "Compact Disc" },
    { "INXS", "Kick", "Atlantic", 1987, "Cassette" },
    { "Judas Priest", "Screaming for Vengeance", "Columbia", 1982, "Record" },
    { "Kenny G", "Live", "Arista", 1989, "Compact Disc" },
    { "R.E.M.", "Out of Time", "Warner Bros.", 1991, "Cassette" },
    { "Sex Pistols", "Never Mind the Bollocks", "Warner Bros.", 1977, "Record" },
    { "Van Halen", "OU812", "Warner Bros.", 1988, "Compact Disc" },
};

static PSTR media[] =
    { "Compact Disc", "Cassette", "Record", "DAT", "Open Reel", "8-Track", NULL };

/*---------------------------------------------------------------------------*/

int FAR PASCAL DialogProc2( HWND hDlg, unsigned message, WORD wParam,
                            LONG lParam )
{
    static HFORM hform = NULL;
    static int curr_rec = 0;
    int new_rec;

    switch( message ) {
        case WM_INITDIALOG:
            hform = define_the_form( hDlg, curr_rec );
            fill_media_list_box( hDlg );
            return( TRUE );

        case WM_COMMAND:
            dproc_enter_wm_command( hform, wParam, lParam );
            switch( wParam ) {
                case IDOK:
                    dproc_enter_idok( hform );
                    if( !form_in_error_cond( hform ) ) {
                        if( form_validate( hform ) != NULL ) break;
                        if( form_has_changed( hform ) )
                            if( okay_to_save_changes( hDlg ) )
                                form_process( hform );
                        EndDialog( hDlg, TRUE );
                        form_terminate( hform );
                    }
                    return( TRUE );
                case IDCANCEL:
                    dproc_enter_idcancel( hform );
                    EndDialog( hDlg, TRUE );
                    form_terminate( hform );
                    return( TRUE );
                case ID_PREVIOUS:
                    new_rec = get_previous_record( hDlg, curr_rec );
                    if( new_rec != -1 ) {
                        if( !okay_to_leave( hDlg, hform, new_rec ) ) break;
                        curr_rec = new_rec;
                    }
                    PostMessage( hDlg, WM_COMMAND, GOTO_ARTIST, 0 );
                    return( TRUE );
                case ID_NEXT:
                    new_rec = get_next_record( hDlg, curr_rec );
                    if( new_rec != -1 ) {
                        if( !okay_to_leave( hDlg, hform, new_rec ) ) break;
                        curr_rec = new_rec;
                    }
                    PostMessage( hDlg, WM_COMMAND, GOTO_ARTIST, 0 );
                    return( TRUE );
                case GOTO_ARTIST:
                    SetFocus( GetDlgItem( hDlg, ID_ARTIST ) );
                    return( TRUE );

            }
            break;
    }
    return( FALSE );
}

/*---------------------------------------------------------------------------*/

static HFORM define_the_form( HWND hDlg, int curr_rec )
{
    HFORM hform;

    hform = form_begin( hDlg );
    form_set_help( hform, "demohelp.hlp" );
    field_define( hform, ID_RECNO, &curr_rec, DT_INTEGER,
                  "999", FDF_UPDATE | FDF_NOTEDIT, NULL, 0, HELPID_RECNO );
    field_define( hform, ID_ARTIST, albums[curr_rec].artist, DT_STRING,
                  "?(25)", FDF_NOTBLANK | FDF_CONDUPD, NULL, 0, HELPID_ARTIST );
    field_define( hform, ID_TITLE, albums[curr_rec].title, DT_STRING,
                  "?(30)", FDF_CONDUPD, NULL, 0, HELPID_TITLE );
    field_define( hform, ID_LABEL, albums[curr_rec].label, DT_STRING,
                  "?(15)", FDF_CONDUPD, NULL, 0, HELPID_LABEL );
    field_define( hform, ID_COPYRIGHT, &albums[curr_rec].copyright, DT_INTEGER,
                  "<1><9>99", FDF_COMPLETE | FDF_UPDATE, NULL, 0, HELPID_COPR );
    field_define( hform, ID_MEDIA, albums[curr_rec].media, DT_STRING,
                  "?(15)", FDF_COMBO | FDF_NOTBLANK | FDF_CONDUPD, NULL, 0, HELPID_MEDIA );
    statmsg_define( hform, ID_INSERT,   SM_INSERT,   "[Ins]",  "" );
    statmsg_define( hform, ID_CAPSLOCK, SM_CAPSLOCK, "[Caps]", "" );
    statmsg_define( hform, ID_NUMLOCK,  SM_NUMLOCK,  "[Num]",  "" );
    form_end( hform );
    return( hform );
}

/*---------------------------------------------------------------------------*/

static void fill_media_list_box( HWND hDlg )
{
    int i;
    for( i = 0 ; media[i] != NULL ; i++ ) {
        SendMessage( GetDlgItem( hDlg, ID_MEDIA ), CB_ADDSTRING, 0,
                     (LONG) ( (LPSTR) media[i] ) );
    }
}

/*---------------------------------------------------------------------------*/

static int get_next_record( HWND hDlg, int curr_rec )
{
    int next_rec, last_rec;

    next_rec = curr_rec + 1;
    last_rec = ( sizeof( albums ) / sizeof( albums[0] ) ) - 1;
    if( next_rec > last_rec ) {
        MessageBeep( 0 );
        MessageBox( hDlg, "End of File", NULL, MB_ICONEXCLAMATION | MB_OK );
        return( -1 );
    }
    return( next_rec );
}

/*---------------------------------------------------------------------------*/

static int get_previous_record( HWND hDlg, int curr_rec )
{
    int prev_rec;

    prev_rec = curr_rec - 1;
    if( prev_rec < 0 ) {
        MessageBeep( 0 );
        MessageBox( hDlg, "Top of File", NULL, MB_ICONEXCLAMATION | MB_OK );
        return( -1 );
    }
    return( prev_rec );
}

/*---------------------------------------------------------------------------*/

static BOOL okay_to_leave( HWND hDlg, HFORM hform, int new_rec )
{
    if( form_validate( hform ) != NULL ) return( FALSE );
    if( form_has_changed( hform ) )
        if( okay_to_save_changes( hDlg ) ) form_process( hform );
    form_terminate( hform );
    define_the_form( hDlg, new_rec );
    return( TRUE );
}

/*---------------------------------------------------------------------------*/

static BOOL okay_to_save_changes( HWND hDlg )
{
    return( MessageBox( hDlg, "Record has changed.  Save changes?",
                        "Record Has Changed",
                        MB_ICONQUESTION | MB_YESNO ) == IDYES );
}

